package com.voxeet.sdk.core.abs.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import retrofit2.Call;
import retrofit2.Response;
import com.voxeet.sdk.core.ConferenceSimpleState;
import com.voxeet.sdk.core.abs.AbstractConferenceSdkObservableProvider;
import com.voxeet.sdk.core.abs.AbstractConferenceSdkService;
import com.voxeet.sdk.core.abs.AbstractPromiseable;
import com.voxeet.sdk.core.abs.information.ConferenceInformation;
import com.voxeet.sdk.core.http.HttpCallback;
import com.voxeet.sdk.core.http.HttpHelper;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.error.ReplayConferenceErrorEvent;
import com.voxeet.sdk.events.success.ReplayConferenceEvent;
import com.voxeet.sdk.json.SdkConferenceReplayBody;

public class ReplayPromise extends AbstractPromiseable<Boolean> {
    private final static String TAG = ReplayPromise.class.getSimpleName();
    private final String conferenceId;
    private final long offset;

    public ReplayPromise(@NonNull AbstractConferenceSdkService parent,
                         @NonNull MediaService mediaService,
                         @NonNull AbstractConferenceSdkObservableProvider provider,
                         @Nullable ConferenceInformation information,
                         @NonNull EventBus eventBus,
                         @NonNull String conferenceId,
                         long offset) {
        super(parent, mediaService, provider, information, eventBus);

        this.conferenceId = conferenceId;
        this.offset = offset;
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final ConferenceInformation information = getConferenceInformation(conferenceId);

                information.setListener(true);
                getParent().setListenerMode(false);

                final Call<ReplayConferenceEvent> user = getProvider().getReplayObservable(conferenceId, new SdkConferenceReplayBody(offset));
                HttpHelper.enqueue(user, new HttpCallback<ReplayConferenceEvent>() {
                    @Override
                    public void onSuccess(@NonNull ReplayConferenceEvent object, @NonNull Response<ReplayConferenceEvent> response) {
                        Log.d(TAG, "Success server answer for replaying mConference with id: " +
                                conferenceId + "at offset: " + offset + " given := " + object.getConferenceId());

                        information.setConferenceType(ConferenceSimpleState.REPLAY);

                        joinConference(information)
                                .then(new PromiseExec<Boolean, Object>() {
                                    @Override
                                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                                        solver.resolve(result);
                                    }
                                })
                                .error(new ErrorPromise() {
                                    @Override
                                    public void onError(Throwable error) {
                                        error.printStackTrace();
                                        solver.resolve(false);
                                    }
                                });
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ReplayConferenceEvent> response) {
                        HttpException.dumpErrorResponse(response);

                        Log.d(TAG, "Failed to replay mConference");

                        getEventBus().post(new ReplayConferenceErrorEvent(getParent().handleError(e)));
                        e.printStackTrace();
                        solver.resolve(false);
                    }
                });

            }
        });
    }
}
