package com.voxeet.sdk.audio;

import android.content.Context;
import android.support.annotation.NonNull;

import com.voxeet.android.media.MediaPowerManager;
import com.voxeet.android.media.audio.AudioRoute;

import java.util.HashMap;
import java.util.List;

import com.voxeet.sdk.utils.AudioType;
import com.voxeet.sdk.utils.SoundPool;
import com.voxeet.sdk.utils.Validate;

/**
 * Created by kevinleperf on 26/10/2018.
 */

public class SoundManager {

    private com.voxeet.android.media.audio.AudioManager mAudioManager;
    private MediaPowerManager mMediaPowerManager;
    private Context mContext;
    private android.media.AudioManager mSystemAudioService;
    private int mSoundMode;
    private SoundPool _sound_pool;
    private HashMap<AudioType, String> mSounds;

    private boolean mEnable;

    private SoundManager() {
        enable();
    }

    public SoundManager(@NonNull Context context) {
        this();

        mSoundMode = android.media.AudioManager.STREAM_VOICE_CALL;

        mAudioManager = new com.voxeet.android.media.audio.AudioManager(context);
        mMediaPowerManager = new MediaPowerManager(context,
                currentRoute());

        mContext = context;
        mSystemAudioService = (android.media.AudioManager) mContext.getSystemService(Context.AUDIO_SERVICE);

        mSounds = new HashMap<>();
        configure();
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return mAudioManager.availableRoutes();
    }

    public AudioRoute currentRoute() {
        return mAudioManager.outputRoute();
    }

    public boolean setAudioRoute(@NonNull AudioRoute route) {
        Validate.runningOnUiThread();

        if(isWiredHeadsetOn()) {
            route = AudioRoute.ROUTE_HEADSET;
        } else if(isBluetoothHeadsetConnected()) {
            route = AudioRoute.ROUTE_BLUETOOTH;
        }

        return mAudioManager.setOutputRoute(route);
    }

    public boolean isBluetoothHeadsetConnected() {
        return mAudioManager.isBluetoothHeadsetConnected();
    }

    public boolean isWiredHeadsetOn() {
        return mAudioManager.isWiredHeadsetOn();
    }

    public SoundManager setSpeakerMode(boolean isSpeaker) {
        mAudioManager.setSpeakerMode(isSpeaker);
        return this;
    }

    public boolean isSpeakerOn() {
        return AudioRoute.ROUTE_SPEAKER.equals(mAudioManager.outputRoute());
    }

    public SoundManager acquireLocks() {
        mMediaPowerManager.acquire();
        return this;
    }

    public SoundManager releaseLocks() {
        mMediaPowerManager.release();
        return this;
    }

    public SoundManager setInVoiceCallSoundType() {
        mAudioManager.forceVolumeControlStream(mSoundMode);
        return this;
    }

    public SoundManager resetDefaultSoundType() {
        mAudioManager.resetDefaultSoundType();
        return this;
    }

    private int getUiSoundsStreamType() {
        return mAudioManager.getDefaultSoundStreamType();
    }

    private SoundManager forceVolumeControlStream(int type) {
        mAudioManager.forceVolumeControlStream(type);
        return this;
    }

    public SoundManager abandonAudioFocusRequest() {
        mAudioManager.abandonAudioFocusRequest();
        return this;
    }

    public SoundManager requestAudioFocus() {
        mAudioManager.requestAudioFocus();
        return this;
    }

    private SoundPool getSoundPool() {
        if (_sound_pool == null) {
            _sound_pool = new SoundPool(mContext, mSoundMode);
        }
        return _sound_pool;
    }

    private void configure() {
        mSounds.put(AudioType.RING, "out.mp3");
        mSounds.put(AudioType.HANGUP, "leave.mp3");

        setSound(AudioType.RING, mSounds.get(AudioType.RING));
        setSound(AudioType.HANGUP, mSounds.get(AudioType.HANGUP));
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return getSoundPool().release(type).setShortResource(type, assetName);
    }

    public SoundManager playSoundType(@NonNull AudioType type) {
        if (isEnabled()) {
            getSoundPool().playShortResource(type, mSounds.get(type));
        }
        return this;
    }

    public SoundManager playSoundTypeForce(@NonNull AudioType type) {
        getSoundPool().playShortResource(type, mSounds.get(type));
        return this;
    }

    public SoundManager stopSoundType(@NonNull AudioType audioType) {
        getSoundPool().stop(audioType);
        return this;
    }


    public SoundManager stop() {
        getSoundPool().stop();
        return this;
    }

    public SoundManager onConferencePreJoinedEvent() {
        mAudioManager.checkOutputRoute();
        return this;
    }

    public SoundManager onConferenceDestroyedPush() {
        abandonAudioFocusRequest();
        return this;
    }

    public void enable() {
        mEnable = true;
        if(null != mAudioManager) {
            mAudioManager.enable();
        }
    }

    public void disable() {
        mEnable = false;
        if(null != mAudioManager) {
            mAudioManager.disable();
        }
    }

    private boolean isEnabled() {
        return mEnable;
    }
}
