package com.voxeet.android.media.audio;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothHeadset;
import android.bluetooth.BluetoothProfile;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.media.AudioDeviceInfo;
import android.os.Build;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;
import android.view.Window;

import com.voxeet.android.media.audio.focus.AudioFocusManager;

import org.greenrobot.eventbus.EventBus;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import com.voxeet.sdk.events.AudioRouteChangeEvent;
import com.voxeet.sdk.utils.Validate;

/**
 * Control and Manager the Audio states of the media
 */

public class AudioManager {
    private static final String TAG = AudioManager.class.getSimpleName();
    private Context mContext;
    private boolean isBluetoothScoStarted;
    private Handler handler = new Handler();

    private static final int MODE_IN_COMMUNICATION = android.media.AudioManager.MODE_IN_COMMUNICATION;
    private static final int MODE_CURRENT = android.media.AudioManager.MODE_CURRENT;
    private final static int STREAM_VOICE_CALL = android.media.AudioManager.STREAM_VOICE_CALL;
    private final static int STREAM_BLUETOOTH_SCO = 6; //from AudioSystem.STREAM_BLUETOOTH_SCO undocumented
    private static final int STREAM_SYSTEM = android.media.AudioManager.STREAM_SYSTEM;
    private static final int STREAM_MUSIC = android.media.AudioManager.STREAM_MUSIC;

    private android.media.AudioManager mServiceAudioManager;

    private HeadsetStateReceiver mHeadsetStateReceiver;
    private BluetoothHeadset mCurrentBluetoothHeadset;

    private BluetoothHeadsetListener mBluetoothHeadsetListener;
    private BluetoothAdapter mBluetoothAdapter;

    private AudioRoute mOutputRoute = AudioRoute.ROUTE_PHONE;
    private List<IMediaStateListener> mMediaStateListeners;

    private AudioFocusManager manager;
    private Runnable runnableBluetoothSco = new Runnable() {
        @Override
        public void run() {
            requestAudioFocus();
            try {
                mServiceAudioManager.startBluetoothSco();
                mServiceAudioManager.setBluetoothScoOn(true);
                //checkForBluetoothDeviceAtStartUp();
                //isBluetoothScoStarted = true;
            } catch (Exception e) {
                e.printStackTrace();
                isBluetoothScoStarted = false;
            }
        }
    };
    private boolean enabled;

    public void requestAudioFocus() {
        Log.d(TAG, "requestAudioFocus: ");
        forceVolumeControlStream(STREAM_VOICE_CALL);
        manager.requestAudioFocus(mServiceAudioManager);
    }

    public void abandonAudioFocusRequest() {
        Log.d(TAG, "abandonAudioFocusRequest: ");
        mServiceAudioManager.setMode(android.media.AudioManager.MODE_NORMAL);
        manager.abandonAudioFocus(mServiceAudioManager);
        forceVolumeControlStream(STREAM_MUSIC);
    }

    public void disable() {
        enabled = false;
    }

    public void enable() {
        enabled = true;
    }

    private class BluetoothHeadsetListener implements BluetoothHeadset.ServiceListener {
        @Override
        public void onServiceConnected(int i, BluetoothProfile bluetoothProfile) {
            if (bluetoothProfile instanceof BluetoothHeadset) {
                mCurrentBluetoothHeadset = (BluetoothHeadset) bluetoothProfile;
            }

            updateBluetoothHeadsetConnectivity(null);
        }

        @Override
        public void onServiceDisconnected(int i) {
            mCurrentBluetoothHeadset = null;

            isBluetoothScoStarted = false;
            Log.d(TAG, "onServiceDisconnected: isBluetoothScoStarted = true");
            for (IMediaStateListener listeners : mMediaStateListeners) {
                try {
                    listeners.onBluetoothHeadsetStateChange(false);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            mServiceAudioManager.stopBluetoothSco();

            checkOutputRoute();
            EventBus.getDefault().post(new AudioRouteChangeEvent());
        }
    }

    private class HeadsetStateReceiver extends BroadcastReceiver {
        @Override
        public void onReceive(Context context, Intent intent) {
            Log.d(TAG, "onReceive: " + intent.getAction());
            if(!enabled) {
                Log.d(TAG, "onReceive: the AudioManager is disabled, nothing to do for action received");
                return;
            }

            String action = intent.getAction();
            if (null == action) action = "";
            switch (action) {
                case Intent.ACTION_HEADSET_PLUG:
                    int state = intent.getIntExtra("state", -1);
                    int has_mic = intent.getIntExtra("microphone", -1);

                    boolean isPlugged = state == 1 && has_mic == 1;

                    if (isPlugged) {
                        for (IMediaStateListener listeners : mMediaStateListeners) {
                            listeners.onHeadsetStateChange(true);
                        }
                        setSpeakerMode(false);
                        checkOutputRoute();

                        EventBus.getDefault().post(new AudioRouteChangeEvent());
                    } else {
                        for (IMediaStateListener listeners : mMediaStateListeners) {
                            listeners.onHeadsetStateChange(false);
                        }
                        checkOutputRoute();
                        EventBus.getDefault().post(new AudioRouteChangeEvent());
                    }
                    break;
                case BluetoothDevice.ACTION_ACL_CONNECTED:
                    BluetoothDevice device = intent.getParcelableExtra(BluetoothDevice.EXTRA_DEVICE);
                    updateBluetoothHeadsetConnectivity(device);
                    break;
                case BluetoothDevice.ACTION_ACL_DISCONNECTED:
                    updateBluetoothHeadsetConnectivity(null);
                    break;
                case android.media.AudioManager.ACTION_SCO_AUDIO_STATE_CHANGED:

                    int l_state = intent.getIntExtra(android.media.AudioManager.EXTRA_SCO_AUDIO_STATE, -1);
                    Log.d(TAG, "Audio SCO: " + android.media.AudioManager.ACTION_SCO_AUDIO_STATE_CHANGED);
                    switch (l_state) {
                        case android.media.AudioManager.SCO_AUDIO_STATE_CONNECTED: {
                            Log.i(TAG, "SCO_AUDIO_STATE_CONNECTED");
                            requestAudioFocus();
                        }
                        break;
                        case android.media.AudioManager.SCO_AUDIO_STATE_DISCONNECTED: {
                            Log.e(TAG, "SCO_AUDIO_STATE_DISCONNECTED");
                            requestAudioFocus();
                        }
                        break;
                        default:
                            Log.e(TAG, "unknown state received:" + l_state);
                    }
            }
        }
    }

    private AudioManager() {

    }

    public AudioManager(Context context) {
        this();

        manager = new AudioFocusManager();

        mContext = context;
        mMediaStateListeners = new ArrayList<>();

        mHeadsetStateReceiver = new HeadsetStateReceiver();
        mBluetoothHeadsetListener = new BluetoothHeadsetListener();

        mCurrentBluetoothHeadset = null;
        isBluetoothScoStarted = false;

        mServiceAudioManager = (android.media.AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
        checkForBluetoothDeviceAtStartUp();

        mBluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
        if (mBluetoothAdapter != null) {
            try {
                mBluetoothAdapter.getProfileProxy(context, mBluetoothHeadsetListener, BluetoothProfile.HEADSET);
            } catch (SecurityException e) {
                e.printStackTrace();
            }
        }

        context.registerReceiver(mHeadsetStateReceiver, new IntentFilter(Intent.ACTION_HEADSET_PLUG));
        context.registerReceiver(mHeadsetStateReceiver, new IntentFilter(BluetoothDevice.ACTION_ACL_CONNECTED));
        context.registerReceiver(mHeadsetStateReceiver, new IntentFilter(BluetoothDevice.ACTION_ACL_DISCONNECTED));
        context.registerReceiver(mHeadsetStateReceiver, new IntentFilter(BluetoothAdapter.ACTION_STATE_CHANGED));
        context.registerReceiver(mHeadsetStateReceiver, new IntentFilter(android.media.AudioManager.ACTION_SCO_AUDIO_STATE_CHANGED));
    }

    public void stop() {
        setBluetooth(false);

        mContext.unregisterReceiver(mHeadsetStateReceiver);

        if (mCurrentBluetoothHeadset != null && mBluetoothAdapter != null) {
            mBluetoothAdapter.closeProfileProxy(BluetoothProfile.HEADSET, mCurrentBluetoothHeadset);
        }
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> availableRoutes() {
        List<AudioRoute> routes = new ArrayList<>();

        routes.add(AudioRoute.ROUTE_SPEAKER);

        if (isWiredHeadsetOn()) {
            routes.add(AudioRoute.ROUTE_HEADSET);
        } else {
            routes.add(AudioRoute.ROUTE_PHONE);
        }

        if (isBluetoothHeadsetConnected()) {
            routes.add(AudioRoute.ROUTE_BLUETOOTH);
        }

        return routes;
    }

    /**
     * Retrieve the current audio route defined in this manager
     *
     * @return a non null audio route
     */
    @NonNull
    public AudioRoute outputRoute() {
        return mOutputRoute;
    }


    /**
     * Set the current route for the manager
     *
     * @param route set the valid audio route
     */
    public boolean setOutputRoute(@NonNull AudioRoute route) {
        if (null == route) return false;

        switch (route) {
            case ROUTE_BLUETOOTH:
                if (isBluetoothHeadsetConnected()) {
                    setBluetooth(true);
                }
                break;
            case ROUTE_HEADSET:
            case ROUTE_PHONE:
                setSpeakerMode(false);
                break;
            case ROUTE_SPEAKER:
                if (isWiredHeadsetOn() || isBluetoothHeadsetConnected()) {
                    return false;
                }
                setSpeakerMode(true);
                break;
            default:
                break;
        }
        EventBus.getDefault().post(new AudioRouteChangeEvent());
        return true;
    }

    /**
     * @see android.app.Activity#setVolumeControlStream(int)
     */
    public void setVolumeControlStream(Window window, int streamType) {
        try {
            window.setVolumeControlStream(streamType);
        } catch (Exception e) {

        }
    }

    public void setSpeakerMode(boolean speakerMode) {
        Log.d(TAG, "setSpeakerMode: " + isWiredHeadsetOn() + " " + isBluetoothHeadsetConnected());
        if (isWiredHeadsetOn()) {
            mServiceAudioManager.setSpeakerphoneOn(false);
            mServiceAudioManager.setMode(MODE_IN_COMMUNICATION);
            forceVolumeControlStream(STREAM_VOICE_CALL);
        } else if (isBluetoothHeadsetConnected()) {
            mServiceAudioManager.setSpeakerphoneOn(false);
            mServiceAudioManager.setMode(MODE_IN_COMMUNICATION);
            forceVolumeControlStream(STREAM_BLUETOOTH_SCO);
        } else if ("samsung".equalsIgnoreCase(Build.BRAND)) {
            if (speakerMode) {
                // route audio to back speaker
                mServiceAudioManager.setSpeakerphoneOn(true);
                mServiceAudioManager.setMode(MODE_CURRENT);
            } else {
                // route audio to earpiece
                mServiceAudioManager.setSpeakerphoneOn(speakerMode);
                if (mServiceAudioManager.isWiredHeadsetOn()) {
                    mServiceAudioManager.setMode(MODE_CURRENT);
                } else {
                    mServiceAudioManager.setMode(MODE_IN_COMMUNICATION);
                }
            }
            forceVolumeControlStream(STREAM_VOICE_CALL);
        } else {
            // Non-Samsung devices
            mServiceAudioManager.setMode(MODE_IN_COMMUNICATION);
            mServiceAudioManager.setSpeakerphoneOn(speakerMode);
            forceVolumeControlStream(STREAM_VOICE_CALL);
        }


        for (IMediaStateListener listeners : mMediaStateListeners) {
            try {
                listeners.onSpeakerChanged(speakerMode);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        checkOutputRoute();
    }

    /**
     * Start the audio manager in bluetooth mode
     * <p>
     * Can lead to a non-bluetooth state if a crash occured internally (android 5.0)
     *
     * @param isEnabled true if it should start
     */
    public void setBluetooth(boolean isEnabled) {
        forceVolumeControlStream(STREAM_VOICE_CALL);

        checkOutputRoute();
        try {
            if (isEnabled) {
                startBluetoothSco();
            } else {
                stopBluetoothSco();
            }
        } catch (NullPointerException e) { // Workaround for lollipop 5.0
            Log.d(TAG, "No bluetooth headset connected");
        }
    }

    /**
     * Register a valid listener to this manager
     *
     * @param listener a persisted listener
     */
    public void registerMediaState(@NonNull IMediaStateListener listener) {
        if (!mMediaStateListeners.contains(listener)) {
            mMediaStateListeners.add(listener);
        }
    }

    /**
     * Remove a listener from the list of listeners
     *
     * @param listener a valid listener, if not currently listening, no crash reported
     */
    public void unregisterMediaState(@NonNull IMediaStateListener listener) {
        mMediaStateListeners.remove(listener);
    }

    /**
     * Get the current default sound type
     *
     * @return the saved value
     */
    public int getDefaultSoundStreamType() {
        return STREAM_MUSIC;
    }

    /**
     * Set the default sound back to the default one
     */
    public void resetDefaultSoundType() {
        forceVolumeControlStream(STREAM_MUSIC);
    }

    /**
     * Force a given volume type
     *
     * @param type a valid type from the android.media.AudioManager class
     */
    public void forceVolumeControlStream(int type) {
        Log.d(TAG, "forceVolumeControlStream: " + type);
        try {
            Method method = mServiceAudioManager.getClass().getDeclaredMethod("forceVolumeControlStream", int.class);

            method.invoke(mServiceAudioManager, type);
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
    }

    /**
     * Check for bluetooth headset
     *
     * @return true if at least one device is properly connected
     */
    public boolean isBluetoothHeadsetConnected() {
        checkForBluetoothDeviceAtStartUp();

        return isBluetoothScoStarted;
    }

    public boolean isWiredHeadsetOn() {
        if (Build.VERSION.SDK_INT >= 23) {
            AudioDeviceInfo[] audioDevices = mServiceAudioManager.getDevices(android.media.AudioManager.GET_DEVICES_ALL);
            for (AudioDeviceInfo deviceInfo : audioDevices) {
                if (deviceInfo.getType() == AudioDeviceInfo.TYPE_WIRED_HEADPHONES || deviceInfo.getType() == AudioDeviceInfo.TYPE_WIRED_HEADSET) {
                    Log.d(TAG, "isWiredHeadsetOn: headphone connected");
                    return true;
                }
            }
        }
        return mServiceAudioManager.isWiredHeadsetOn();
    }

    /**
     * Check for wired or bluetooth headset
     *
     * @return true if at least one device is properly connected
     */
    public boolean isHeadphonesPlugged() {
        return isWiredHeadsetOn() || isBluetoothHeadsetConnected();
    }

    public void checkOutputRoute() {
        if (isBluetoothHeadsetConnected()) {
            mOutputRoute = AudioRoute.ROUTE_BLUETOOTH;
        } else if (isWiredHeadsetOn()) {
            mOutputRoute = AudioRoute.ROUTE_HEADSET;
        } else if (null != mServiceAudioManager && mServiceAudioManager.isSpeakerphoneOn()) {
            mOutputRoute = AudioRoute.ROUTE_SPEAKER;
        } else {
            mOutputRoute = AudioRoute.ROUTE_PHONE;
        }

        requestAudioFocus();
    }

    private int getUiSoundsStreamType() {
        try {
            Method method = mServiceAudioManager.getClass().getDeclaredMethod("getUiSoundsStreamType");

            Object result = method.invoke(mServiceAudioManager);
            return (int) result;
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
        return STREAM_SYSTEM;
    }

    private void checkForBluetoothDeviceAtStartUp() {
        Log.d(TAG, "checkForBluetoothDeviceAtStartUp");
        try {
            /*if (Build.VERSION.SDK_INT >= 23) {
                AudioDeviceInfo[] audioDevices = mServiceAudioManager.getDevices(android.media.AudioManager.GET_DEVICES_ALL);
                for (AudioDeviceInfo deviceInfo : audioDevices) {
                    Log.d(TAG, "checkForBluetoothDeviceAtStartUp: " + deviceInfo.getProductName() + " " + deviceInfo.isSource() + " " + deviceInfo.isSink());
                    if (deviceInfo.getType() == AudioDeviceInfo.TYPE_BLUETOOTH_SCO) {
                        isBluetoothScoStarted = true;
                    }
                }
            } else {*/
            BluetoothAdapter mBluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
            if (mBluetoothAdapter != null && mBluetoothAdapter.isEnabled()
                    && mBluetoothAdapter.getProfileConnectionState(BluetoothHeadset.HEADSET) == BluetoothHeadset.STATE_CONNECTED) {
                Log.d(TAG, "checkForBluetoothDeviceAtStartUp: bluetooth connected");
                isBluetoothScoStarted = true;
            }
            /*}*/
        } catch (Exception e) {
            Log.d(TAG, "checkForBluetoothDeviceAtStartUp: exception thrown, have you the BLUETOOTH permission?");
            e.printStackTrace();
        }
    }

    private void startBluetoothSco() {
        handler.postDelayed(runnableBluetoothSco, 3000);
    }

    private void stopBluetoothSco() {
        handler.removeCallbacks(runnableBluetoothSco);
        if (isBluetoothScoStarted) {
            try {
                mServiceAudioManager.setBluetoothScoOn(false);
                mServiceAudioManager.stopBluetoothSco();
            } catch (Exception e) {
                e.printStackTrace();
            }
            isBluetoothScoStarted = false;
        }
    }

    private void updateBluetoothHeadsetConnectivity(@Nullable BluetoothDevice device) {
        Log.d(TAG, "updateBluetoothHeadsetConnectivity: " + device);
        if (null != mCurrentBluetoothHeadset) {
            if (Validate.hasBluetoothPermissions(mContext)) {
                isBluetoothScoStarted = mCurrentBluetoothHeadset.getConnectedDevices().size() > 0;
                isBluetoothScoStarted |= null != device;
            } else {
                Log.e(TAG, "onServiceConnected: BLUETOOTH PERMISSION MISSING");
            }

            for (IMediaStateListener listeners : mMediaStateListeners) {
                listeners.onBluetoothHeadsetStateChange(isBluetoothScoStarted);
            }

            Log.d(TAG, "onServiceConnected: isBluetoothScoStarted = " + isBluetoothScoStarted);

            startBluetoothSco();
            checkOutputRoute();
            EventBus.getDefault().post(new AudioRouteChangeEvent());
        }
    }
}
