package voxeet.com.sdk.core.abs.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import org.greenrobot.eventbus.EventBus;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Response;
import voxeet.com.sdk.core.abs.AbstractConferenceSdkObservableProvider;
import voxeet.com.sdk.core.abs.AbstractConferenceSdkService;
import voxeet.com.sdk.core.abs.AbstractPromiseable;
import voxeet.com.sdk.core.abs.information.ConferenceInformation;
import voxeet.com.sdk.core.http.HttpCallback;
import voxeet.com.sdk.core.http.HttpHelper;
import voxeet.com.sdk.core.preferences.VoxeetPreferences;
import voxeet.com.sdk.core.services.MediaService;
import voxeet.com.sdk.events.error.HttpException;
import voxeet.com.sdk.events.success.AddConferenceParticipantResultEvent;
import voxeet.com.sdk.events.success.ConferenceRefreshedEvent;
import voxeet.com.sdk.json.SdkConferenceInvitation;
import voxeet.com.sdk.models.ConferenceUserStatus;

public class InvitePromise extends AbstractPromiseable<List<ConferenceRefreshedEvent>> {

    private final String TAG = InvitePromise.class.getSimpleName();

    private final List<String> ids;
    private final String conferenceId;

    public InvitePromise(@NonNull AbstractConferenceSdkService parent,
                         @NonNull MediaService mediaService,
                         @NonNull AbstractConferenceSdkObservableProvider provider,
                         @Nullable ConferenceInformation information,
                         @NonNull EventBus eventBus,
                         @NonNull String conferenceId,
                         @NonNull List<String> ids) {
        super(parent, mediaService, provider, information, eventBus);

        this.ids = ids;
        this.conferenceId = conferenceId;
    }

    @NonNull
    @Override
    public Promise<List<ConferenceRefreshedEvent>> createPromise() {
        return new Promise<>(new PromiseSolver<List<ConferenceRefreshedEvent>>() {
            @Override
            public void onCall(@NonNull final Solver<List<ConferenceRefreshedEvent>> solver) {
                //remove the timeout
                removeTimeoutCallbacks();

                //a new one is now sent
                sendTimeoutCallbacks();

                boolean sdk = isSDK();
                List<String> voxeetIds = sdk ? null : new ArrayList<>();
                final List<String> externalIds = new ArrayList<>();

                externalIds.addAll(ids);
                if (!sdk) voxeetIds.addAll(ids);

                //send an error if you're not in a conference
                if (null == conferenceId) {
                    try {
                        throw new IllegalStateException("You're not in a conference");
                    } catch (Exception e) {
                        getEventBus().post(new AddConferenceParticipantResultEvent(getParent().handleError(e), false));
                        solver.reject(e);
                        return;
                    }
                }

                final Call<ResponseBody> user = getProvider().getInviteObservable(conferenceId, new SdkConferenceInvitation(voxeetIds, externalIds));
                HttpHelper.enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        List<ConferenceRefreshedEvent> list = new ArrayList<>();
                        getEventBus().post(new AddConferenceParticipantResultEvent(response.code() == 200));

                        try {
                            String body = response.body().string();
                            Log.d(TAG, "onNext: " + body);
                        } catch (IOException e) {
                            e.printStackTrace();
                        }

                        if (response.code() == 200) {
                            for (String userId : externalIds) {
                                if (!userId.equals(VoxeetPreferences.id())) {
                                    Log.d(TAG, "Conference participant with id: " + userId + " invited");
                                    ConferenceRefreshedEvent event = new ConferenceRefreshedEvent(userId, updateConferenceParticipants(userId, ConferenceUserStatus.IN_PROGRESS));


                                    getEventBus().post(event);
                                    list.add(event);
                                }
                            }
                        }
                        solver.resolve(list);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        e.printStackTrace();

                        getEventBus().post(new AddConferenceParticipantResultEvent(getParent().handleError(e), false));
                        solver.reject(e);
                    }
                });
            }
        });
    }
}
