package com.voxeet.sdk.core.services;

import android.content.Context;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaSDK;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.camera.CameraEnumeratorLollipopWrapper;
import com.voxeet.android.media.camera.CameraEnumeratorPreLollipopWrapper;
import com.voxeet.android.media.camera.CameraEnumeratorWrapper;
import com.voxeet.android.media.crypto.AbstractNativeMediaCryptoCallback;
import com.voxeet.android.media.stats.LocalStats;

import org.webrtc.Camera1Enumerator;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBaseMethods;
import org.webrtc.VideoSink;

import com.voxeet.android.media.video.Camera2Enumerator;
import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdkTemplate;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.events.error.CameraSwitchErrorEvent;
import com.voxeet.sdk.events.success.CameraSwitchSuccessEvent;
import com.voxeet.sdk.utils.Validate;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;

/**
 * Created by kevinleperf on 17/10/2018.
 */

public class MediaService extends AbstractVoxeetService {

    private CameraEnumeratorWrapper enumerator;
    private AbstractNativeMediaCryptoCallback mediaCryptoCallback;
    private MediaSDK media;

    //hold the last camera state
    private boolean isFrontCamera;

    public MediaService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            enumerator = new CameraEnumeratorLollipopWrapper(instance.getApplicationContext());
        } else {
            enumerator = new CameraEnumeratorPreLollipopWrapper(instance.getApplicationContext());
        }
    }

    public boolean hasMedia() {
        return null != media;
    }

    public MediaSDK getMedia() {
        return media;
    }

    /**
     * @param peer
     * @return
     * @deprecated use {@link LocalStatsService#getLocalStats(String)}
     */
    @Deprecated
    @Nullable
    public LocalStats getLocalStats(@NonNull String peer) {
        if (null == peer) return null; //may happen in wrong implementation
        MediaSDK media = getMedia();
        if (null != media) return media.getLocalStats(peer);
        return null;
    }

    public void releaseMedia() {
        if (null != media) media.stop();

        media = null;
    }

    public void createMedia(Context context, String userId, MediaEngine.StreamListener mediaStreamListener,
                            CameraVideoCapturer.CameraEventsHandler cameraEventsHandler,
                            boolean videoOn, boolean useMic) throws MediaEngineException {
        media = new MediaSDK(context, userId, mediaStreamListener, cameraEventsHandler, videoOn, useMic, mediaCryptoCallback);
    }

    public EglBaseMethods.Context getEglContext() {
        if (null != media) {
            return media.getEglBase().getEglBaseContext();
        } else {
            return null;
        }
    }

    /**
     * Set a MediaCryptoCallback to be used "before" joining the conference
     * <p>
     * Note : it's not released automatically, it must be un/-set before any conference
     *
     * @param callback the callback to use, null to unset the callback
     */
    public void setCryptoCallback(@Nullable AbstractNativeMediaCryptoCallback callback) {
        this.mediaCryptoCallback = callback;
    }

    public boolean attachMediaStream(@NonNull MediaStream stream, @NonNull VideoSink videoSink) {
        if (null != media) getMedia().attachMediaStream(videoSink, stream);
        return null != media;
    }

    public boolean unAttachMediaStream(@NonNull MediaStream stream, @NonNull VideoSink videoSink) {
        if (null != media) getMedia().unattachMediaStream(videoSink, stream);
        return null != media;
    }

    public boolean isFrontCamera() {
        return isFrontCamera;
    }

    @NonNull
    public Promise<Boolean> switchCamera() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(getMedia(), "media");

                getMedia().switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean isFrontCamera) {
                        MediaService.this.isFrontCamera = isFrontCamera;

                        getEventBus().post(new CameraSwitchSuccessEvent(isFrontCamera));
                        solver.resolve(true);
                    }

                    @Override
                    public void onCameraSwitchError(String errorDescription) {
                        getEventBus().post(new CameraSwitchErrorEvent(errorDescription));
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    public CameraEnumeratorWrapper getCameraEnumerator() {
        return enumerator;
    }

    public String getDefaultCamera() {
        return enumerator.getDefaultName();
    }

    public void setDefaultCamera(@Nullable String cameraName) {
        enumerator.setDefaultName(cameraName);
    }
}
