package com.voxeet.sdk.core.services.authenticate;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.Request;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import com.voxeet.sdk.core.http.RefreshTokenBody;
import com.voxeet.sdk.core.services.authenticate.token.TokenResponseProvider;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.json.GrandTypeEvent;
import com.voxeet.sdk.models.TokenResponse;
import com.voxeet.sdk.utils.AbstractVoxeetEnvironmentHolder;
import com.voxeet.sdk.utils.Auth64;

public class KeySecretTokenResponseProvider extends TokenResponseProvider {


    /**
     * The current application id obtained from Voxeet Developer Portal
     * Note : if non null, password must be also non null
     */
    @NonNull
    private String appId;
    /**
     * The current application id obtained from Voxeet Developer Portal
     * Note : if non null, secret must be also non null
     */
    @NonNull
    private String secret;

    private KeySecretTokenResponseProvider(@NonNull AbstractVoxeetEnvironmentHolder holder) {
        super(holder);
        this.appId = "";
        this.secret = "";
    }

    public KeySecretTokenResponseProvider(@NonNull String appId,
                                          @NonNull String secret,
                                          @NonNull AbstractVoxeetEnvironmentHolder holder) {
        this(holder);
        this.appId = appId;
        this.secret = secret;
    }

    @Nullable
    @Override
    public TokenResponse refreshTokenResponse() {
        Log.d(TAG, "refreshTokenResponse: calling refreshing token");
        TokenResponse previous = tokenResponse;
        TokenResponse new_token_response = null;
        Call<TokenResponse> token = null;

        String tokenizedCredentials = Auth64.serialize(appId, secret);


        try {
            if (null != previous && null != previous.getRefreshToken()) {
                token = service.refreshToken(tokenizedCredentials, new RefreshTokenBody(previous.getRefreshToken()));
            }
            if (null != token) {
                new_token_response = token.execute().body();
            }
        } catch (Exception e) {
            Log.d(TAG, "authenticate: error on refresh token " + e.getMessage());
            ExceptionManager.sendException(e);
            e.printStackTrace();
        }

        if (null != new_token_response && null != new_token_response.getAccessToken()) {
            return new_token_response;
        }

        try {
            token = service.getToken(tokenizedCredentials, new GrandTypeEvent("client_credentials"));
            Response<TokenResponse> response = token.execute();
            TokenResponse body = response.body();
            if(null == body || response.code() == 503) {
                try {
                    throw new IllegalStateException("Access refused in callback :: embargoed ?");
                }catch (Exception e){
                    ExceptionManager.sendException(e);
                }
            }
            return body;
        } catch (Exception e) {
            Log.d(TAG, "refreshTokenResponse: second token attempt error");
            ExceptionManager.sendException(e);

            e.printStackTrace();
            return null;
        }
    }

    /**
     * Retrieve a valid TokenResponse from the servers
     * <p>
     * It will use on of the two available method available :
     * - resolve the injected tokenAccess
     * - resolve a token response obtained from Voxeet servers
     * <p>
     * Note : don't forget to call .then().error()
     *
     * @return a promise to resolve and manage the error
     */
    @NonNull
    @Override
    protected Promise<TokenResponse> retrieveTokenResponse() {
        return new Promise<>(new PromiseSolver<TokenResponse>() {
            @Override
            public void onCall(@NonNull final Solver<TokenResponse> solver) {
                Call<TokenResponse> token = service.getToken(Auth64.serialize(appId, secret), new GrandTypeEvent("client_credentials"));
                token.enqueue(new Callback<TokenResponse>() {
                    @Override
                    public void onResponse(Call<TokenResponse> call, retrofit2.Response<TokenResponse> response) {
                        Log.d(TAG, "onResponse: " + response.body() + " " + response.code());
                        if (null != response && response.code() == 503) {
                            try {
                                throw new IllegalStateException("Access refused");
                            }catch (Exception e){
                                solver.reject(e);
                            }
                        } else {
                            solver.resolve(response.body());
                        }
                    }

                    @Override
                    public void onFailure(Call<TokenResponse> call, Throwable t) {
                        Log.d(TAG, "onFailure: " + t);
                        t.printStackTrace();
                        solver.reject(t);
                    }
                });
            }
        });
    }

    @NonNull
    @Override
    protected Request.Builder addHeader(@NonNull Request.Builder builder) {
        return builder.addHeader(HEADER_NAME_AUTHORIZATION, Auth64.serialize(appId, secret));
    }
}
