package voxeet.com.sdk.core.abs;

import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaSDK;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.SdpCandidate;
import com.voxeet.android.media.audio.AudioRoute;
import com.voxeet.android.media.peer.PendingPeerCallback;
import com.voxeet.android.media.peer.SdpDescription;
import com.voxeet.android.media.peer.SdpMessage;
import com.voxeet.android.media.video.Camera2Enumerator;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.Camera1Enumerator;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBase;
import org.webrtc.VideoSink;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import voxeet.com.sdk.core.ConferenceSimpleState;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.abs.information.ConferenceInformation;
import voxeet.com.sdk.core.abs.information.ConferenceInformationHolder;
import voxeet.com.sdk.core.abs.information.ConferenceState;
import voxeet.com.sdk.core.abs.promises.CreateConferencePromiseable;
import voxeet.com.sdk.core.abs.promises.DeclinePromise;
import voxeet.com.sdk.core.abs.promises.GetConferenceHistoryPromiseable;
import voxeet.com.sdk.core.abs.promises.GetConferenceStatus;
import voxeet.com.sdk.core.abs.promises.GetConferenceStatusPromiseable;
import voxeet.com.sdk.core.abs.promises.InvitePromise;
import voxeet.com.sdk.core.abs.promises.JoinPromise;
import voxeet.com.sdk.core.abs.promises.LeavePromise;
import voxeet.com.sdk.core.abs.promises.LogoutPromise;
import voxeet.com.sdk.core.abs.promises.ReplayPromise;
import voxeet.com.sdk.core.abs.promises.SendBroadcastMessagePromise;
import voxeet.com.sdk.core.abs.promises.StartRecordingPromiseable;
import voxeet.com.sdk.core.abs.promises.StartScreensharePromise;
import voxeet.com.sdk.core.abs.promises.StartVideoPromise;
import voxeet.com.sdk.core.abs.promises.StopRecordingPromiseable;
import voxeet.com.sdk.core.abs.promises.StopScreenSharePromise;
import voxeet.com.sdk.core.abs.promises.StopVideoPromise;
import voxeet.com.sdk.core.abs.promises.SubscribeConferenceEventPromiseable;
import voxeet.com.sdk.core.abs.promises.SubscribeForCallStartPromiseable;
import voxeet.com.sdk.core.abs.promises.UnsubscribeConferenceEventPromiseable;
import voxeet.com.sdk.core.abs.promises.UnsubscribeForCallStartPromiseable;
import voxeet.com.sdk.core.http.HttpCallback;
import voxeet.com.sdk.core.http.HttpHelper;
import voxeet.com.sdk.core.preferences.VoxeetPreferences;
import voxeet.com.sdk.core.services.AudioService;
import voxeet.com.sdk.core.services.MediaService;
import voxeet.com.sdk.core.services.SdkConferenceService;
import voxeet.com.sdk.core.services.TimeoutRunnable;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;
import voxeet.com.sdk.events.error.CameraSwitchErrorEvent;
import voxeet.com.sdk.events.error.ConferenceCreatedError;
import voxeet.com.sdk.events.error.ConferenceJoinedError;
import voxeet.com.sdk.events.error.HttpException;
import voxeet.com.sdk.events.error.ParticipantAddedErrorEvent;
import voxeet.com.sdk.events.error.PermissionRefusedEvent;
import voxeet.com.sdk.events.promises.InConferenceException;
import voxeet.com.sdk.events.promises.NotInConferenceException;
import voxeet.com.sdk.events.promises.PromiseConferenceJoinedErrorException;
import voxeet.com.sdk.events.promises.PromiseParticipantAddedErrorEventException;
import voxeet.com.sdk.events.success.CameraSwitchSuccessEvent;
import voxeet.com.sdk.events.success.ConferenceCreationSuccess;
import voxeet.com.sdk.events.success.ConferenceDestroyedPushEvent;
import voxeet.com.sdk.events.success.ConferenceEndedEvent;
import voxeet.com.sdk.events.success.ConferenceJoinedSuccessEvent;
import voxeet.com.sdk.events.success.ConferencePreJoinedEvent;
import voxeet.com.sdk.events.success.ConferenceRefreshedEvent;
import voxeet.com.sdk.events.success.ConferenceStatsEvent;
import voxeet.com.sdk.events.success.ConferenceUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUserAddedEvent;
import voxeet.com.sdk.events.success.ConferenceUserCallDeclinedEvent;
import voxeet.com.sdk.events.success.ConferenceUserJoinedEvent;
import voxeet.com.sdk.events.success.ConferenceUserLeftEvent;
import voxeet.com.sdk.events.success.ConferenceUserQualityUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUserUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUsersInvitedEvent;
import voxeet.com.sdk.events.success.DeclineConferenceResultEvent;
import voxeet.com.sdk.events.success.GetConferenceHistoryEvent;
import voxeet.com.sdk.events.success.GetConferenceStatusEvent;
import voxeet.com.sdk.events.success.IncomingCallEvent;
import voxeet.com.sdk.events.success.InvitationReceived;
import voxeet.com.sdk.events.success.OfferCreatedEvent;
import voxeet.com.sdk.events.success.OwnConferenceStartedEvent;
import voxeet.com.sdk.events.success.ParticipantAddedEvent;
import voxeet.com.sdk.events.success.ParticipantUpdatedEvent;
import voxeet.com.sdk.events.success.QualityIndicators;
import voxeet.com.sdk.events.success.QualityUpdatedEvent;
import voxeet.com.sdk.events.success.RecordingStatusUpdate;
import voxeet.com.sdk.events.success.RenegociationEndedEvent;
import voxeet.com.sdk.events.success.RenegociationUpdate;
import voxeet.com.sdk.events.success.ResumeConference;
import voxeet.com.sdk.events.success.ScreenStreamAddedEvent;
import voxeet.com.sdk.events.success.ScreenStreamRemovedEvent;
import voxeet.com.sdk.events.success.SocketConnectEvent;
import voxeet.com.sdk.events.success.StartVideoAnswerEvent;
import voxeet.com.sdk.factories.VoxeetIntentFactory;
import voxeet.com.sdk.json.ConferenceStats;
import voxeet.com.sdk.json.ConferenceUserAdded;
import voxeet.com.sdk.json.InvitationReceivedEvent;
import voxeet.com.sdk.json.JoinParameters;
import voxeet.com.sdk.json.OfferCreated;
import voxeet.com.sdk.json.RecordingStatusUpdateEvent;
import voxeet.com.sdk.json.UserInfo;
import voxeet.com.sdk.json.internal.MetadataHolder;
import voxeet.com.sdk.json.internal.ParamsHolder;
import voxeet.com.sdk.models.CandidatesPush;
import voxeet.com.sdk.models.ConferenceQuality;
import voxeet.com.sdk.models.ConferenceResponse;
import voxeet.com.sdk.models.ConferenceType;
import voxeet.com.sdk.models.ConferenceUserStatus;
import voxeet.com.sdk.models.NormalConferenceResponse;
import voxeet.com.sdk.models.OfferCandidate;
import voxeet.com.sdk.models.OfferDescription;
import voxeet.com.sdk.models.RecordingStatus;
import voxeet.com.sdk.models.abs.Conference;
import voxeet.com.sdk.models.abs.ConferenceUser;
import voxeet.com.sdk.models.impl.DefaultConference;
import voxeet.com.sdk.models.impl.DefaultConferenceUser;
import voxeet.com.sdk.models.impl.DefaultInvitation;
import voxeet.com.sdk.networking.DeviceType;
import voxeet.com.sdk.utils.ConferenceListener;
import voxeet.com.sdk.utils.Twig;
import voxeet.com.sdk.utils.Validate;

/**
 * Conference SDK Service
 * <p>
 * Warning : always implements the error for each promises you may call to start resolve them
 * You can also use the execute method but in case of Exception, you will trigger errors...
 *
 * @see SdkConferenceService
 */
public abstract class AbstractConferenceSdkService<T, COP extends AbstractConferenceSdkObservableProvider<T>>
        extends voxeet.com.sdk.core.AbstractVoxeetService<T>
        implements SdkConferenceService, ConferenceListener {

    private final static String TAG = AbstractConferenceSdkService.class.getSimpleName();
    private final VoxeetSdkTemplate mInstance;
    private VoxeetSdkTemplate mSDK;
    private ConferenceListener mListener;
    private AbstractConferenceSdkObservableProvider mConferenceObservableProvider;
    private Twig mTwig;

    private EventBus mEventBus;
    private boolean isRecording = false;
    private boolean mInConference = false;
    private long mTimeOutTimer = -1;
    private HashMap<String, MediaStream> mapOfStreams = new HashMap<>();
    private HashMap<String, MediaStream> mapOfScreenShareStreams = new HashMap<>();
    private ReentrantLock joinLock = new ReentrantLock();

    @Nullable
    private String mDefaultCamera;
    private boolean isDefaultOnSpeaker;

    private String mConferenceId = null;
    private ConferenceInformationHolder mConferenceInformationHolder = new ConferenceInformationHolder();

    private Context mContext;

    private @Nullable
    TimeoutRunnable timeoutRunnable = null;

    private MediaEngine.StreamListener mediaStreamListener = new MediaEngine.StreamListener() {
        @Override
        public void onStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onStreamAdded: stream for peer " + peer);
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    getTwig().i("New mConference user joined with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());

                    mapOfStreams.put(peer, stream);
                    DefaultConferenceUser user = updateConferenceParticipants(peer, ConferenceUserStatus.ON_AIR);
                    if (user != null) {
                        getTwig().i("New mConference user joined with id: " + user.getUserId());

                        if (!peer.equalsIgnoreCase(VoxeetPreferences.id()) && mTimeOutTimer != -1) {
                            mTwig.i("Cancelling timeout timer");
                            removeTimeoutCallbacks();
                        }

                        //update the current conference state specificly from the users point of view
                        updateConferenceFromUsers();
                        mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    } else {
                        Log.d(TAG, "run: unknown user in stream added");
                    }
                }
            });
        }

        @Override
        public void onStreamUpdated(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onStreamUpdated: screen for peer " + peer);
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    getTwig().i("Conference user updated with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());
                    getTwig().i("stream updated having video ? " + (stream.videoTracks().size() > 0) + " screenshare := " + stream.isScreenShare());

                    if (stream.isScreenShare()) mapOfScreenShareStreams.put(peer, stream);
                    else mapOfStreams.put(peer, stream);

                    DefaultConferenceUser user = findUserById(peer);
                    if (user != null) {
                        getTwig().i("Conference user updated with id: " + user.getUserId());

                        //update the current conference state specificly from the users point of view
                        updateConferenceFromUsers();
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    } else {
                        Log.d(TAG, "run: unknown user in stream updated");
                    }
                }
            });
        }

        @Override
        public void onStreamRemoved(@NonNull final String peer) {
            Log.d(TAG, "onStreamRemoved: OnStreamRemoved");
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    onUserLeft(peer);
                }
            });
        }

        @Override
        public void onScreenStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onScreenStreamAdded: screen for peer " + peer);
            getTwig().i("(for ScreenShare) Conference user updated with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());
            getTwig().i("screenshare stream updated having video ? " + stream.videoTracks().size() + " :: has stream inside ? " + (stream.videoTracks().size() > 0));

            mapOfScreenShareStreams.put(peer, stream);
            mEventBus.post(new ScreenStreamAddedEvent(peer, stream));
        }

        @Override
        public void onScreenStreamRemoved(@NonNull String peer) {
            getTwig().i("Screen share stream removed: " + peer);

            mapOfScreenShareStreams.remove(peer);
            mEventBus.post(new ScreenStreamRemovedEvent(peer));
        }

        @Override
        public void onShutdown() {

        }

        @Override
        public void onIceCandidateDiscovered(String peer, SdpCandidate[] candidates) {

            if (null == getConferenceId()) return;

            Call<ResponseBody> call = mConferenceObservableProvider.candidates(getConferenceId(),
                    peer,
                    new CandidatesPush(candidates));

            call.enqueue(new Callback<ResponseBody>() {
                @Override
                public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
                    Log.d(TAG, "onResponse: Candidates sent ");
                }

                @Override
                public void onFailure(Call<ResponseBody> call, Throwable t) {
                    t.printStackTrace();
                }
            });
        }
    };

    private CameraVideoCapturer.CameraEventsHandler cameraEventsHandler = new CameraVideoCapturer.CameraEventsHandler() {
        @Override
        public void onCameraError(String s) {
            Log.d(TAG, "onCameraError: error...");

            stopVideo().then(new PromiseExec<Boolean, Object>() {
                @Override
                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                    Log.d(TAG, "onCall: stopped video after camera issue " + result);
                }
            }).error(new ErrorPromise() {
                @Override
                public void onError(@NonNull Throwable error) {
                    error.printStackTrace();
                }
            });
        }

        @Override
        public void onCameraDisconnected() {
            Log.d(TAG, "onCameraDisconnected");
        }

        @Override
        public void onCameraFreezed(String s) {
            Log.d(TAG, "onCameraFreezed: " + s);
        }

        @Override
        public void onCameraOpening(String s) {
            Log.d(TAG, "onCameraOpening: " + s);
        }

        @Override
        public void onFirstFrameAvailable() {
            Log.d(TAG, "onFirstFrameAvailable");
        }

        @Override
        public void onCameraClosed() {
            Log.d(TAG, "onCameraClosed");
        }
    };

    private boolean isDefaultMute = false;
    private boolean isICERestartEnabled = false;

    public AbstractConferenceSdkService(VoxeetSdkTemplate instance, COP conference_observable_provider, long timeout, @NonNull ServiceProviderHolder<T> holder) {
        super(instance, holder);

        mInstance = instance;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mDefaultCamera = new Camera2Enumerator(instance.getApplicationContext()).getNameOfFrontFacingDevice();
        } else {
            Camera1Enumerator enumerator = new Camera1Enumerator(false);

            String[] names = enumerator.getDeviceNames();
            if (null != names && names.length > 0) {
                for (String name : names) {
                    if (enumerator.isFrontFacing(name)) mDefaultCamera = name;
                }
            }
        }

        mConferenceObservableProvider = conference_observable_provider;
        mConferenceObservableProvider.setRetrofitInstantiatedProvider(getService());
        mListener = instance;
        mSDK = instance;
        mTimeOutTimer = timeout;
        mTwig = instance.getTwig();
        mEventBus = EventBus.getDefault();
        mContext = instance.getApplicationContext();
        setDefaultBuiltInSpeaker(true);


        register();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     * @deprecated Please use {@link #mute(boolean)} ()} instead.
     */
    @Deprecated
    public boolean muteConference(boolean mute) {
        return mute(mute);
    }

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     */
    @Override
    public boolean mute(boolean mute) {
        if (getMediaService().hasMedia()) {
            if (!mute && getMediaService().getMedia().isMuted()) {
                if (!Validate.hasMicrophonePermissions(context)) {
                    getTwig().i("No permission for mic... please check it");
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                } else {
                    getTwig().i("Conference unmuted");
                    getAudioService().setInVoiceCallSoundType();

                    getMedia().unMute();
                }
            } else if (mute) {
                getTwig().i("Conference muted");

                getAudioService().setInVoiceCallSoundType();
                getMedia().mute();
            }
        }
        return true;
    }

    @Override
    public boolean isMuted() {
        return getMedia() != null && getMedia().isMuted();
    }

    @Override
    public boolean isUserMuted(String userId) {
        DefaultConferenceUser user = findUserById(userId);
        return user != null && user.isMuted();
    }

    @Override
    public void setListenerMode(boolean isListener) {
        if (getMediaService().hasMedia()) {
            MediaSDK media = getMediaService().getMedia();
            if (media.isMuted()) {
                media.unMute();
            } else {
                media.mute();
            }
            getAudioService().setInVoiceCallSoundType();
        }
    }

    //TODO WEBRTC V72
    @Deprecated
    @Override
    public boolean attachMediaStream(@NonNull MediaStream stream, @NonNull VideoSink videoSink) {
        if (getMediaService().hasMedia()) getMedia().attachMediaStream(videoSink, stream);
        return getMediaService().hasMedia();
    }

    //TODO WEBRTC V72
    @Deprecated
    @Override
    public boolean unAttachMediaStream(@NonNull MediaStream stream, @NonNull VideoSink videoSink) {
        if (getMediaService().hasMedia()) getMedia().unattachMediaStream(videoSink, stream);
        return getMediaService().hasMedia();
    }

    @Override
    public void register() {
        registerEventBus();
    }

    @Override
    public void unregister() {
        unRegisterEventBus();
    }

    public void setDefaultCamera(String cameraName) {
        mDefaultCamera = cameraName;
    }

    @Override
    public void toggleVideo() {
        Promise<Boolean> promise = isVideoOn() ? stopVideo() : startVideo();
        promise.then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onSuccess: toggleVideo " + result);
            }
        }).error(ConferenceServiceHelper.manageError());
    }

    public void toggleScreenShare() {
        if (isScreenShareOn()) {
            stopScreenShare().then(new PromiseExec<Boolean, Object>() {
                @Override
                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                    Log.d(TAG, "onSuccess: toggleScreenShare " + result);
                }
            }).error(ConferenceServiceHelper.manageError());
        } else {
            getVoxeetSDK().getScreenShareService().sendRequestStartScreenShare();
        }
    }

    @Override
    public String getCurrentConferenceId() {
        return mConferenceId;
    }

    @Override
    public int getConferenceRoomSize() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            return information.getConference().getConferenceRoomSize();
        }
        return 0;
    }

    @Override
    public String currentSpeaker() {
        Conference conference = getConference();
        if (!hasMedia() || conference == null) {
            return VoxeetPreferences.id();
        } else {
            String currentSpeaker = null;
            for (DefaultConferenceUser user : conference.getConferenceUsers()) {
                if (user.getUserId() != null
                        && !user.getUserId().equals(VoxeetPreferences.id())
                        && ConferenceUserStatus.ON_AIR.equals(user.getConferenceStatus())) {
                    double peerVuMeter = getMedia().getPeerVuMeter(user.getUserId());
                    if (currentSpeaker == null || (peerVuMeter > 0.001 && getMedia().getPeerVuMeter(currentSpeaker) < peerVuMeter))
                        currentSpeaker = user.getUserId();
                }
            }
            return currentSpeaker;
        }
    }

    @NonNull
    private String currentUserOrEmpty() {
        String currentUserId = VoxeetPreferences.id();
        return null != currentUserId ? currentUserId : "";
    }

    @Override
    public double getSdkPeerVuMeter(@Nullable String peerId) {
        Validate.runningOnUiThread();
        return hasMedia() && null != peerId ? getMedia().getPeerVuMeter(peerId) : 0;
    }

    @Override
    public DefaultConferenceUser findUserById(final String userId) {
        DefaultConference conference = getConference();
        return null != conference ? conference.findUserById(userId) : null;
    }

    @Override
    public String getAliasId() {
        DefaultConference conference = getConference();
        return null != conference ? conference.getConferenceAlias() : null;
    }

    @Override
    public String getConferenceId() {
        return mConferenceId;
    }

    @Deprecated
    public List<AudioRoute> getAvailableRoutes() {
        return getAudioService().getAvailableRoutes();
    }

    @Deprecated
    public AudioRoute currentRoute() {
        return getAudioService().currentRoute();
    }

    @Override
    @Nullable
    public EglBase.Context getEglContext() {
        return getMediaService().getEglContext();
    }

    @Override
    public long getTimeout() {
        return mTimeOutTimer;
    }

    @Override
    public void toggleRecording() {
        Promise<Boolean> promise = isRecording ? stopRecording() : startRecording();
        promise.then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onSuccess: toggle done " + result);
            }
        }).error(new ErrorPromise() {
            @Override
            public void onError(Throwable error) {
                error.printStackTrace();
            }
        });
    }

    @Override
    public boolean muteUser(@NonNull String userId, boolean shouldMute) {
        Validate.runningOnUiThread();

        DefaultConferenceUser user = findUserById(userId);
        if (user != null) {
            getTwig().i("Setting mute property for mConference participant with id " + userId + " to " + shouldMute);

            startTransactionConferenceUser();
            user.setMuted(shouldMute);
            getMedia().changePeerGain(userId, shouldMute ? MUTE_FACTOR : UNMUTE_FACTOR);
            commitTransactionConferenceUser();
        }
        return user != null;
    }

    @Deprecated
    public boolean isVideoOn() {
        ConferenceInformation information = getCurrentConferenceInformation();
        return null != information && information.isOwnVideoStarted();
    }

    @Deprecated
    public boolean isScreenShareOn() {
        ConferenceInformation information = getCurrentConferenceInformation();
        return null != information && information.isScreenShareOn();
    }

    @Deprecated
    public boolean setAudioRoute(AudioRoute route) {
        return getAudioService().setAudioRoute(route);
    }

    public boolean setDefaultMute(boolean default_state) {
        isDefaultMute = default_state;
        return true;
    }

    public boolean setDefaultBuiltInSpeaker(boolean default_state) {
        isDefaultOnSpeaker = default_state;
        VoxeetPreferences.setDefaultBuiltInSpeakerOn(default_state);
        return true;
    }

    @Override
    public boolean setTimeOut(long timeout) {
        getTwig().i("Timeout set to " + timeout);
        mTimeOutTimer = timeout;
        return true;
    }

    @Override
    public boolean setUserPosition(final String userId, final double angle, final double distance) {
        if (hasMedia()) getMedia().changePeerPosition(userId, angle, distance);
        return hasMedia();
    }

    /**
     * Get the current conference type
     *
     * @return a ConferenceType, ConferenceType.NONE is the default value
     */
    @NonNull
    public ConferenceSimpleState getConferenceType() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information || null == mConferenceId) return ConferenceSimpleState.NONE;
        return information.getConferenceType();
    }

    /**
     * Get the current mConference object
     *
     * @return a nullable object
     */
    @Nullable
    public DefaultConference getConference() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return null;

        return information.getConference();
    }

    @NonNull
    private DefaultConference getOrSetConference(@NonNull String conferenceId) {
        ConferenceInformation information = getConferenceInformation(conferenceId);
        if (null != information) return information.getConference();
        Log.d(TAG, "getOrSetConference: INVALID CALL FOR GET OR SET CONFERENCE WITH " + conferenceId);
        return new DefaultConference();
    }

    /**
     * Get the current state of joining
     * in conjunction of the getConference() it is easy to check the current state :
     * <p>
     * mInConference() + getConference() non null = mConference joined and on air
     * mInConference() + getConference() null = joining mConference with the answer from server for now
     * !mInConference() = not currently joining, joined mConference (or left)
     *
     * @return if in mConference or "attempting" to join
     */
    public boolean isInConference() {
        return mInConference;
    }

    protected void setIsInConference(boolean status) {
        mInConference = status;
    }

    public double getPeerVuMeter(@Nullable String peerId) {
        Validate.runningOnUiThread();

        if (hasMedia() && null != peerId) return getMedia().getPeerVuMeter(peerId);
        return 0.;
    }

    @NonNull
    @Override
    public List<DefaultConferenceUser> getConferenceUsers() {
        DefaultConference conference = getConference();
        if (null == conference) {
            Log.d(TAG, "getConferenceUsers: returning a new instance :: NOT IN A CONFERENCE");
            return new ArrayList<>();
        }

        return conference.getConferenceUsers();
    }

    @NonNull
    public List<DefaultConferenceUser> getLastInvitationUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return new ArrayList<>();
        return information.getLastInvitationReceived();
    }

    @Override
    public boolean isLive() {
        return null != getConference();
    }

    @Override
    public boolean isListenerMode() {
        ConferenceInformation information = getCurrentConferenceInformation();
        return null != information && information.isListener();
    }

    @Nullable
    public ConferenceUser getUser(final String userId) {
        Conference conference = getConference();
        if (null != conference) {
            return Iterables.find(conference.getConferenceUsers(), new Predicate<ConferenceUser>() {
                @Override
                public boolean apply(ConferenceUser input) {
                    return userId.equals(input.getUserId());
                }
            }, null);
        }
        return null;
    }

    /**
     * Deprecated since it will possibly be removed in the near future
     *
     * @return the map of streams available
     */
    @Deprecated
    @NonNull
    public HashMap<String, MediaStream> getMapOfStreams() {
        return mapOfStreams;
    }

    /**
     * Deprecated since it will possibly be removed in the near future
     *
     * @return the map of screenshare streams available
     */
    @Deprecated
    @NonNull
    public HashMap<String, MediaStream> getMapOfScreenShareStreams() {
        return mapOfScreenShareStreams;
    }

    /**
     * Give the ability to cancel the timeout in some cases
     */
    public AbstractConferenceSdkService<T, COP> cancelTimeout() {
        removeTimeoutCallbacks();

        return this;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected abstract Promise<Boolean> onCreateDemoSuccess(ConferenceResponse response);

    /**
     * Join a conference with a conference
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     *
     * @param conferenceId id of the conference to join as a listener
     * @return a promise to resolve
     */
    @Override
    public Promise<Boolean> listenConference(@NonNull String conferenceId) {
        return join(conferenceId, true);
    }

    /**
     * resolve -> the result event
     * reject -> network error
     *
     * @param conferenceId the conference id to decline
     * @return a promise to resolve
     */
    @Override
    public Promise<DeclineConferenceResultEvent> decline(final String conferenceId) {
        return new DeclinePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId).createPromise();
    }

    public void sendRequestStartScreenShare() {
        getVoxeetSDK().getScreenShareService().sendRequestStartScreenShare();
    }

    public void onUserCanceledScreenShare() {
        //call before any specific code so stopScreenCapturer() should do nothing /!\
        if (hasMedia()) {
            getMedia().stopScreenCapturer();
        }
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            information.setScreenShareOn(false);
        }
    }

    public Promise<Boolean> startScreenShare(@NonNull final Intent intent,
                                             final int width,
                                             final int height) {
        return new StartScreensharePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                intent,
                width,
                height).createPromise();
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopScreenShare() {
        return new StopScreenSharePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * Start the video
     * resolve -> StartVideoAnswerEvent
     * reject -> PromisePermissionRefusedEventException
     * reject -> MediaException
     * <p>
     * If the app does not have the permission, will fire a PermissionRefusedEvent
     */
    public Promise<Boolean> startVideo() {
        return new StartVideoPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopVideo() {
        return new StopVideoPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * Create a video answer and post it
     * <p>
     * resolve -> true -> everything went fine
     * resolve -> false -> on error occured
     * reject -> media exception
     *
     * @param userId
     * @param offerDescription
     * @param offerCandidates
     * @return
     */
    protected Promise<Boolean> createVideoAnswer(final String userId, final OfferDescription offerDescription, final List<OfferCandidate> offerCandidates) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Log.d(TAG, "createVideoAnswer: " + userId + " " + offerDescription.getSdp() + " " + offerDescription.getType());
                Log.d("SDKMEDIA", "createVideoAnswer: " + userId + " " + offerDescription.getSdp() + " " + offerDescription.getType());

                try {
                    if (!hasMedia()) {
                        throw new MediaEngineException("media is null");
                    }
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                }

                SdpDescription description = new SdpDescription(offerDescription.getType(), offerDescription.getSdp());

                List<SdpCandidate> candidates = new ArrayList<>();
                if (offerCandidates != null) {
                    for (OfferCandidate candidate : offerCandidates) {
                        candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                    }
                }

                try {
                    getMedia().createAnswerForPeer(userId,
                            description.getSsrc(),
                            description,
                            candidates,
                            VoxeetPreferences.id().equalsIgnoreCase(userId),
                            new PendingPeerCallback() {
                                @Override
                                public void onMessage(@Nullable SdpMessage message) {
                                    Log.d(TAG, "onMessage: having a message " + message);
                                    answer(userId, message)
                                            .then(new PromiseExec<Integer, Object>() {
                                                @Override
                                                public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                    Log.d(TAG, "onSuccess: " + result);

                                                    //resolve true value : all clear
                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    if (error instanceof PromiseParticipantAddedErrorEventException)
                                                        mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                    else
                                                        error.printStackTrace();

                                                    //resolve false value : something happened
                                                    solver.resolve(false);
                                                }
                                            });
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @return a promise to resolve
     */
    @Override
    public Promise<ConferenceResponse> create() {
        return create(null, null, null);
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param metadata the MetadataHolder
     * @return a promise to resolve
     */
    public Promise<ConferenceResponse> create(@Nullable MetadataHolder metadata,
                                              @Nullable ParamsHolder params) {
        return create(null, metadata, params);
    }


    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceAlias create a given conference then join it
     * @return a promise to resolve
     */
    public Promise<ConferenceResponse> create(@Nullable final String conferenceAlias) {
        return create(conferenceAlias, null, null);
    }

    public Promise<ConferenceResponse> create(@Nullable String conferenceAlias,
                                              @Nullable MetadataHolder metadata,
                                              @Nullable ParamsHolder paramsholder) {
        return new CreateConferencePromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                conferenceAlias,
                metadata,
                paramsholder).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceId the existing conference to join
     * @return a promise to resolve
     */
    public Promise<Boolean> join(@NonNull final String conferenceId) {
        return join(conferenceId, false);
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceId the existing conference to join
     * @param listener     the participant type
     * @return a promise to resolve
     */
    private Promise<Boolean> join(@NonNull final String conferenceId, boolean listener) {
        Log.d(TAG, "join: joining conference " + conferenceId + " listener:=" + listener);
        return new JoinPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                mInstance,
                conferenceId,
                listener).createPromise();
    }


    protected Promise<ResumeConference> extendedJoin(final String conferenceId) {
        return new Promise<>(new PromiseSolver<ResumeConference>() {
            @Override
            public void onCall(@NonNull final Solver<ResumeConference> solver) {
                setIsInConference(true);
                mConferenceId = conferenceId;
                final ConferenceInformation information = getCurrentConferenceInformation();


                Call<ResumeConference> observable = mConferenceObservableProvider.joinConference(conferenceId, new JoinParameters(DeviceType.ANDROID, information.isListener()));
                HttpHelper.enqueue(observable, new HttpCallback<ResumeConference>() {
                    @Override
                    public void onSuccess(@NonNull ResumeConference object, @NonNull Response<ResumeConference> response) {
                        solver.resolve(object);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable t, @Nullable Response<ResumeConference> response) {
                        HttpException.dumpErrorResponse(response);

                        solver.reject(t);
                    }
                });
            }
        });
    }


    protected Promise<Boolean> onConferenceCreated(@NonNull final String conferenceId,
                                                   @NonNull final String conferenceAlias,
                                                   @Nullable final NormalConferenceResponse other) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                createOrSetConferenceWithParams(conferenceId, conferenceAlias);
                getConferenceInformation(conferenceId).setConferenceState(ConferenceState.CREATED);

                getTwig().i("IConference created with id: " + mConferenceId);

                mEventBus.post(new ConferenceCreationSuccess(conferenceId, conferenceAlias));

                joinVoxeetConference(conferenceId).then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * resole -> true
     * reject -> network error or on created error
     *
     * @return
     */
    @Override
    public Promise<ConferenceResponse> demo() {
        return new Promise<>(new PromiseSolver<ConferenceResponse>() {
            @Override
            public void onCall(@NonNull final Solver<ConferenceResponse> solver) {
                getTwig().i("Attempting to create demo mConference");

                final Call<ConferenceResponse> user = mConferenceObservableProvider.getCreateDemoObservable();
                HttpHelper.enqueue(user, new HttpCallback<ConferenceResponse>() {
                    @Override
                    public void onSuccess(@NonNull ConferenceResponse object, @NonNull Response<ConferenceResponse> response) {
                        onCreateDemoSuccess(object)
                                .then(new PromiseExec<Boolean, Object>() {
                                    @Override
                                    public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                        ConferenceInformation information = getCurrentConferenceInformation();
                                        information.setConferenceType(ConferenceSimpleState.CONFERENCE);

                                        solver.resolve(object);
                                    }
                                })
                                .error(new ErrorPromise() {
                                    @Override
                                    public void onError(Throwable error) {
                                        solver.reject(error);
                                    }
                                });
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ConferenceResponse> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e(e);

                        mEventBus.post(new ConferenceCreatedError(handleError(e)));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> list of ConferenceRefreshedUsers
     * reject -> network error
     *
     * @param userInfos a non null list of non null strings
     * @return a non null promise
     */
    @NonNull
    public Promise<List<ConferenceRefreshedEvent>> inviteUserInfos(final String conferenceId, final List<UserInfo> userInfos) {
        return new Promise<>(new PromiseSolver<List<ConferenceRefreshedEvent>>() {
            @Override
            public void onCall(@NonNull Solver<List<ConferenceRefreshedEvent>> solver) {
                //Warning = getConferenceUsers is returning a new non retained if not in a conference
                List<DefaultConferenceUser> users = getConferenceUsers();
                List<String> strings = new ArrayList<>();

                if (null != userInfos) {
                    for (UserInfo info : userInfos) {
                        if (null != info && info.getExternalId() != null) {
                            int i = 0;
                            while (i < users.size()) {
                                DefaultConferenceUser user = users.get(i);
                                if (null != user && null != user.getUserInfo()
                                        && info.getExternalId().equalsIgnoreCase(user.getUserInfo().getExternalId())) {
                                    user.updateIfNeeded(info.getName(), info.getAvatarUrl());
                                }
                                i++;
                            }
                            strings.add(info.getExternalId());
                        }
                    }
                }

                solver.resolve(invite(conferenceId, strings));
            }
        });
    }

    /**
     * resolve -> list of ConferenceRefreshedUsers
     * reject -> network error
     *
     * @param conferenceId
     * @param ids          a non null list of non null strings
     * @return a non null promise
     */
    @NonNull
    @Deprecated
    @Override
    public Promise<List<ConferenceRefreshedEvent>> invite(final String conferenceId, final List<String> ids) {
        return new InvitePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                conferenceId,
                ids).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false response from server is an error
     * reject -> network error
     *
     * @return
     */
    @Override
    public Promise<Boolean> logout() {
        return new LogoutPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                null,
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false in case of error
     * reject -> dangerous issue
     *
     * @param conferenceId the mConference id
     * @param offset
     * @return
     */
    @Override
    public Promise<Boolean> replay(final String conferenceId, final long offset) {
        return new ReplayPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId,
                offset).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false in case of non 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> startRecording() {
        return new StartRecordingPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false not a 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> stopRecording() {
        return new StopRecordingPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * Get the current mConference status
     * send events but also managed by the promise
     * <p>
     * resolve -> GetConferenceStatusEvent
     * reject -> error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<GetConferenceStatusEvent> getConferenceStatus(final String conferenceId) {
        return new GetConferenceStatusPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> event
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<GetConferenceHistoryEvent> conferenceHistory(final String conferenceId) {
        return new GetConferenceHistoryPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> none
     *
     * @return
     */
    @Override
    public Promise<Boolean> switchCamera() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(getMedia(), "media");

                getMedia().switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean isFrontCamera) {
                        getTwig().i("Successfully switched camera");

                        mEventBus.post(new CameraSwitchSuccessEvent(isFrontCamera));
                        solver.resolve(true);
                    }

                    @Override
                    public void onCameraSwitchError(String errorDescription) {
                        getTwig().e("Failed to switch camera " + errorDescription);

                        mEventBus.post(new CameraSwitchErrorEvent(errorDescription));
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<Boolean> subscribe(@NonNull final String conferenceId) {
        return new SubscribeConferenceEventPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @return a valid promise
     */
    @Override
    public Promise<Boolean> unSubscribe(@NonNull final String conferenceId) {
        return new UnsubscribeConferenceEventPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return
     */
    @Override
    public Promise<Boolean> subscribeForCall(final String conferenceId) {
        return new SubscribeForCallStartPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return
     */
    @Override
    public Promise<Boolean> unSubscribeFromCall(final String conferenceId) {
        return new UnsubscribeForCallStartPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     *
     * @param conferenceId the mConference id
     * @return a promise to catch
     */
    @Deprecated
    @NonNull
    public Promise<Boolean> joinUsingConferenceId(@NonNull final String conferenceId) {
        return joinVoxeetConference(conferenceId);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(@NonNull SocketConnectEvent event) {
        String conferenceId = getConferenceId();
        ConferenceInformation information = getCurrentConferenceInformation();

        if (null != conferenceId && null != information && ConferenceState.JOINED.equals(information.getConferenceState())) {
            if (isICERestartEnabled()) {
                Log.d(TAG, "onEvent: SocketConnectEvent Joined <3");
                final Call<ResponseBody> user = mConferenceObservableProvider.iceRestart(conferenceId);
                HttpHelper.enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        Log.d(TAG, "onEvent: SocketConnectEvent Joined responded <3");
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        Log.d(TAG, "onEvent: SocketConnectEvent Joined error </3");
                        HttpException.dumpErrorResponse(response);
                    }
                });
            } else {
                Log.d(TAG, "onEvent: socket state opened while in conference but no isICERestartEnabled() = true. A reconnect may be longer");
            }
        } else {
            ConferenceState state = ConferenceState.DEFAULT;
            if (null != information) {
                state = information.getConferenceState();
            }
            Log.d(TAG, "onEvent: SocketConnectEvent not rejoined </3 " + state + " " + conferenceId);
        }
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     *
     * @param conferenceId the mConference id
     * @return a promise to catch
     */
    @Deprecated
    @NonNull
    @Override
    public Promise<Boolean> joinVoxeetConference(@NonNull final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                String existingConferenceId = getConferenceId();

                Log.d(TAG, "onCall: " + mInConference + " " + existingConferenceId + " " + conferenceId);
                if (mInConference && (null == existingConferenceId || !existingConferenceId.equals(conferenceId))) {
                    try {
                        throw new InConferenceException();
                    } catch (InConferenceException exception) {
                        solver.reject(exception);
                        return;
                    }
                }

                //remove the timeout, a new one should be started when user will be invited
                removeTimeoutCallbacks();

                Log.d(TAG, "joining " + conferenceId);

                setIsInConference(true);
                mConferenceId = conferenceId;
                final ConferenceInformation information = getCurrentConferenceInformation();

                if (null != information) {
                    solver.resolve(joinConference(information));
                } else {
                    Log.d(TAG, "onCall: WARNING :: IMPROPER RESULT !!");
                    solver.resolve(false);
                }

            }
        });
    }


    protected void setCurrentConferenceIfNotInPreviousConference(@NonNull ConferenceInformation conference) {
        String conferenceId = conference.getConference().getConferenceId();

        boolean invalidId = null == conferenceId;
        boolean inConference = isInConference();
        boolean sameConference = null == mConferenceId || mConferenceId.equals(conferenceId);
        boolean canOverride = !inConference || sameConference;

        if (!invalidId && canOverride) {
            mConferenceId = conferenceId;
        }

        Log.d(TAG, "setCurrentConferenceIfNotInPreviousConference: "
                + " invalidId = " + invalidId
                + " inConference = " + inConference
                + " sameConference = " + sameConference + " (" + mConferenceId + " " + conferenceId + ") "
                + " canOverride = " + canOverride
                + " mConferenceId = " + mConferenceId);
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     * <p>
     * Warning : if multiple attempt to join the same conference are made, it is not managed...
     * May be fix in a future version but currently not in the flow
     *
     * @param conference the conference to join
     * @return a promise to catch
     */
    @NonNull
    protected Promise<Boolean> joinConference(@NonNull final ConferenceInformation conference) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                String conferenceId = conference.getConference().getConferenceId();

                boolean invalidId = null == conferenceId;
                boolean inConference = isInConference();
                boolean sameConference = null == mConferenceId || mConferenceId.equals(conferenceId);

                Log.d(TAG, "onCall: join conference " + conference
                        + " invalidId:=" + invalidId
                        + " inConference:=" + inConference
                        + " sameConference:=" + sameConference
                        + " canOverride:=" + (inConference && !sameConference));
                if (invalidId || (inConference && !sameConference)) {
                    try {
                        throw new InConferenceException();
                    } catch (InConferenceException exception) {
                        solver.reject(exception);
                        return;
                    }
                }

                //remove the timeout, a new one should be started when user will be invited
                removeTimeoutCallbacks();

                Log.d(TAG, "joining " + conferenceId);

                setIsInConference(true);
                mConferenceId = conferenceId;

                conference.setConferenceState(ConferenceState.JOINING);

                AudioService service = mInstance.getAudioService();
                if (null != service) {
                    service.enableAec(true);
                    service.enableNoiseSuppressor(true);
                }

                boolean isSuccess = initMedia(conference.isListener());

                if (!isSuccess) {
                    Log.d(TAG, "onCall: InitMedia failed... new state = left");
                    try {
                        setIsInConference(false);
                        conference.setConferenceState(ConferenceState.LEFT);
                        closeMedia();
                        ConferenceJoinedError error = new ConferenceJoinedError(handleError(null));
                        mEventBus.post(error);
                        throw new PromiseConferenceJoinedErrorException(error, null);
                    } catch (PromiseConferenceJoinedErrorException exception) {
                        solver.reject(exception);
                    }
                    return;
                }

                mEventBus.post(new ConferencePreJoinedEvent(getConferenceId(), getAliasId()));

                final Call<ResumeConference> user = mConferenceObservableProvider.joinConference(conferenceId, new JoinParameters(DeviceType.ANDROID, conference.isListener()));

                HttpHelper.enqueue(user, new HttpCallback<ResumeConference>() {
                    @Override
                    public void onSuccess(@NonNull ResumeConference object, @NonNull Response<ResumeConference> response) {
                        createOrSetConferenceWithParams(object.getConferenceId(),
                                object.getConferenceAlias());
                        initMedia(conference.isListener());

                        ConferenceInformation information = getCurrentConferenceInformation();
                        if (null != information) {
                            information.setConferenceState(ConferenceState.JOINED);
                        }

                        onConferenceResumedInternal(object, solver);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResumeConference> response) {
                        HttpException.dumpErrorResponse(response);

                        try {
                            e.printStackTrace();
                            getTwig().e("Failed to Join mConference with id " + conferenceId);

                            setIsInConference(false);

                            //ConferenceInformation information = getCurrentConferenceInformation();
                            conference.setConferenceState(ConferenceState.LEFT);
                            closeMedia();
                            ConferenceJoinedError error = new ConferenceJoinedError(handleError(e));
                            mEventBus.post(error);
                            throw new PromiseConferenceJoinedErrorException(error, e);
                        } catch (PromiseConferenceJoinedErrorException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }

    protected void onConferenceResumedInternal(ResumeConference response, Solver<Boolean> solver) {
        mConferenceId = response.getConferenceId();

        ConferenceInformation information = getCurrentConferenceInformation();

        //information should not be null in this case

        DefaultConference conference = information.getConference();

        conference.setConferenceId(response.getConferenceId());
        conference.setConferenceAlias(response.getConferenceId());

        if (response.getConferenceAlias() != null) {
            conference.setConferenceAlias(response.getConferenceAlias());
        }

        information.participantsToConferenceUsers(response.getParticipants());

        setIsInConference(true);

        setAudioRoute(AudioRoute.ROUTE_PHONE);

        getTwig().i("Joined mConference with id " + getConferenceId());

        mEventBus.post(new ConferenceJoinedSuccessEvent(getConferenceId(), getAliasId()));
        solver.resolve(true);
    }

    /**
     * resolve -> the event with the list
     * reject -> any error
     *
     * @return
     */
    @Override
    public Promise<ConferenceUsersInvitedEvent> getInvitedUsers(@NonNull String conferenceId) {
        return new GetConferenceStatus(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * Send a message. Deprecated, use the conferenceId/message signature
     *
     * @param message
     * @return
     */
    @Override
    @Deprecated
    public Promise<Boolean> sendBroadcastMessage(@NonNull String message) {
        return sendBroadcastMessage(getConferenceId(), message);
    }

    public Promise<Boolean> sendBroadcastMessage(@NonNull String conferenceId, @NonNull final String message) {
        return new SendBroadcastMessagePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId,
                message).createPromise();
    }

    /**
     * This method attempts to leave the current mConference
     * <p>
     * The previous version used a boolean, the new method fingerprint is now void
     */
    @Override
    public Promise<Boolean> leave() {
        return new LeavePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus())
                .createPromise();
    }


    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Event management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceStatsEvent event) {
        ConferenceStats stats = event.getEvent();
        if (stats.getConference_id() != null && stats.getConference_id().equals(getConferenceId())) {
            float mos = stats.getScore(VoxeetPreferences.id());

            mEventBus.post(new QualityIndicators(mos));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUpdatedEvent event) {
        //TODO maybe manage users here too ?
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceived invitation) {
        try {
            onEvent(invitation.getEvent());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceivedEvent invitation) {

        String conferenceId = invitation.getConferenceId();

        if (null == conferenceId && null != invitation.getConference()) {
            conferenceId = invitation.getConference().getConferenceId();
        }

        Log.d(TAG, "onEvent: current mConferenceId " + mConferenceId + " vs " + invitation.getConferenceId() + " vs " + conferenceId);

        if (null != mConferenceId && mConferenceId.equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for our conference, we prevent this");
            return;
        }

        //we update the information for this conference only
        final ConferenceInformation information = getConferenceInformation(conferenceId);

        DefaultConference conference = invitation.getConference();
        boolean is_own_user = invitation.getUserId().equals(VoxeetPreferences.id());
        ConferenceType type = ConferenceType.fromId(conference.getConferenceType());

        if (isLive() && null != getConferenceId() && getConferenceId().equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for the same conference, invalidate the call");
            return;
        }

        if (!is_own_user || ConferenceType.SCHEDULED.equals(type)) {
            InvitationReceivedEvent.UserInviter inviter = invitation.getInviter();
            List<DefaultInvitation> invitations = invitation.getInvitations();

            if (null == information) {
                Log.d(TAG, "onEvent: INVALID INFORMATION FOR THE CONFERENCE");
            }

            if (null != inviter && null != inviter.externalId && null != inviter.userId) {
                Log.d(TAG, "onEvent: Invitation with inviter " + inviter.userId + " " + inviter.externalId);
                UserInfo info = createUserInfo(inviter.nickName, inviter.externalId, inviter.externalAvatarUrl);

                DefaultConferenceUser inviter_user = new DefaultConferenceUser(inviter.userId,
                        "", info);
                if (null != information) {
                    information.getLastInvitationReceived().add(inviter_user);
                    information.getUserIdsCached().put(inviter.userId, inviter.externalId);
                }
            } else {
                Log.d(TAG, "onEvent: Invitation with invalid inviter");
            }

            final String finalConferenceId = conferenceId;
            getConferenceStatus(conference.getConferenceId())
                    .then(new PromiseExec<GetConferenceStatusEvent, Object>() {
                        @Override
                        public void onCall(@Nullable GetConferenceStatusEvent result, @NonNull Solver<Object> internal_solver) {
                            Log.d(TAG, "onSuccess: " + result);

                            //removed the check for live Event since now it is able to have multiple

                            if (result.getConferenceUsers().size() > 0) {
                                Log.d(TAG, "onEvent: users " + result.getConferenceUsers().size());

                                DefaultConferenceUser foundUser = null;
                                String foundExternalId = null;

                                List<DefaultConferenceUser> merged_list = new ArrayList<>();
                                //add every user from the mConference
                                merged_list.addAll(result.getConferenceUsers());
                                //add every use from the invitation
                                merged_list.addAll(getLastInvitationUsers());

                                Iterable<DefaultConferenceUser> list = Iterables.filter(merged_list, new Predicate<DefaultConferenceUser>() {
                                    @Override
                                    public boolean apply(@Nullable DefaultConferenceUser input) {
                                        return !currentUserOrEmpty().equals(input.getUserId());
                                    }
                                });

                                for (DefaultConferenceUser conferenceUser : list) {
                                    String userId = conferenceUser.getUserId();
                                    if (null == foundExternalId && null != information) {
                                        String externalId = conferenceUser.getUserInfo().getExternalId();
                                        String cachedExternalId = information.getUserIdsCached().get(userId);
                                        Log.d(TAG, "onEvent: " + userId + " " + externalId + " " + cachedExternalId);

                                        foundUser = conferenceUser;
                                        if (null != userId && (null == externalId || userId.equals(externalId))) {
                                            //should be different than null
                                            externalId = cachedExternalId;
                                            if (null != externalId) {
                                                foundExternalId = cachedExternalId;
                                            }
                                        }
                                    }
                                }

                                if (foundUser != null && null == foundExternalId) {
                                    foundExternalId = foundUser.getUserId();
                                    Log.d(TAG, "externalId is null, setting it to userId");
                                }

                                Map<String, String> infos = new HashMap<>();

                                if (null != foundUser) {

                                }
                                infos.put(VoxeetIntentFactory.INVITER_ID, null != foundUser ? foundUser.getUserId() : "");
                                infos.put(VoxeetIntentFactory.INVITER_NAME, null != foundUser ? foundUser.getUserInfo().getName() : "");
                                infos.put(VoxeetIntentFactory.NOTIF_TYPE, null != result.getType() ? result.getType() : "");
                                infos.put(VoxeetIntentFactory.INVITER_EXTERNAL_ID, null != foundUser ? foundExternalId : "");
                                infos.put(VoxeetIntentFactory.INVITER_URL, null != foundUser ? foundUser.getUserInfo().getAvatarUrl() : "");
                                infos.put(VoxeetIntentFactory.CONF_ID, null != result.getConferenceId() ? result.getConferenceId() : "");


                                getTwig().i("Starting default incoming view " + finalConferenceId + " " + result.getConferenceId());

                                Intent intent = VoxeetIntentFactory.buildFrom(mContext, VoxeetPreferences.getDefaultActivity(), infos);
                                if (intent != null)
                                    mContext.startActivity(intent);

                                mEventBus.post(new IncomingCallEvent());
                            }
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            error.printStackTrace();
                        }
                    });
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(OwnConferenceStartedEvent ownConferenceStartedEvent) {
        if (null != ownConferenceStartedEvent.event().getConferenceInfos()) {
            mConferenceId = ownConferenceStartedEvent.event().getConferenceInfos().getConferenceId();
        }

        DefaultConference conference = getConference();
        getTwig().i("Own mConference started :: " + mConferenceId + " " + conference);

        if (null != conference) {
            conference.setConferenceInfos(ownConferenceStartedEvent.event().getConferenceInfos());
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final StartVideoAnswerEvent event) {
        Log.d(TAG, "onEvent: start video event " + event.isSuccess() + " " + event.message());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(RenegociationUpdate event) {
        RenegociationEndedEvent renego = event.getEvent();

        Log.d(TAG, "onEvent: Renegociation " + renego.getConferenceId()
                + " " + renego.getType() + " " + renego.getAnswerReceived()
                + " " + renego.getOfferSent());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final OfferCreatedEvent event) {
        Log.d(TAG, "onEvent: OfferCreatedEvent " + event.message());
        try {
            handleAnswer(event.offer())
                    .then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onCall: answer called, result is " + aBoolean + " " + event.offer().getUserId() + " " + VoxeetPreferences.id());
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            error.printStackTrace();
                        }
                    });
        } catch (MediaEngineException e) {
            e.printStackTrace();
            getTwig().e(e);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserAddedEvent event) {
        if (!mConferenceId.equals(event.getEvent().getConferenceId())) {
            Log.d(TAG, "onEvent: USER ADDED FOR ANOTHER CONFERENCE");
        }

        DefaultConference conference = getOrSetConference(event.getEvent().getConferenceId());
        ConferenceUserAdded content = event.getEvent();

        //if (null == conference) {
        //    Log.d(TAG, "onEvent: ERROR THE CONFERENCE IS NULL, PLEASE CHECK WORKFLOW");
        //    createOrSetConferenceWithParams(event.getEvent().getConferenceId(),
        //            event.getEvent().getConferenceAlias());
        //    conference = getConference();
        //}

        if (null != content) {
            Log.d(TAG, "onEvent: ParticipantAdded " + content + " " + content.getUserId() + " " + conference.getConferenceUsers().size());

            DefaultConferenceUser user = findUserById(content.getUserId());

            if (null == user) {
                user = new DefaultConferenceUser(content.getUserId(), null);
                user.setIsOwner(false);
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));

                conference.getConferenceUsers().add(user);
            }

            if (null == user.getUserInfo()) {
                Log.d(TAG, "onEvent: existing user info are invalid, refreshing");
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));
            }

            user.updateIfNeeded(content.getName(), content.getAvatarUrl());

            ConferenceUserStatus status = ConferenceUserStatus.fromString(content.getStatus());
            if (null != status) {
                user.setConferenceStatus(status);
                user.setStatus(content.getStatus());
            }

            mTwig.i("from Event ConferenceUserAddedEvent, user joined with id := " + user.getUserId() + " " + (user.getUserInfo() != null ? user.getUserInfo().getExternalId() : "no external id"));

            MediaStream stream = null;
            if (hasMedia()) {
                HashMap<String, MediaStream> streams = getMapOfStreams();
                if (streams.containsKey(user.getUserId())) {
                    stream = streams.get(user.getUserId());
                }
            }

            //update the current conference state specificly from the users point of view
            updateConferenceFromUsers();
            mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
        }

        //in a previous version in the v1.0, the createOffer was forced right here
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantUpdatedEvent event) {
        DefaultConferenceUser user = findUserById(event.getUserId());
        final ConferenceUserStatus status = ConferenceUserStatus.valueOf(event.getStatus());

        if (user != null) {
            startTransactionConferenceUser();
            user.setStatus(event.getStatus());
            commitTransactionConferenceUser();
        } else {
            getTwig().i("Not in mConference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
        }

        //update the current conference state specificly from the users point of view
        updateConferenceFromUsers();

        switch (status) {
            case CONNECTING:
                if (null != user && !user.getUserId().equals(VoxeetPreferences.id())) {
                    mTwig.i("Cancelling timeout timer from user connecting");
                    removeTimeoutCallbacks();
                }
                break;
            case LEFT:
            case DECLINE:
                if (hasMedia() && user != null) {
                    getTwig().i("In mConference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
                    getMedia().removePeer(event.getUserId());
                }
                break;
            default:
                getTwig().i("status not managed updated to " + event.getStatus());

        }

        //check if the timeout runnable was not triggered
        boolean timeout_managed = timeoutRunnable != null && timeoutRunnable.isTriggered();

        if (!timeout_managed) {
            switch (status) {
                case DECLINE:
                    if (!event.getUserId().equals(VoxeetPreferences.id())) {
                        getTwig().i("Conference user with id: " + event.getUserId() + " declined the call");

                        mEventBus.post(new ConferenceUserCallDeclinedEvent(event.getConfId(), event.getUserId(), event.getStatus()));
                    }
                    break;
                case LEFT:
                    onUserLeft(event.getUserId());
                    break;
                default:
                    MediaStream stream = null;
                    if (hasMedia() && null != user) {
                        HashMap<String, MediaStream> streams = getMapOfStreams();
                        if (streams.containsKey(user.getUserId())) {
                            stream = streams.get(user.getUserId());
                        }
                    }

                    if (null != user) {
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    }
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantAddedEvent event) {
        DefaultConferenceUser user = event.getUser();

        if (user != null) {
            mTwig.i("Conference user joined with id: " + user.getUserId());

            MediaStream stream = null;
            if (hasMedia()) {
                HashMap<String, MediaStream> streams = getMapOfStreams();
                if (streams.containsKey(user.getUserId())) {
                    stream = streams.get(user.getUserId());
                }
            }

            updateConferenceFromUsers();
            mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceDestroyedPushEvent event) {
        if (null == mConferenceId || mConferenceId.equals(event.getPush().getConferenceId())) {
            closeMedia();

            getTwig().i("Conference has ended");
            mEventBus.post(event.getPush());
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceEndedEvent event) {
        if (null == mConferenceId || mConferenceId.equals(event.getEvent().getConferenceId())) {
            closeMedia();

            getTwig().i("Conference has ended");

            mEventBus.post(event.getEvent());
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final QualityUpdatedEvent event) {
        for (DefaultConferenceUser conferenceUser : event.getEvent().getUser()) {
            DefaultConferenceUser user = findUserById(conferenceUser.getUserId());
            if (user != null) {
                mTwig.i("Quality updated for " + user.getUserId());

                startTransactionConferenceUser();
                user.setQuality(conferenceUser.getQuality());
                commitTransactionConferenceUser();

                mEventBus.post(new ConferenceUserQualityUpdatedEvent(user));
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final RecordingStatusUpdate event) {
        String conferenceId = event.getEvent().getConferenceId();
        Log.d(TAG, "onEvent: eevnt for " + conferenceId);

        DefaultConference conference = getConferenceInformation(conferenceId).getConference();
        RecordingStatusUpdateEvent recordingStatusUpdateEvent = event.getEvent();
        DefaultConferenceUser user = findUserById(recordingStatusUpdateEvent.getUserId());
        if (user != null) {
            RecordingStatus status = RecordingStatus.valueOf(recordingStatusUpdateEvent.getRecordingStatus());
            if (status == RecordingStatus.RECORDING) {
                isRecording = true;
                conference.setStartRecordTimestamp(new Date(recordingStatusUpdateEvent.getTimeStamp()));
                conference.setRecordingStatus(RecordingStatus.RECORDING);
                conference.setRecordingUser(recordingStatusUpdateEvent.getUserId());
                user.setIsRecordingOwner(true);
            } else {
                isRecording = false;
                conference.setStartRecordTimestamp(null);
                conference.setRecordingStatus(RecordingStatus.NOT_RECORDING);
                conference.setRecordingUser(null);
                user.setIsRecordingOwner(false);
            }
        }

        getTwig().i("Conference's recording status has changed to: " + event.getEvent().getRecordingStatus());

        mEventBus.post(recordingStatusUpdateEvent);
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Protected management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    //new DefaultConference()
    //new DefaultConferenceRealm()
    protected abstract DefaultConference createConference();

    //mConference.getConferenceUsers().add(new DefaultConferenceUser(offer.getUserId(), offer.getDevice(), new UserInfo("Sophie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png")));
    protected abstract DefaultConferenceUser createConferenceUser(String userId, String device, UserInfo userInfo);

    protected abstract UserInfo createUserInfo(String userName, String externalId, String avatarUrl);

    protected Twig getTwig() {
        return mTwig;
    }

    protected void setConference(@NonNull DefaultConference conference) {
        mConferenceId = conference.getConferenceId();
        ConferenceInformation information = getCurrentConferenceInformation();
        information.setConference(conference);
    }

    protected void setConferenceAlias(String alias) {
        DefaultConference conference = getConference();
        conference.setConferenceAlias(alias);
        //null ? throw error B)
    }

    public EventBus getEventBus() {
        return mEventBus;
    }

    protected ConferenceListener getConferenceListener() {
        return mListener;
    }

    protected MediaSDK getMedia() {
        return getMediaService().getMedia();
    }

    protected abstract void startTransactionConferenceUser();

    protected abstract void commitTransactionConferenceUser();

    /**
     * Return true if the current implementation is SDK
     * false otherwise
     *
     * @return the state of this implementation
     */
    protected abstract boolean isSDK();

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected Promise<Integer> answer(final String peer, final SdpMessage message) {
        return new Promise<>(new PromiseSolver<Integer>() {
            @Override
            public void onCall(@NonNull final Solver<Integer> solver) {
                try {
                    if (null == mConferenceId) {
                        throw new NotInConferenceException();
                    }
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                    return;
                }

                Log.d("SDKMEDIA", "answer: peer := " + peer + " message := " + message);

                final Call<ResponseBody> user = mConferenceObservableProvider.answerConference(mConferenceId, peer, message.getDescription());
                HttpHelper.enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        if (response.code() != 200) {
                            getTwig().i("Offer created for " + peer + " : KO");

                            try {
                                ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(response.code() + "");
                                throw new PromiseParticipantAddedErrorEventException(event);
                            } catch (PromiseParticipantAddedErrorEventException exception) {
                                solver.reject(exception);
                            }
                        } else {
                            getTwig().i("Offer created for " + peer + " : OK");
                            solver.resolve(response.code());
                        }
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        getTwig().e("Offer created for " + peer + " : KO");

                        HttpException.dumpErrorResponse(response);
                        try {
                            ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(handleError(e));
                            throw new PromiseParticipantAddedErrorEventException(event);
                        } catch (PromiseParticipantAddedErrorEventException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }


    /**
     * resolve -> true everything went fine
     * resolve -> false an error occured
     * reject -> media exception
     *
     * @param offer
     * @return a promise with the result of the call
     * @throws MediaEngineException
     */
    protected Promise<Boolean> handleAnswer(final OfferCreated offer) throws MediaEngineException {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                if (!hasMedia()) {
                    try {
                        throw new MediaEngineException("handleAnswer media is null");
                    } catch (MediaEngineException exception) {
                        solver.reject(exception);
                    }
                }


                final String conferenceId = offer.getConferenceId();

                if (!conferenceId.equals(mConferenceId)) {
                    Log.d(TAG, "onCall: CONFERENCE IS NOT THE SAME ! ANSWER SHALL BE DISCARDED");
                }

                SdpDescription description = new SdpDescription(offer.getDescription().getType(), offer.getDescription().getSdp());

                List<SdpCandidate> candidates = new ArrayList<>();
                for (OfferCandidate candidate : offer.getCandidates()) {
                    Log.d(TAG, "onCall: sdpcandidate " + candidate.getMid() + " " + candidate.getmLine());
                    candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                }

                String userId = offer.getUserId();//isSDK() ? offer.getExternalId() : offer.getUserId();
                Log.d(TAG, "handleAnswer: offer := " + offer.getUserId() + " " + offer.getExternalId());
                Log.d("SDKMEDIA", "handleAnswer: " + offer.getUserId() + " " + offer.getExternalId() + " " + description.getSsrc());


                try {
                    //TODO in the callback, check for the conferenceId being the same !
                    getMedia().createAnswerForPeer(userId,
                            description.getSsrc(),
                            description,
                            candidates,
                            offer.isMaster(),
                            new PendingPeerCallback() {
                                @Override
                                public void onMessage(@Nullable SdpMessage message) {
                                    try {
                                        if (null == mConferenceId) {
                                            Log.d(TAG, "onMessage: INVALID CONFERENCE ID WHEN OFFER IS RECEIVED");
                                            mConferenceId = offer.getConferenceId();
                                        }
                                        DefaultConference conference = getConference();

                                        DefaultConferenceUser user = null;

                                        for (DefaultConferenceUser in_conf : conference.getConferenceUsers()) {
                                            if (in_conf.getUserId() != null && in_conf.getUserId().equals(offer.getUserId())) {
                                                user = in_conf;
                                            }
                                        }

                                        UserInfo infos;
                                        if (offer.getUserId().contains("11111"))
                                            infos = createUserInfo("Julie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_1.png");
                                        else if (offer.getUserId().contains("22222"))
                                            infos = createUserInfo("Sophie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png");
                                        else if ((offer.getUserId().contains("33333")))
                                            infos = createUserInfo("Mike", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_3.png");
                                        else
                                            infos = createUserInfo(offer.getName(), offer.getExternalId(), offer.getAvatarUrl());

                                        if (user == null) {
                                            user = createConferenceUser(offer.getUserId(), offer.getDevice(), infos);
                                            conference.getConferenceUsers().add(user);
                                        } else {
                                            user.setUserInfo(infos);
                                        }


                                        setUserPosition(offer.getUserId(), 0, 0.5);

                                        answer(offer.getUserId(), message)
                                                .then(new PromiseExec<Integer, Object>() {
                                                    @Override
                                                    public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                        Log.d(TAG, "onSuccess: " + result);

                                                        solver.resolve(true);
                                                    }
                                                })
                                                .error(new ErrorPromise() {
                                                    @Override
                                                    public void onError(Throwable error) {
                                                        if (error instanceof PromiseParticipantAddedErrorEventException)
                                                            mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                        else
                                                            error.printStackTrace();

                                                        solver.resolve(false);
                                                    }
                                                });
                                    } catch (Exception e) {
                                        Log.d(TAG, "onMessage: unlockPeerOperation" + e.getMessage());
                                        solver.resolve(false);
                                    }
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }
    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected DefaultConferenceUser updateConferenceParticipants(String userId, ConferenceUserStatus status) {
        DefaultConferenceUser user = findUserById(userId);
        DefaultConference conference = getConference();

        if (null != user && null != conference) {
            startTransactionConferenceUser();
            user.setIsOwner(conference.getOwnerProfile() == null || conference.getOwnerProfile().getUserId().equals(user.getProfile().getUserId()))
                    .setConferenceStatus(status)
                    .setQuality(ConferenceQuality.ULTRA_HIGH.StringValue());
            commitTransactionConferenceUser();

            return user;
        }
        return null;
    }

    /**
     * @return false if permission refused -> cancel ! true otherwise
     */
    protected boolean initMedia(boolean listenerMode) {
        Validate.notNull(mContext, "mContext");
        Validate.notNull(VoxeetPreferences.id(), "user id");

        if (!hasMedia()) {
            try {
                if (!listenerMode && !Validate.hasMicrophonePermissions(context)) {
                    getTwig().i("the app does not seem to have mic permission, disabling mic");
                    mute(true);
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                }

                getAudioService().enable();
                getMediaService().createMedia(context, VoxeetPreferences.id(), mediaStreamListener, cameraEventsHandler,
                        isVideoOn(), !listenerMode && Validate.hasMicrophonePermissions(context));
                getAudioService().setSpeakerMode(isDefaultOnSpeaker);
                getAudioService().requestAudioFocus();

                if (isDefaultMute) {
                    getMedia().mute();
                } else {
                    getMedia().unMute();
                }
            } catch (MediaEngineException e) {
                //TODO check for excception as a "false"
                getTwig().e(e);
            }
        }

        return true;
    }

    protected void closeMedia() {
        if (hasMedia()) {
            MediaSDK media = getMedia();
            Conference conference = getConference();
            ConferenceInformation information = getCurrentConferenceInformation();

            if (null != information) {
                information.setOwnVideoStarted(false);
                information.setScreenShareOn(false);
            }

            try {
                if (null != conference) {
                    for (DefaultConferenceUser user : conference.getConferenceUsers())
                        if (user != null && user.getUserId() != null)
                            media.removePeer(user.getUserId());
                }
            } catch (Exception ex) {
                mTwig.e(ex);
            }

            getAudioService().abandonAudioFocusRequest();
            getAudioService().disable();

            getMediaService().releaseMedia();
            getTwig().i("Microphone value restored to true");

            mTwig.i("Video value restored to false");

            getTwig().i("Listener mode set back to false");
        }

        getTwig().i("Media closed");
        setIsInConference(false);
        mConferenceId = null;
    }

    /**
     * Remove any previous timeout callback
     * <p>
     * To be used when it is mandatory to remove such behaviour
     * for instance, when joining a new mConference
     * <p>
     * It happens since we do not want to manage every edge case :
     * - decline
     * - network error
     * - etc...
     */
    protected void removeTimeoutCallbacks() {
        if (timeoutRunnable != null) {
            timeoutRunnable.setCanceled(true);
            handler.removeCallbacks(timeoutRunnable);
        }
    }

    protected void sendTimeoutCallbacks() {
        if (mTimeOutTimer != -1) {

            timeoutRunnable = new TimeoutRunnable(this,
                    getTwig(),
                    mEventBus,
                    mTimeOutTimer);

            getTwig().i("scheduling timer to leave the mConference in " + mTimeOutTimer);

            handler.postDelayed(timeoutRunnable, mTimeOutTimer);
        }
    }

    private void onUserLeft(@NonNull String peer) {
        DefaultConferenceUser user = findUserById(peer);
        if (user != null) {
            startTransactionConferenceUser();
            user.setConferenceStatus(ConferenceUserStatus.LEFT);

            if (getConferenceUsers().contains(user)) getConferenceUsers().remove(user);

            getTwig().i("Conference user left with id: " + user.getUserId());
            commitTransactionConferenceUser();

            if (mapOfStreams.containsKey(user.getUserId())) {
                mapOfStreams.remove(user.getUserId());
            }

            if (mapOfScreenShareStreams.containsKey(user.getUserId())) {
                mapOfScreenShareStreams.remove(user.getUserId());
            }

            //refresh the conference state users
            updateConferenceFromUsers();
            mEventBus.post(new ConferenceUserLeftEvent(user));
        } else {
            Log.d(TAG, "run: unknown user in stream left");
        }
    }

    protected VoxeetSdkTemplate getVoxeetSDK() {
        return mSDK;
    }

    private AudioService getAudioService() {
        return getVoxeetSDK().getAudioService();
    }

    @NonNull
    protected ConferenceInformation createOrSetConferenceWithParams(@NonNull String conferenceId, @Nullable String conferenceAlias) {
        Log.d(TAG, "createOrSetConferenceWithParams: set conference id := " + conferenceId);
        ConferenceInformation information = getConferenceInformation(conferenceId);

        DefaultConference conference = information.getConference();

        conference.setConferenceId(conferenceId);
        if (null != conferenceAlias || null == conference.getConferenceAlias()) {
            conference.setConferenceAlias(conferenceAlias);
        }

        //the information is always valid
        return information;
    }

    @Nullable
    public ConferenceInformation getCurrentConferenceInformation() {
        return getConferenceInformation(mConferenceId);
    }

    @Nullable
    protected ConferenceInformation getConferenceInformation(@Nullable String conferenceId) {
        if (null == conferenceId) {
            return null;
        }
        return mConferenceInformationHolder.getInformation(conferenceId);
    }

    private void updateConferenceFromUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            ConferenceState state = information.getState();
            switch (state) {
                case JOINED:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceState.FIRST_PARTICIPANT);
                    break;
                case FIRST_PARTICIPANT:
                    if (!hasParticipants())
                        information.setConferenceState(ConferenceState.NO_MORE_PARTICIPANT);
                    break;
                case NO_MORE_PARTICIPANT:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceState.FIRST_PARTICIPANT);
                    break;
                default:
                    //nothing
            }
            state = information.getState();
        }
    }

    public boolean hasParticipants() {
        List<DefaultConferenceUser> users = getConferenceUsers();

        for (DefaultConferenceUser user : users) {
            ConferenceUserStatus status = user.getConferenceStatus();
            if (null != user.getUserId() && !user.getUserId().equals(VoxeetPreferences.id())
                    && ConferenceUserStatus.ON_AIR.equals(status)) {
                return true;
            }
        }
        return false;
    }

    public void setICERestartEnabled(boolean new_state) {
        isICERestartEnabled = new_state;
    }

    public boolean isICERestartEnabled() {
        return isICERestartEnabled;
    }

    @NonNull
    public Context getContext() {
        return mContext;
    }

    @Nullable
    public String getDefaultCamera() {
        return mDefaultCamera;
    }

    private boolean hasMedia() {
        return null != getMedia();
    }

    @NonNull
    private MediaService getMediaService() {
        return mInstance.getMediaService();
    }

    protected void joinLock() {
        try {
            joinLock.lock();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    protected void joinUnlock() {
        try {
            if (joinLock.isLocked())
                joinLock.unlock();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
