package voxeet.com.sdk.core.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.audio.AudioRoute;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.voiceengine.WebRtcAudioEffects;
import org.webrtc.voiceengine.WebRtcAudioUtils;

import java.util.List;

import voxeet.com.sdk.audio.SoundManager;
import voxeet.com.sdk.core.AbstractVoxeetService;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;
import voxeet.com.sdk.events.success.ConferencePreJoinedEvent;
import voxeet.com.sdk.json.ConferenceDestroyedPush;
import voxeet.com.sdk.utils.AudioType;

public class AudioService extends AbstractVoxeetService {

    private static SoundManager sSoundManager;
    private static final String TAG = AudioService.class.getSimpleName();

    /**
     * Instantiate a new Audio Service
     *
     * @param instance the voxeet parent instance
     */
    public AudioService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        AudioService.preInitSounds(instance.getApplicationContext());
        instance.getEventBus().register(this);
    }

    @Nullable
    public static SoundManager getSoundManager() {
        return sSoundManager;
    }

    public static boolean preInitSounds(@NonNull Context applicationContext) {
        if (null == sSoundManager) {
            sSoundManager = new SoundManager(applicationContext);
            return true;
        }

        return false;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferencePreJoinedEvent event) {
        //logging the webrtc Audio Utils information
        logWebRtcInfos();

        enableAec(true);
        enableNoiseSuppressor(true);

        sSoundManager.onConferencePreJoinedEvent();
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPush event) {
        sSoundManager.onConferenceDestroyedPush();
    }


    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return sSoundManager.getAvailableRoutes();
    }

    public AudioRoute currentRoute() {
        return sSoundManager.currentRoute();
    }

    public boolean setAudioRoute(@NonNull AudioRoute route) {
        return sSoundManager.setAudioRoute(route);
    }

    public void setSpeakerMode(boolean isSpeaker) {
        sSoundManager.setSpeakerMode(isSpeaker);
    }

    public boolean isSpeakerOn() {
        return sSoundManager.isSpeakerOn();
    }

    public void acquireLocks() {
        sSoundManager.acquireLocks();
    }

    public void releaseLocks() {
        sSoundManager.releaseLocks();
    }

    public void setInVoiceCallSoundType() {
        sSoundManager.setInVoiceCallSoundType();
    }

    public void resetDefaultSoundType() {
        sSoundManager.resetDefaultSoundType();
    }

    public void abandonAudioFocusRequest() {
        sSoundManager.abandonAudioFocusRequest();
    }

    public void requestAudioFocus() {
        sSoundManager.requestAudioFocus();
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return sSoundManager.setSound(type, assetName);
    }

    public void playSoundType(@NonNull AudioType type) {
        sSoundManager.playSoundType(type);
    }

    public void stopSoundType(@NonNull AudioType audioType) {
        sSoundManager.stopSoundType(audioType);
    }

    public void stop() {
        sSoundManager.stop();
    }

    public void enable() {
        sSoundManager.enable();
    }

    public void disable() {
        sSoundManager.disable();
        stop();
    }

    public void logWebRtcInfos() {
        try {
            Log.d(TAG, "logWebRtcInfos: AEC (hw) := " + WebRtcAudioEffects.canUseAcousticEchoCanceler());
        } catch (Exception e) {
            Log.d(TAG, "logWebRtcInfos: AEC (hw) := call crashed");
        }
        try {
            Log.d(TAG, "logWebRtcInfos: NS (hw) := " + WebRtcAudioEffects.canUseNoiseSuppressor());
        } catch (Exception e) {
            Log.d(TAG, "logWebRtcInfos: NS (hw) := call crashed");
        }
        Log.d(TAG, "logWebRtcInfos: AEC (WebRTC) := " + WebRtcAudioUtils.useWebRtcBasedAcousticEchoCanceler());
        Log.d(TAG, "logWebRtcInfos: NS (WebRTC) := " + WebRtcAudioUtils.useWebRtcBasedNoiseSuppressor());
        try {
            Log.d(TAG, "logWebRtcInfos: isAcousticEchoCancelerBlacklisted := " + WebRtcAudioEffects.isAcousticEchoCancelerBlacklisted());
        } catch (Exception e) {
            Log.d(TAG, "logWebRtcInfos: isAcousticEchoCancelerBlacklisted := call crashed");
        }
        try {
            Log.d(TAG, "logWebRtcInfos: isAcousticEchoCancelerSupported := " + WebRtcAudioEffects.isAcousticEchoCancelerSupported());
        } catch (Exception e) {
            Log.d(TAG, "logWebRtcInfos: isAcousticEchoCancelerSupported := call crashed");
        }
        Log.d(TAG, "logWebRtcInfos: isNoiseSuppressorBlacklisted := " + WebRtcAudioEffects.isNoiseSuppressorBlacklisted());
        Log.d(TAG, "logWebRtcInfos: isNoiseSuppressorSupported := " + WebRtcAudioEffects.isNoiseSuppressorSupported());
    }

    public void enableAec(boolean enable) {
        boolean blacklisted = WebRtcAudioEffects.isAcousticEchoCancelerBlacklisted();
        boolean checkExcludedUUID = !WebRtcAudioEffects.canUseAcousticEchoCanceler();

        boolean rejected = false;
        try {
            rejected = !WebRtcAudioEffects.isAcousticEchoCancelerSupported() || blacklisted || checkExcludedUUID;
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (rejected) {
            WebRtcAudioUtils.setWebRtcBasedAcousticEchoCanceler(enable);
        } else {
            Log.d(TAG, "enableAec: do not enableAec, already managed by the Hardware");
        }
    }

    public void enableNoiseSuppressor(boolean enable) {
        //boolean blacklisted = WebRtcAudioEffects.isNoiseSuppressorBlacklisted();
        boolean checkExcludedUUID = !WebRtcAudioEffects.canUseNoiseSuppressor();
        boolean rejected = false;
        try {
            rejected = !WebRtcAudioEffects.isNoiseSuppressorSupported() || checkExcludedUUID;
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (rejected) {
            WebRtcAudioUtils.setWebRtcBasedNoiseSuppressor(enable);
        } else {
            Log.d(TAG, "enableNoiseSuppressor: do not enableNoiseSuppressor, already managed by the Hardware");
        }
    }
}
