package voxeet.com.sdk.core;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.fasterxml.jackson.databind.ObjectMapper;

import org.greenrobot.eventbus.EventBus;
import org.webrtc.EglBase;

import java.io.IOException;
import java.util.Date;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;

/**
 * Created by RomainB on 5/13/16.
 *
 * @param <T> is the class which will be instantiated
 */
public abstract class AbstractVoxeetService<T> {

    private static final String TAG = AbstractVoxeetService.class.getSimpleName();

    @NonNull
    private ServiceProviderHolder<T> holder = ServiceProviderHolder.DEFAULT;


    /**
     * last time we asked the API for data
     */
    private Date lastUpdate;

    private boolean isLoaded = false;

    private int UPDATE_TIME = 5 * 60 * 1000;

    /**
     * The Handler.
     */
    protected final static Handler handler = new Handler(Looper.getMainLooper());

    /**
     * The Executor service.
     */
    protected ExecutorService executorService;

    /**
     * The Context.
     */
    protected Context context;

    private AbstractVoxeetService() {
        this.lastUpdate = null;

        this.executorService = Executors.newSingleThreadExecutor();
    }

    /**
     * Instantiates a new Voxeet service.
     *
     * @param instance      the voxeet parent instance
     * @param service_klass
     */
    protected AbstractVoxeetService(@NonNull VoxeetSdkTemplate instance, @Nullable Class<T> service_klass) {
        this(instance, new ServiceProviderHolder.Builder<T>()
                .setClient(instance.getClient())
                .setRetrofit(instance.getRetrofit())
                .setService(service_klass)
                .build());
    }

    /**
     * Instantiates a new Voxeet service.
     *
     * @param instance the instance of the SDK
     */
    protected AbstractVoxeetService(@NonNull VoxeetSdkTemplate instance, @NonNull ServiceProviderHolder<T> holder) {
        this();

        this.context = instance.getApplicationContext();

        this.holder = holder;
    }

    /**
     * Update time stamp.
     */
    protected void updateTimeStamp() {
        this.lastUpdate = new Date();
    }

    /**
     * Retuns if the API has been reached at least once successfully
     *
     * @return true if it's been loaded, false otherwise
     */
    protected boolean isLoaded() {
        return isLoaded;
    }

    /**
     * Sets loaded.
     *
     * @param loaded the loaded
     */
    protected void setLoaded(boolean loaded) {
        isLoaded = loaded;
    }

    /**
     * Should update boolean.
     *
     * @return the boolean
     */
    protected boolean shouldUpdate() {
        if (lastUpdate == null)
            return true;
        else
            return (new Date().getTime() - lastUpdate.getTime() > UPDATE_TIME);
    }

    final protected T getService() {
        return holder.getService();
    }

    /**
     * Retrieves the error string from the request.
     *
     * @param e the exception
     * @return the error content
     */
    protected String handleError(@Nullable Throwable e) {
        if(null == e) {
            return "invalid error";
        }
        Log.d(TAG, "handleError: " + e.getClass().getSimpleName());
        /*if (e instanceof HttpException) {
            try {
                ResponseBody body = ((HttpException) e).response().errorBody();
                return body.string();
            } catch (IOException exception) {
                Log.e(TAG, "error: " + exception.getMessage());
            }
        }*/
        return e.getMessage();
    }

    /**
     * Gets error body.
     *
     * @param message the message
     * @return the error body
     */
    protected static VoxeetJsonError getErrorBody(String message) {
        ObjectMapper om = new ObjectMapper();
        VoxeetJsonError command = null;
        try {
            command = om.readValue(message, VoxeetJsonError.class);
        } catch (IOException e) {
            Log.e(TAG, "failed to get error body", e);
        }

        return command;
    }

    /**
     * Reset service.
     */
    public void resetService() {
        lastUpdate = null;

        isLoaded = false;
    }


    /**
     * Get the current endpoint root url
     *
     * @param template
     * @return
     */
    @Nullable
    protected final String getURLRoot(@NonNull VoxeetSdkTemplate template) {
        return template.getVoxeetHttp().getBuiltServerUrl();
    }

    /**
     * @param template the token provider
     * @return the current token of the session
     */
    @Nullable
    protected final String getInternalToken(@NonNull VoxeetSdkTemplate template) {
        return template.getVoxeetHttp().getToken();
    }

    /**
     * @param template the token provider
     * @return the current token of the session
     */
    @Nullable
    protected final String getInternalJwtToken(@NonNull VoxeetSdkTemplate template) {
        return template.getVoxeetHttp().getJwtToken();
    }

    protected EventBus getEventBus() {
        return holder.getEventBus();
    }

    protected void registerEventBus() {
        EventBus eventBus = holder.getEventBus();
        if (!eventBus.isRegistered(this))
            eventBus.register(this);
    }

    protected void unRegisterEventBus() {
        EventBus eventBus = holder.getEventBus();
        if (eventBus.isRegistered(this))
            eventBus.unregister(this);
    }

    protected void postOnMainThread(@NonNull Runnable runnable) {
        handler.post(runnable);
    }

}
