package com.voxeet.android.media;

import android.content.Context;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.net.wifi.WifiManager;
import android.net.wifi.WifiManager.WifiLock;
import android.os.Build;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.audio.AudioRoute;

/**
 * Class that manage the difference WakeLock required for the conference.
 *
 * @author Thomas Gourgues
 */
public class MediaPowerManager implements SensorEventListener {
    private final static String TAG = MediaPowerManager.class.getSimpleName();

    private final static int PROXIMITY_SCREEN_OFF_WAKE_LOCK = 32;

    private SensorManager mSensorManager;

    @Nullable
    private WakeLock mProximityWakeLock;

    private WakeLock mPartialWakeLock;

    private WifiLock mWifiLock;

    @Nullable
    private Sensor mProximity;

    private boolean shouldRegisterForProximity = false;
    private AudioRoute mAudioRoute = AudioRoute.ROUTE_SPEAKER;

    public MediaPowerManager(@NonNull Context context,
                             @NonNull AudioRoute defaultAudioRoute) {

        setAudioRoute(defaultAudioRoute);

        PowerManager powerManager = (PowerManager) context.getSystemService(Context.POWER_SERVICE);

        WifiManager wifiManager = (WifiManager) context.getApplicationContext().getSystemService(Context.WIFI_SERVICE);

        mSensorManager = (SensorManager) context.getSystemService(Context.SENSOR_SERVICE);

        if (null != mSensorManager) {
            mProximity = mSensorManager.getDefaultSensor(Sensor.TYPE_PROXIMITY);
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP
                || (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP && !Build.BRAND.contains("samsung"))) {
            shouldRegisterForProximity = true;
        }

        try {
            if (null != powerManager) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP)
                    mProximityWakeLock = powerManager.newWakeLock(PowerManager.PROXIMITY_SCREEN_OFF_WAKE_LOCK, TAG);
                else
                    mProximityWakeLock = powerManager.newWakeLock(PROXIMITY_SCREEN_OFF_WAKE_LOCK, TAG);

                mPartialWakeLock = powerManager.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, TAG);
            }

            if (null != wifiManager) {
                mWifiLock = wifiManager.createWifiLock(WifiManager.WIFI_MODE_FULL_HIGH_PERF, TAG);
            }
        } catch (Exception e) {
            Log.e(TAG, "Failed to create required WakeLock");
        }
    }

    public void acquire() {
        if (shouldRegisterForProximity && null != mProximity)
            mSensorManager.registerListener(this, mProximity, SensorManager.SENSOR_DELAY_NORMAL);

        if (null != mPartialWakeLock && !mPartialWakeLock.isHeld())
            acquirePartial();

        if (null != mWifiLock && !mWifiLock.isHeld())
            mWifiLock.acquire();
    }

    public void release() {
        if (shouldRegisterForProximity)
            mSensorManager.unregisterListener(this);

        if (null != mPartialWakeLock && mPartialWakeLock.isHeld())
            mPartialWakeLock.release();

        if (null != mWifiLock && mWifiLock.isHeld())
            mWifiLock.release();

        if (null != mProximityWakeLock && mProximityWakeLock.isHeld())
            mProximityWakeLock.release();
    }

    @Override
    public final void onAccuracyChanged(Sensor sensor, int accuracy) {
        // Do something here if sensor accuracy changes.
    }

    @Override
    public final void onSensorChanged(SensorEvent event) {
        float distance = event.values[0];

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            if (AudioRoute.ROUTE_PHONE.equals(mAudioRoute) // black off screen if default audio route
                    && mProximityWakeLock != null
                    && !mProximityWakeLock.isHeld()) {
                if (distance <= 5) { // distance changes from one brand to another (Samsung: 0; Motorola: 3)
                    acquireProximity();
                }
            } else if (mProximityWakeLock != null && mProximityWakeLock.isHeld()) { // enable screen
                mProximityWakeLock.release(0);
            }
        }
    }

    public void updateProximitySensor(@NonNull AudioRoute route) {
        if (null != mProximityWakeLock) {
            if (AudioRoute.ROUTE_PHONE.equals(route)) {
                acquireProximity();
            } else {
                mProximityWakeLock.release();
            }
        }
    }

    public void setAudioRoute(@NonNull AudioRoute route) {
        mAudioRoute = route;
    }

    private void acquireProximity() {
        if (null != mProximityWakeLock) {
            mProximityWakeLock.acquire();
        }
    }

    private void acquirePartial() {
        if (null != mPartialWakeLock) {
            mPartialWakeLock.acquire();
        }
    }

}
