package voxeet.com.sdk.core.abs.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.MediaEngineException;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import retrofit2.Call;
import retrofit2.Response;
import voxeet.com.sdk.core.abs.AbstractConferenceSdkObservableProvider;
import voxeet.com.sdk.core.abs.AbstractConferenceSdkService;
import voxeet.com.sdk.core.abs.AbstractPromiseable;
import voxeet.com.sdk.core.abs.information.ConferenceInformation;
import voxeet.com.sdk.core.http.HttpCallback;
import voxeet.com.sdk.core.http.HttpHelper;
import voxeet.com.sdk.core.preferences.VoxeetPreferences;
import voxeet.com.sdk.core.services.MediaService;
import voxeet.com.sdk.events.error.HttpException;
import voxeet.com.sdk.events.error.PermissionRefusedEvent;
import voxeet.com.sdk.events.promises.PromisePermissionRefusedEventException;
import voxeet.com.sdk.events.success.StartVideoAnswerEvent;
import voxeet.com.sdk.json.StartVideoResponse;
import voxeet.com.sdk.utils.Validate;

public class StartVideoPromise extends AbstractPromiseable<Boolean> {
    private final static String TAG = StartVideoPromise.class.getSimpleName();

    public StartVideoPromise(@NonNull AbstractConferenceSdkService parent,
                             @NonNull MediaService mediaService,
                             @NonNull AbstractConferenceSdkObservableProvider provider,
                             @Nullable ConferenceInformation information,
                             @NonNull EventBus eventBus) {
        super(parent, mediaService, provider, information, eventBus);
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final ConferenceInformation information = getInformation();
                try {
                    log("starting video");
                    if (!Validate.hasCameraPermissions(getParent().getContext())) {
                        PermissionRefusedEvent event = new PermissionRefusedEvent(PermissionRefusedEvent.Permission.CAMERA);
                        getEventBus().post(event);
                        throw new PromisePermissionRefusedEventException(event);
                    }

                    if (!getMediaService().hasMedia()) {
                        Log.d(TAG, "startVideo: media is null");
                        throw new MediaEngineException("Media is null, invalid state");
                    }

                    if (null == information) {
                        Log.d(TAG, "startVideo: information is null");
                        throw new MediaEngineException("ConferenceInformation is null, invalid state");
                    }

                    if (information.isOwnVideoStarted()) {
                        Log.d(TAG, "startVideo: already started... please wait if you wanted to stop");
                        solver.resolve(true);//new StartVideoAnswerEvent(false, true));
                        return;
                    }

                    if (null == getParent().getDefaultCamera()) {
                        Log.d(TAG, "startVideo: unable to load video getDefaultCamera():=" + getParent().getDefaultCamera());
                        throw new MediaEngineException("DefaultCamera is null, invalid state");
                    }

                    information.setOwnVideoStarted(true);

                    getMediaService().getMedia().startVideo(getParent().getDefaultCamera());

                    final Call<StartVideoResponse> startVideo = getProvider().getStartVideoObservable(getInformation().getConference().getConferenceId(), VoxeetPreferences.id());
                    HttpHelper.enqueue(startVideo, new HttpCallback<StartVideoResponse>() {
                        @Override
                        public void onSuccess(@NonNull StartVideoResponse object, @NonNull Response<StartVideoResponse> response) {
                            //remove start video since it is not to be used
                            //media.startVideo(mDefaultCamera);

                            getInformation().setOwnVideoStarted(true);

                            final StartVideoAnswerEvent event = new StartVideoAnswerEvent(true);
                            getEventBus().post(event);
                            createVideoAnswer(object.getUserId(), object.getDescription(), object.getCandidates())
                                    .then(new PromiseExec<Boolean, Object>() {
                                        @Override
                                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                            solver.resolve(true);
                                        }
                                    })
                                    .error(new ErrorPromise() {
                                        @Override
                                        public void onError(Throwable error) {
                                            error.printStackTrace();
                                            solver.resolve(true);
                                        }
                                    });
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<StartVideoResponse> response) {
                            HttpException.dumpErrorResponse(response);

                            e.printStackTrace();
                            information.setOwnVideoStarted(false);
                            StartVideoAnswerEvent event = new StartVideoAnswerEvent(false);
                            getEventBus().post(event);
                            solver.reject(e);
                        }
                    });
                    //explicitly keep the list of the different exception
                    //this methods throws
                } catch (PromisePermissionRefusedEventException exception) {
                    solver.reject(exception);
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                } catch (Exception exception) {
                    solver.reject(exception);
                }
            }
        });
    }
}
