package voxeet.com.sdk.core.services.authenticate;

import android.support.annotation.NonNull;
import android.util.Log;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import okhttp3.Cookie;
import okhttp3.CookieJar;
import okhttp3.HttpUrl;
import voxeet.com.sdk.core.preferences.VoxeetPreferences;

public class VoxeetCookieJar implements CookieJar {

    @NonNull
    private final HashMap<String, List<Cookie>> cookieStore;

    @NonNull
    private final String url;

    public VoxeetCookieJar(@NonNull String url) {
        cookieStore = new HashMap<>();
        this.url = url;
    }

    /**
     * Gets cookies.
     *
     * @param url the url
     * @return the cookies
     */
    public List<Cookie> getCookies(String url) {
        for (Object o : cookieStore.entrySet()) {
            Map.Entry pair = (Map.Entry) o;
            if (((String) pair.getKey()).contains(url)) {
                return cookieStore.get(pair.getKey());
            }
        }
        return null;
    }

    /**
     * Create cookie.
     *
     * @param value the value
     * @return the string
     */
    public String createCookie(String value) {
        HttpUrl parsed = HttpUrl.parse(url);
        if (null == parsed) return null;

        Cookie cookie = Cookie.parse(parsed, value);
        if (cookie != null && cookie.toString().length() > 0) {
            cookieStore.put(url, Collections.singletonList(cookie));
            return cookie.value();
        }
        return null;
    }

    /**
     * Retrieve cookie.
     *
     * @return the string
     */
    public String retrieveCookie() {
        String cookie;
        if ((cookie = VoxeetPreferences.getLoginCookie()) != null && cookie.length() > 0) {
            cookie = createCookie(cookie);
        } else {
            List<Cookie> cookies = getCookies(url);
            if (cookies != null && cookies.size() > 0) {
                cookie = cookies.get(0).value();
                VoxeetPreferences.saveLoginCookie(cookies.get(0).toString());
            }
        }
        return cookie;
    }

    @Override
    public void saveFromResponse(@NonNull HttpUrl url,
                                 @NonNull List<Cookie> cookies) {
        String base = url.scheme() + "://" + url.host() + ":" + url.port();
        Log.d("VoxeetCookieJar", "saveFromResponse: " + base+" "+ Arrays.toString(cookies.toArray()));

        cookieStore.put(base, Lists.newArrayList(Iterables.filter(cookies, new Predicate<Cookie>() {
            @Override
            public boolean apply(Cookie input) {
                return input != null && !input.value().equalsIgnoreCase("deleted") && new Date().before(new Date(input.expiresAt()));
            }
        })));
    }

    @Override
    public List<Cookie> loadForRequest(@NonNull HttpUrl url) {
        String base = url.scheme() + "://" + url.host() + ":" + url.port();

        List<Cookie> cookies = cookieStore.get(base);
        if(null != cookies) {
            for (Cookie cookie : cookies) {
                Log.d("VoxeetCookieJar", "loadForRequest: " + cookie.value() + " " + cookie.toString());
            }
        }

        return cookies != null ? cookies : new ArrayList<Cookie>();
    }
}
