package voxeet.com.sdk.core.services.authenticate;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import java.util.concurrent.CountDownLatch;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.Request;
import voxeet.com.sdk.core.services.authenticate.token.RefreshTokenCallback;
import voxeet.com.sdk.core.services.authenticate.token.TokenCallback;
import voxeet.com.sdk.core.services.authenticate.token.TokenResponseProvider;
import voxeet.com.sdk.models.TokenResponse;
import voxeet.com.sdk.utils.AbstractVoxeetEnvironmentHolder;

public class OAuthTokenResponseProvider extends TokenResponseProvider {


    private CountDownLatch countDownLatch = null;

    /**
     * The current tokenAccess set by the SDK initialization
     * Note : if non null, tokenRefresh must be also non null
     */
    @Nullable
    private final String tokenAccess;
    /**
     * The current tokenRefresh callback set by the SDK initialization
     * Note : if non null, tokenAccess must be also non null
     * Callbable means that if any network call is done, it must be done synchronically
     * <p>
     * it is made sure by VoxeetHttp that the call will not block the UI interface
     */
    @NonNull
    private final RefreshTokenCallback tokenRefresh;

    public OAuthTokenResponseProvider(@Nullable String tokenAccess,
                                      @NonNull RefreshTokenCallback tokenRefresh,
                                      @NonNull AbstractVoxeetEnvironmentHolder holder) {
        super(holder);
        this.tokenAccess = tokenAccess;
        this.tokenRefresh = tokenRefresh;

        //create a temporary tokenResponse object which will hold the values
        tokenResponse = new TokenResponse(tokenAccess, null);
    }

    @Nullable
    @Override
    public TokenResponse refreshTokenResponse() {
        Log.d(TAG, "refreshTokenResponse: call...");
        boolean interrupted = false;
        final String[] tokenAccess = {null};
        Log.d(TAG, "refreshTokenResponse: countdown created");
        boolean creator = false;
        if(countDownLatch == null) {
            countDownLatch = new CountDownLatch(1);
            creator = true;

            tokenRefresh.onRequired(new TokenCallback() {
                @Override
                public void ok(@NonNull String callable) {
                    Log.d(TAG, "ok: refreshTokenResponse response OK " + callable);
                    tokenAccess[0] = callable;
                    if(null != countDownLatch) {
                        countDownLatch.countDown();
                    }
                }

                @Override
                public void error(@NonNull Throwable error) {
                    Log.d(TAG, "ok: refreshTokenResponse response KO " + error);
                    tokenAccess[0] = null;
                    error.printStackTrace();
                    if(null != countDownLatch) {
                        countDownLatch.countDown();
                    }
                }
            });
        }

        Log.d(TAG, "refreshTokenResponse: waiting for countdown latch");
        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
            interrupted = true;
        }

        if(creator) {
            Log.d(TAG, "refreshTokenResponse: releasing");
            countDownLatch = null;
        }

        if(interrupted) {
            Log.e(TAG, "refreshTokenResponse: INTERRUPTED");
            return null;
        }

        Log.d(TAG, "refreshTokenResponse: tokenAccess := " + tokenAccess[0]);
        if(null != tokenAccess[0]) {
            return new TokenResponse(tokenAccess[0], null);
        } else {
            return null;
        }
    }

    /**
     * Retrieve a valid TokenResponse from the servers
     * <p>
     * It will use on of the two available method available :
     * - resolve the injected tokenAccess
     * - resolve a token response obtained from Voxeet servers
     * <p>
     * Note : don't forget to call .then().error()
     *
     * @return a promise to resolve and manage the error
     */
    @NonNull
    @Override
    protected Promise<TokenResponse> retrieveTokenResponse() {
        return new Promise<>(new PromiseSolver<TokenResponse>() {
            @Override
            public void onCall(@NonNull final Solver<TokenResponse> solver) {
                solver.resolve(new TokenResponse(tokenAccess, null));
            }
        });
    }

    @NonNull
    @Override
    protected Request.Builder addHeader(@NonNull Request.Builder builder) {
        return builder;
    }
}
