package voxeet.com.sdk.core.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.audio.AudioRoute;
import com.voxeet.android.media.sensors.ConferenceLock;
import com.voxeet.android.media.sensors.ProximitySensor;
import com.voxeet.android.media.sensors.ScreenSensor;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.voiceengine.WebRtcAudioEffects;
import org.webrtc.voiceengine.WebRtcAudioUtils;

import java.util.ArrayList;
import java.util.List;

import voxeet.com.sdk.audio.SoundManager;
import voxeet.com.sdk.core.AbstractVoxeetService;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;
import voxeet.com.sdk.events.success.ConferenceEndedEvent;
import voxeet.com.sdk.events.success.ConferencePreJoinedEvent;
import voxeet.com.sdk.json.ConferenceDestroyedPush;
import voxeet.com.sdk.utils.AudioType;

public class AudioService extends AbstractVoxeetService {

    private static SoundManager sSoundManager;
    private static final String TAG = AudioService.class.getSimpleName();
    private boolean isAecSoftEnabled = false;
    private boolean isNSSoftEnabled = false;
    private List<ConferenceLock> locks = new ArrayList<>();

    /**
     * Instantiate a new Audio Service
     *
     * @param instance the voxeet parent instance
     */
    public AudioService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        AudioService.preInitSounds(instance.getApplicationContext());
        instance.getEventBus().register(this);

        locks.add(new ProximitySensor(instance.getApplicationContext()));
        locks.add(new ScreenSensor(instance.getApplicationContext()));
    }

    @Nullable
    public static SoundManager getSoundManager() {
        return sSoundManager;
    }

    public static boolean preInitSounds(@NonNull Context applicationContext) {
        if (null == sSoundManager) {
            sSoundManager = new SoundManager(applicationContext);
            return true;
        }

        return false;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferencePreJoinedEvent event) {
        enableAec(true);
        enableNoiseSuppressor(true);

        updateSensors(currentRoute());
        sSoundManager.onConferencePreJoinedEvent();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceEndedEvent event) {
        sSoundManager.onConferenceDestroyedPush();
        releaseSensors();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPush event) {
        sSoundManager.onConferenceDestroyedPush();
        releaseSensors();
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return sSoundManager.getAvailableRoutes();
    }

    public AudioRoute currentRoute() {
        return sSoundManager.currentRoute();
    }

    public boolean setAudioRoute(@NonNull AudioRoute route) {
        boolean new_switch = sSoundManager.setAudioRoute(route);
        updateSensors(currentRoute());

        return new_switch;
    }

    public void setSpeakerMode(boolean isSpeaker) {
        sSoundManager.setSpeakerMode(isSpeaker);
        updateSensors(currentRoute());
    }

    public boolean isBluetoothHeadsetConnected() {
        return sSoundManager.isBluetoothHeadsetConnected();
    }

    public boolean isWiredHeadsetOn() {
        return sSoundManager.isWiredHeadsetOn();
    }

    public boolean isSpeakerOn() {
        return sSoundManager.isSpeakerOn();
    }

    public void acquireLocks() {
        sSoundManager.acquireLocks();
    }

    public void releaseLocks() {
        sSoundManager.releaseLocks();
    }

    public void setInVoiceCallSoundType() {
        sSoundManager.setInVoiceCallSoundType();
    }

    public void resetDefaultSoundType() {
        sSoundManager.resetDefaultSoundType();
    }

    public void abandonAudioFocusRequest() {
        sSoundManager.abandonAudioFocusRequest();
    }

    public void requestAudioFocus() {
        sSoundManager.requestAudioFocus();
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return sSoundManager.setSound(type, assetName);
    }

    public void playSoundType(@NonNull AudioType type) {
        sSoundManager.playSoundType(type);
    }

    public void stopSoundType(@NonNull AudioType audioType) {
        sSoundManager.stopSoundType(audioType);
    }

    public void stop() {
        sSoundManager.stop();
    }

    public void enable() {
        sSoundManager.enable();
    }

    public void disable() {
        sSoundManager.disable();
        stop();
    }

    public void updateSensors(@NonNull AudioRoute route) {
        for (ConferenceLock lock : locks) {
            if (!route.useProximitySensor() && lock.isProximity()) {
                lock.release();
            } else {
                lock.acquire();
            }
        }
    }

    public void releaseSensors() {
        Log.d("ProximitySensor", "releaseSensors: ");
        for (ConferenceLock lock : locks) {
            lock.release();
        }
    }

    public boolean enableAec(boolean enable) {
        WebRtcAudioUtils.setWebRtcBasedAcousticEchoCanceler(enable);
        return true;
    }

    public boolean enableNoiseSuppressor(boolean enable) {
        WebRtcAudioUtils.setWebRtcBasedNoiseSuppressor(enable);
        return true;
    }
}
