package voxeet.com.sdk.core.abs;

import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaSDK;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.SdpCandidate;
import com.voxeet.android.media.audio.AudioRoute;
import com.voxeet.android.media.peer.PendingPeerCallback;
import com.voxeet.android.media.peer.SdpDescription;
import com.voxeet.android.media.peer.SdpMessage;
import com.voxeet.android.media.video.Camera2Enumerator;
import com.voxeet.kernel.R;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.Camera1Enumerator;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBase;
import org.webrtc.VideoRenderer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import voxeet.com.sdk.audio.MessagingEnvironment;
import voxeet.com.sdk.core.ConferenceSimpleState;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.abs.information.ConferenceInformation;
import voxeet.com.sdk.core.abs.information.ConferenceInformationHolder;
import voxeet.com.sdk.core.abs.information.ConferenceState;
import voxeet.com.sdk.core.preferences.VoxeetPreferences;
import voxeet.com.sdk.core.services.AudioService;
import voxeet.com.sdk.core.services.MediaService;
import voxeet.com.sdk.core.services.SdkConferenceService;
import voxeet.com.sdk.core.services.TimeoutRunnable;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;
import voxeet.com.sdk.events.error.CameraSwitchErrorEvent;
import voxeet.com.sdk.events.error.ConferenceCreatedError;
import voxeet.com.sdk.events.error.ConferenceJoinedError;
import voxeet.com.sdk.events.error.ConferenceLeftError;
import voxeet.com.sdk.events.error.GetConferenceHistoryErrorEvent;
import voxeet.com.sdk.events.error.GetConferenceStatusErrorEvent;
import voxeet.com.sdk.events.error.HttpException;
import voxeet.com.sdk.events.error.ParticipantAddedErrorEvent;
import voxeet.com.sdk.events.error.PermissionRefusedEvent;
import voxeet.com.sdk.events.error.ReplayConferenceErrorEvent;
import voxeet.com.sdk.events.error.SdkLogoutErrorEvent;
import voxeet.com.sdk.events.error.SubscribeConferenceErrorEvent;
import voxeet.com.sdk.events.error.SubscribeForCallConferenceErrorEvent;
import voxeet.com.sdk.events.error.UnsubscribeFromCallConferenceErrorEvent;
import voxeet.com.sdk.events.promises.InConferenceException;
import voxeet.com.sdk.events.promises.NotInConferenceException;
import voxeet.com.sdk.events.promises.PromiseConferenceJoinedErrorException;
import voxeet.com.sdk.events.promises.PromiseDeclineConferenceResultEventException;
import voxeet.com.sdk.events.promises.PromiseGetConferenceStatusErrorEventException;
import voxeet.com.sdk.events.promises.PromiseParticipantAddedErrorEventException;
import voxeet.com.sdk.events.promises.PromisePermissionRefusedEventException;
import voxeet.com.sdk.events.success.AddConferenceParticipantResultEvent;
import voxeet.com.sdk.events.success.CameraSwitchSuccessEvent;
import voxeet.com.sdk.events.success.ConferenceCreatingEvent;
import voxeet.com.sdk.events.success.ConferenceCreationSuccess;
import voxeet.com.sdk.events.success.ConferenceDestroyedPushEvent;
import voxeet.com.sdk.events.success.ConferenceEndedEvent;
import voxeet.com.sdk.events.success.ConferenceJoinedSuccessEvent;
import voxeet.com.sdk.events.success.ConferenceLeftSuccessEvent;
import voxeet.com.sdk.events.success.ConferencePreJoinedEvent;
import voxeet.com.sdk.events.success.ConferenceRefreshedEvent;
import voxeet.com.sdk.events.success.ConferenceStatsEvent;
import voxeet.com.sdk.events.success.ConferenceUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUserAddedEvent;
import voxeet.com.sdk.events.success.ConferenceUserCallDeclinedEvent;
import voxeet.com.sdk.events.success.ConferenceUserJoinedEvent;
import voxeet.com.sdk.events.success.ConferenceUserLeftEvent;
import voxeet.com.sdk.events.success.ConferenceUserQualityUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUserUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUsersInvitedEvent;
import voxeet.com.sdk.events.success.DeclineConferenceResultEvent;
import voxeet.com.sdk.events.success.GetConferenceHistoryEvent;
import voxeet.com.sdk.events.success.GetConferenceStatusEvent;
import voxeet.com.sdk.events.success.IncomingCallEvent;
import voxeet.com.sdk.events.success.InvitationReceived;
import voxeet.com.sdk.events.success.OfferCreatedEvent;
import voxeet.com.sdk.events.success.OwnConferenceStartedEvent;
import voxeet.com.sdk.events.success.ParticipantAddedEvent;
import voxeet.com.sdk.events.success.ParticipantUpdatedEvent;
import voxeet.com.sdk.events.success.QualityIndicators;
import voxeet.com.sdk.events.success.QualityUpdatedEvent;
import voxeet.com.sdk.events.success.RecordingStatusUpdate;
import voxeet.com.sdk.events.success.RenegociationEndedEvent;
import voxeet.com.sdk.events.success.RenegociationUpdate;
import voxeet.com.sdk.events.success.ReplayConferenceEvent;
import voxeet.com.sdk.events.success.ResumeConference;
import voxeet.com.sdk.events.success.ScreenStreamAddedEvent;
import voxeet.com.sdk.events.success.ScreenStreamRemovedEvent;
import voxeet.com.sdk.events.success.SdkLogoutSuccessEvent;
import voxeet.com.sdk.events.success.SendBroadcastResultEvent;
import voxeet.com.sdk.events.success.SocketConnectEvent;
import voxeet.com.sdk.events.success.StartRecordingResultEvent;
import voxeet.com.sdk.events.success.StartScreenShareAnswerEvent;
import voxeet.com.sdk.events.success.StartVideoAnswerEvent;
import voxeet.com.sdk.events.success.StopRecordingResultEvent;
import voxeet.com.sdk.events.success.StopScreenShareAnswerEvent;
import voxeet.com.sdk.events.success.StopVideoAnswerEvent;
import voxeet.com.sdk.events.success.SubscribeConferenceEvent;
import voxeet.com.sdk.events.success.SubscribeForCallConferenceAnswerEvent;
import voxeet.com.sdk.events.success.UnSubscribeConferenceAnswerEvent;
import voxeet.com.sdk.events.success.UnSubscribeFromConferenceAnswerEvent;
import voxeet.com.sdk.factories.VoxeetIntentFactory;
import voxeet.com.sdk.json.BroadcastEvent;
import voxeet.com.sdk.json.ConferenceStats;
import voxeet.com.sdk.json.ConferenceUserAdded;
import voxeet.com.sdk.json.CreateConferenceParams;
import voxeet.com.sdk.json.DeviceEvent;
import voxeet.com.sdk.json.InvitationReceivedEvent;
import voxeet.com.sdk.json.OfferCreated;
import voxeet.com.sdk.json.RecordingStatusUpdateEvent;
import voxeet.com.sdk.json.SdkConferenceInvitation;
import voxeet.com.sdk.json.SdkConferenceReplayBody;
import voxeet.com.sdk.json.StartScreenSharingResponse;
import voxeet.com.sdk.json.StartVideoResponse;
import voxeet.com.sdk.json.StopScreenSharingResponse;
import voxeet.com.sdk.json.StopVideoResponse;
import voxeet.com.sdk.json.UserInfo;
import voxeet.com.sdk.json.internal.MetadataHolder;
import voxeet.com.sdk.json.internal.ParamsHolder;
import voxeet.com.sdk.models.CandidatesPush;
import voxeet.com.sdk.models.ConferenceQuality;
import voxeet.com.sdk.models.ConferenceResponse;
import voxeet.com.sdk.models.ConferenceType;
import voxeet.com.sdk.models.ConferenceUserStatus;
import voxeet.com.sdk.models.HistoryConference;
import voxeet.com.sdk.models.NormalConferenceResponse;
import voxeet.com.sdk.models.OfferCandidate;
import voxeet.com.sdk.models.OfferDescription;
import voxeet.com.sdk.models.RecordingStatus;
import voxeet.com.sdk.models.SubscribeConference;
import voxeet.com.sdk.models.abs.Conference;
import voxeet.com.sdk.models.abs.ConferenceUser;
import voxeet.com.sdk.models.impl.DefaultConference;
import voxeet.com.sdk.models.impl.DefaultConferenceUser;
import voxeet.com.sdk.models.impl.DefaultInvitation;
import voxeet.com.sdk.models.impl.DefaultParticipant;
import voxeet.com.sdk.networking.DeviceType;
import voxeet.com.sdk.utils.ConferenceListener;
import voxeet.com.sdk.utils.Twig;
import voxeet.com.sdk.utils.Validate;

/**
 * Conference SDK Service
 * <p>
 * Warning : always implements the error for each promises you may call to start resolve them
 * You can also use the execute method but in case of Exception, you will trigger errors...
 *
 * @see SdkConferenceService
 */
public abstract class AbstractConferenceSdkService<T, COP extends AbstractConferenceSdkObservableProvider<T, DemoEvent>, DemoEvent>
        extends voxeet.com.sdk.core.AbstractVoxeetService<T>
        implements SdkConferenceService, ConferenceListener {

    private final static String TAG = AbstractConferenceSdkService.class.getSimpleName();
    private final VoxeetSdkTemplate mInstance;
    private VoxeetSdkTemplate mSDK;
    private ConferenceListener mListener;
    private AbstractConferenceSdkObservableProvider mConferenceObservableProvider;
    private Twig mTwig;

    private EventBus mEventBus;
    private final MessagingEnvironment mEnvironment;
    private boolean isListenerMode = false;
    private boolean isVideoOn = false; // default camera setting
    private boolean isScreenshareOn = false;
    private boolean isRecording = false;
    private boolean mInConference = false;
    private long mTimeOutTimer = -1;
    private HashMap<String, MediaStream> mapOfStreams = new HashMap<>();
    private HashMap<String, MediaStream> mapOfScreenShareStreams = new HashMap<>();
    private ReentrantLock joinLock = new ReentrantLock();

    @Nullable
    private String mDefaultCamera;
    private boolean mEnableStats;
    private boolean isDefaultOnSpeaker;

    private String mConferenceId = null;
    private ConferenceInformationHolder mConferenceInformationHolder = new ConferenceInformationHolder();

    protected Context mContext;

    private @Nullable
    TimeoutRunnable timeoutRunnable = null;

    private MediaEngine.StreamListener mediaStreamListener = new MediaEngine.StreamListener() {
        @Override
        public void onStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onStreamAdded: stream for peer " + peer);
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    getTwig().i("New mConference user joined with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());

                    mapOfStreams.put(peer, stream);
                    DefaultConferenceUser user = updateConferenceParticipants(peer, ConferenceUserStatus.ON_AIR);
                    if (user != null) {
                        getTwig().i("New mConference user joined with id: " + user.getUserId());

                        if (!peer.equalsIgnoreCase(VoxeetPreferences.id()) && mTimeOutTimer != -1) {
                            mTwig.i("Cancelling timeout timer");
                            removeTimeoutCallbacks();
                        }

                        //update the current conference state specificly from the users point of view
                        updateConferenceFromUsers();
                        mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    } else {
                        Log.d(TAG, "run: unknown user in stream added");
                    }
                }
            });
        }

        @Override
        public void onStreamUpdated(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onStreamUpdated: screen for peer " + peer);
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    getTwig().i("Conference user updated with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());
                    getTwig().i("stream updated having video ? " + (stream.videoTracks().size() > 0) + " screenshare := " + stream.isScreenShare());

                    if (stream.isScreenShare()) mapOfScreenShareStreams.put(peer, stream);
                    else mapOfStreams.put(peer, stream);

                    DefaultConferenceUser user = findUserById(peer);
                    if (user != null) {
                        getTwig().i("Conference user updated with id: " + user.getUserId());

                        //update the current conference state specificly from the users point of view
                        updateConferenceFromUsers();
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    } else {
                        Log.d(TAG, "run: unknown user in stream updated");
                    }
                }
            });
        }

        @Override
        public void onStreamRemoved(@NonNull final String peer) {
            Log.d(TAG, "onStreamRemoved: OnStreamRemoved");
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    onUserLeft(peer);
                }
            });
        }

        @Override
        public void onScreenStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onScreenStreamAdded: screen for peer " + peer);
            getTwig().i("(for ScreenShare) Conference user updated with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());
            getTwig().i("screenshare stream updated having video ? " + stream.videoTracks().size() + " :: has stream inside ? " + (stream.videoTracks().size() > 0));

            mapOfScreenShareStreams.put(peer, stream);
            mEventBus.post(new ScreenStreamAddedEvent(peer, stream));
        }

        @Override
        public void onScreenStreamRemoved(@NonNull String peer) {
            getTwig().i("Screen share stream removed: " + peer);

            mapOfScreenShareStreams.remove(peer);
            mEventBus.post(new ScreenStreamRemovedEvent(peer));
        }

        @Override
        public void onShutdown() {

        }

        @Override
        public void onIceCandidateDiscovered(String peer, SdpCandidate[] candidates) {

            if (null == getConferenceId()) return;

            Call<ResponseBody> call = mConferenceObservableProvider.candidates(getConferenceId(),
                    peer,
                    new CandidatesPush(candidates));

            call.enqueue(new Callback<ResponseBody>() {
                @Override
                public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
                    Log.d(TAG, "onResponse: Candidates sent ");
                }

                @Override
                public void onFailure(Call<ResponseBody> call, Throwable t) {
                    t.printStackTrace();
                }
            });
        }
    };

    private CameraVideoCapturer.CameraEventsHandler cameraEventsHandler = new CameraVideoCapturer.CameraEventsHandler() {
        @Override
        public void onCameraError(String s) {
            Log.d(TAG, "onCameraError: error...");

            stopVideo().then(new PromiseExec<Boolean, Object>() {
                @Override
                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                    Log.d(TAG, "onCall: stopped video after camera issue " + result);
                }
            }).error(new ErrorPromise() {
                @Override
                public void onError(@NonNull Throwable error) {
                    error.printStackTrace();
                }
            });
        }

        @Override
        public void onCameraDisconnected() {
            Log.d(TAG, "onCameraDisconnected");
        }

        @Override
        public void onCameraFreezed(String s) {
            Log.d(TAG, "onCameraFreezed: " + s);
        }

        @Override
        public void onCameraOpening(String s) {
            Log.d(TAG, "onCameraOpening: " + s);
        }

        @Override
        public void onFirstFrameAvailable() {
            Log.d(TAG, "onFirstFrameAvailable");
        }

        @Override
        public void onCameraClosed() {
            Log.d(TAG, "onCameraClosed");
        }
    };

    private boolean isDefaultMute = false;
    private boolean isICERestartEnabled = false;

    public AbstractConferenceSdkService(VoxeetSdkTemplate instance, COP conference_observable_provider, long timeout, @NonNull ServiceProviderHolder<T> holder) {
        super(instance, holder);

        mInstance = instance;

        mEnableStats = false;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mDefaultCamera = new Camera2Enumerator(instance.getApplicationContext()).getNameOfFrontFacingDevice();
        } else {
            Camera1Enumerator enumerator = new Camera1Enumerator(false);

            String[] names = enumerator.getDeviceNames();
            if (null != names && names.length > 0) {
                for (String name : names) {
                    if (enumerator.isFrontFacing(name)) mDefaultCamera = name;
                }
            }
        }

        mConferenceObservableProvider = conference_observable_provider;
        mConferenceObservableProvider.setRetrofitInstantiatedProvider(getService());
        mListener = instance;
        mSDK = instance;
        mTimeOutTimer = timeout;
        mTwig = instance.getTwig();
        mEventBus = EventBus.getDefault();
        mContext = instance.getApplicationContext();
        setDefaultBuiltInSpeaker(true);
        mEnvironment = new MessagingEnvironment.Builder(mContext)
                //.setServerHost(R.string.connectionManagerHost_prod)
                .setServerHost(R.string.session_manager_url)
                .setServerPort(R.string.connectionManagerPort)
                .setStunHost(R.string.stunHost)
                .setStunPort(R.string.stunPort)
                .build();


        register();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     * @deprecated Please use {@link #mute(boolean)} ()} instead.
     */
    @Deprecated
    public boolean muteConference(boolean mute) {
        return mute(mute);
    }

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     */
    @Override
    public boolean mute(boolean mute) {
        if (getMediaService().hasMedia()) {
            if (!mute && getMediaService().getMedia().isMuted()) {
                if (!Validate.hasMicrophonePermissions(context)) {
                    getTwig().i("No permission for mic... please check it");
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                } else {
                    getTwig().i("Conference unmuted");
                    getAudioService().setInVoiceCallSoundType();

                    getMedia().unMute();
                }
            } else if (mute) {
                getTwig().i("Conference muted");

                getAudioService().setInVoiceCallSoundType();
                getMedia().mute();
            }
        }
        return true;
    }

    @Override
    public boolean isMuted() {
        return getMedia() != null && getMedia().isMuted();
    }

    @Override
    public boolean isUserMuted(String userId) {
        DefaultConferenceUser user = findUserById(userId);
        return user != null && user.isMuted();
    }

    @Override
    public void setListenerMode(boolean isListener) {
        if (getMediaService().hasMedia()) {
            MediaSDK media = getMediaService().getMedia();
            if (media.isMuted()) {
                media.unMute();
            } else {
                media.mute();
            }
            getAudioService().setInVoiceCallSoundType();
        }
    }

    @Deprecated
    @Override
    public boolean attachMediaStream(MediaStream stream, VideoRenderer.Callbacks render) {
        if (getMediaService().hasMedia()) getMedia().attachMediaStream(render, stream);
        return getMediaService().hasMedia();
    }

    @Deprecated
    @Override
    public boolean unAttachMediaStream(MediaStream stream, VideoRenderer.Callbacks render) {
        if (getMediaService().hasMedia()) getMedia().unattachMediaStream(render, stream);
        return getMediaService().hasMedia();
    }

    @Override
    public void register() {
        registerEventBus();
    }

    @Override
    public void unregister() {
        unRegisterEventBus();
    }

    public void setDefaultCamera(String cameraName) {
        mDefaultCamera = cameraName;
    }

    @Override
    public void toggleVideo() {
        Promise<Boolean> promise = isVideoOn() ? stopVideo() : startVideo();
        promise
                .then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                        Log.d(TAG, "onSuccess: toggleVideo " + result);
                    }
                })
                .error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                    }
                });
    }

    public void toggleScreenShare() {
        if (isScreenShareOn()) {
            stopScreenShare()
                    .then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onSuccess: toggleScreenShare " + result);
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            Log.d(TAG, "onError: toggleScreenShare");
                            error.printStackTrace();
                        }
                    });
        } else {
            getVoxeetSDK().getScreenShareService().sendRequestStartScreenShare();
        }
    }

    @Override
    public String getCurrentConferenceId() {
        Validate.runningOnUiThread();
        return mConferenceId;
    }

    @Override
    public int getConferenceRoomSize() {
        Conference conference = getCurrentConferenceInformation().getConference();
        return conference.getConferenceRoomSize();
    }

    @Override
    public String currentSpeaker() {
        Conference conference = getConference();
        if (!hasMedia() || conference == null) {
            return VoxeetPreferences.id();
        } else {
            String currentSpeaker = null;
            for (DefaultConferenceUser user : conference.getConferenceUsers()) {
                if (user.getUserId() != null
                        && !user.getUserId().equals(VoxeetPreferences.id())
                        && ConferenceUserStatus.ON_AIR.equals(user.getConferenceStatus())) {
                    double peerVuMeter = getMedia().getPeerVuMeter(user.getUserId());
                    if (currentSpeaker == null || (peerVuMeter > 0.001 && getMedia().getPeerVuMeter(currentSpeaker) < peerVuMeter))
                        currentSpeaker = user.getUserId();
                }
            }
            return currentSpeaker;
        }
    }

    @NonNull
    private String currentUserOrEmpty() {
        String currentUserId = VoxeetPreferences.id();
        return null != currentUserId ? currentUserId : "";
    }

    @Override
    public double getSdkPeerVuMeter(@Nullable String peerId) {
        Validate.runningOnUiThread();
        return hasMedia() && null != peerId ? getMedia().getPeerVuMeter(peerId) : 0;
    }

    @Override
    public DefaultConferenceUser findUserById(final String userId) {
        Conference conference = getConference();
        return null != conference ? findUserById(conference, userId) : null;
    }

    private DefaultConferenceUser findUserById(@NonNull Conference conference,
                                               final String userId) {
        //modification from kevin : no need to crash <3
        //Validate.notNull(userId, "user id");

        return Iterables.find(conference.getConferenceUsers(), new Predicate<DefaultConferenceUser>() {
            @Override
            public boolean apply(DefaultConferenceUser input) {
                return userId.equalsIgnoreCase(input.getUserId());
            }
        }, null);
    }

    @Override
    public String getAliasId() {
        DefaultConference conference = getConference();
        return null != conference ? conference.getConferenceAlias() : null;
    }

    @Override
    public String getConferenceId() {
        return mConferenceId;
    }

    @Deprecated
    public List<AudioRoute> getAvailableRoutes() {
        return getAudioService().getAvailableRoutes();
    }

    @Deprecated
    public AudioRoute currentRoute() {
        return getAudioService().currentRoute();
    }

    @Override
    @Nullable
    public EglBase.Context getEglContext() {
        return getMediaService().getEglContext();
    }

    @Override
    public long getTimeout() {
        return mTimeOutTimer;
    }

    @Override
    public void toggleRecording() {
        Promise<Boolean> promise = isRecording ? stopRecording() : startRecording();
        promise
                .then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                        Log.d(TAG, "onSuccess: toggle done " + result);
                    }
                })
                .error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                    }
                });
    }

    @Override
    public boolean muteUser(@NonNull String userId, boolean shouldMute) {
        Validate.runningOnUiThread();

        DefaultConferenceUser user = findUserById(userId);
        if (user != null) {
            getTwig().i("Setting mute property for mConference participant with id " + userId + " to " + shouldMute);

            startTransactionConferenceUser();
            user.setMuted(shouldMute);
            getMedia().changePeerGain(userId, shouldMute ? MUTE_FACTOR : UNMUTE_FACTOR);
            commitTransactionConferenceUser();
        }
        return user != null;
    }

    public boolean isVideoOn() {
        return isVideoOn;
    }

    public boolean isScreenShareOn() {
        return isScreenshareOn;
    }

    @Deprecated
    public boolean setAudioRoute(AudioRoute route) {
        return getAudioService().setAudioRoute(route);
    }

    public boolean setDefaultMute(boolean default_state) {
        isDefaultMute = default_state;
        return true;
    }

    public boolean setDefaultBuiltInSpeaker(boolean default_state) {
        isDefaultOnSpeaker = default_state;
        VoxeetPreferences.setDefaultBuiltInSpeakerOn(default_state);
        return true;
    }

    @Override
    public boolean setTimeOut(long timeout) {
        getTwig().i("Timeout set to " + timeout);
        mTimeOutTimer = timeout;
        return true;
    }

    @Override
    public boolean setUserPosition(final String userId, final double angle, final double distance) {
        if (hasMedia()) getMedia().changePeerPosition(userId, angle, distance);
        return hasMedia();
    }

    /**
     * Get the current conference type
     *
     * @return a ConferenceType, ConferenceType.NONE is the default value
     */
    @NonNull
    public ConferenceSimpleState getConferenceType() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information || null == mConferenceId) return ConferenceSimpleState.NONE;
        return information.getConferenceType();
    }

    /**
     * Get the current mConference object
     *
     * @return a nullable object
     */
    @Nullable
    public DefaultConference getConference() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return null;

        return information.getConference();
    }

    @NonNull
    private DefaultConference getOrSetConference(@NonNull String conferenceId) {
        ConferenceInformation information = getConferenceInformation(conferenceId);
        if (null != information) return information.getConference();
        Log.d(TAG, "getOrSetConference: INVALID CALL FOR GET OR SET CONFERENCE WITH " + conferenceId);
        return new DefaultConference();
    }

    /**
     * Get the current state of joining
     * in conjunction of the getConference() it is easy to check the current state :
     * <p>
     * mInConference() + getConference() non null = mConference joined and on air
     * mInConference() + getConference() null = joining mConference with the answer from server for now
     * !mInConference() = not currently joining, joined mConference (or left)
     *
     * @return if in mConference or "attempting" to join
     */
    public boolean isInConference() {
        return mInConference;
    }

    protected void setIsInConference(boolean status) {
        mInConference = status;
    }

    public double getPeerVuMeter(@Nullable String peerId) {
        Validate.runningOnUiThread();

        if (hasMedia() && null != peerId) return getMedia().getPeerVuMeter(peerId);
        return 0.;
    }

    @NonNull
    @Override
    public List<DefaultConferenceUser> getConferenceUsers() {
        DefaultConference conference = getConference();
        if (null == conference) {
            Log.d(TAG, "getConferenceUsers: returning a new instance :: NOT IN A CONFERENCE");
            return new ArrayList<>();
        }

        //Log.d(TAG, "getConferenceUsers: >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>");
        //for (DefaultConferenceUser user : conference.getConferenceUsers()) {
        //    Log.d(TAG, "getConferenceUsers: user " + user.toString());
        //}
        //Log.d(TAG, "getConferenceUsers: <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<");

        return conference.getConferenceUsers();
    }

    @NonNull
    public List<DefaultConferenceUser> getLastInvitationUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return new ArrayList<>();
        return information.getLastInvitationReceived();
    }

    @Override
    public boolean isLive() {
        return null != getConference();
    }

    @Override
    public boolean isListenerMode() {
        return isListenerMode;
    }

    @Nullable
    public ConferenceUser getUser(final String userId) {
        Conference conference = getConference();
        if (null != conference && conference.getConferenceUsers() != null) {
            return Iterables.find(conference.getConferenceUsers(), new Predicate<ConferenceUser>() {
                @Override
                public boolean apply(ConferenceUser input) {
                    return userId.equals(input.getUserId());
                }
            }, null);
        }
        return null;
    }

    /**
     * Deprecated since it will possibly be removed in the near future
     *
     * @return the map of streams available
     */
    @Deprecated
    @NonNull
    public HashMap<String, MediaStream> getMapOfStreams() {
        return mapOfStreams;
    }

    /**
     * Deprecated since it will possibly be removed in the near future
     *
     * @return the map of screenshare streams available
     */
    @Deprecated
    @NonNull
    public HashMap<String, MediaStream> getMapOfScreenShareStreams() {
        return mapOfScreenShareStreams;
    }

    /**
     * Give the ability to cancel the timeout in some cases
     */
    public AbstractConferenceSdkService<T, COP, DemoEvent> cancelTimeout() {
        removeTimeoutCallbacks();

        return this;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected abstract Promise<Boolean> onCreateDemoSuccess(DemoEvent response);

    /**
     * Join a conference with a conference
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     *
     * @param conferenceId id of the conference to join as a listener
     * @return a promise to resolve
     */
    @Override
    public Promise<Boolean> listenConference(@NonNull String conferenceId) {
        isListenerMode = true;
        getTwig().i("Listener mode set to true");

        //init the conference with the proper conferenceId and alias to null
        createOrSetConferenceWithParams(conferenceId, null);

        return joinVoxeetConference(conferenceId);
    }

    /**
     * resolve -> the result event
     * reject -> network error
     *
     * @param conferenceId the conference id to decline
     * @return a promise to resolve
     */
    @Override
    public Promise<DeclineConferenceResultEvent> decline(final String conferenceId) {
        return new Promise<>(new PromiseSolver<DeclineConferenceResultEvent>() {
            @Override
            public void onCall(@NonNull final Solver<DeclineConferenceResultEvent> solver) {

                final Call<ResponseBody> user = mConferenceObservableProvider.getDeclineObservable(conferenceId);
                enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        getTwig().i("IConference declined with id: " + conferenceId);

                        DeclineConferenceResultEvent event = new DeclineConferenceResultEvent(response.code() == 200);
                        mEventBus.post(event);
                        solver.resolve(event);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable t, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e(t);

                        DeclineConferenceResultEvent event = new DeclineConferenceResultEvent(handleError(t), false);
                        mEventBus.post(event);
                        solver.reject(new PromiseDeclineConferenceResultEventException(event));
                    }
                });
            }
        });
    }

    public void sendRequestStartScreenShare() {
        getVoxeetSDK().getScreenShareService().sendRequestStartScreenShare();
    }

    public void onUserCanceledScreenShare() {
        //call before any specific code so stopScreenCapturer() should do nothing /!\
        if (hasMedia()) {
            getMedia().stopScreenCapturer();
        }
        isScreenshareOn = false;
    }

    public Promise<Boolean> startScreenShare(final Intent intent,
                                             final int width,
                                             final int height) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    if (isScreenShareOn()) {
                        solver.resolve(true);
                    }

                    if (!hasMedia()) {
                        Log.d(TAG, "startScreenShare: media is null");
                        throw new MediaEngineException("Media is null, invalid state");
                    }

                    isScreenshareOn = true;
                    getMedia().startScreenCapturer(intent, width, height);

                    final Call<StartScreenSharingResponse> startScreenShare = mConferenceObservableProvider.getStartScreenShareObservable(mConferenceId, VoxeetPreferences.id());
                    enqueue(startScreenShare, new HttpCallback<StartScreenSharingResponse>() {
                        @Override
                        public void onSuccess(@NonNull StartScreenSharingResponse object, @NonNull Response<StartScreenSharingResponse> response) {
                            StartScreenSharingResponse body = response.body();
                            isScreenshareOn = true;

                            final StartScreenShareAnswerEvent event = new StartScreenShareAnswerEvent(true);
                            mEventBus.post(event);
                            createVideoAnswer(body.getUserId(), body.getDescription(), body.getCandidates())
                                    .then(new PromiseExec<Boolean, Object>() {
                                        @Override
                                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                            Log.d(TAG, "onCall: createVideoAnswer := having result " + aBoolean);
                                            solver.resolve(true);
                                        }
                                    })
                                    .error(new ErrorPromise() {
                                        @Override
                                        public void onError(Throwable error) {
                                            error.printStackTrace();
                                            solver.resolve(true);
                                        }
                                    });
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<StartScreenSharingResponse> response) {
                            HttpException.dumpErrorResponse(response);

                            mTwig.e(e);
                            isScreenshareOn = false;

                            StartScreenShareAnswerEvent event = new StartScreenShareAnswerEvent(false);
                            mEventBus.post(event);
                            solver.reject(e);
                        }
                    });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                }
            }
        });
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopScreenShare() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    String conferenceId = getCurrentConferenceId();

                    if (!isInConference()) {
                        Log.d(TAG, "stopScreenShare: not in conf");

                        if (hasMedia()) getMedia().stopScreenCapturer();
                        throw new NotInConferenceException();
                    }

                    if (!hasMedia()) {
                        Log.d(TAG, "stopScreenShare: media is null");
                        throw new MediaEngineException("media is null");
                    }

                    getMedia().stopScreenCapturer();

                    final Call<StopScreenSharingResponse> stopVideo = mConferenceObservableProvider.getStopScreenShareObservable(conferenceId, VoxeetPreferences.id());
                    enqueue(stopVideo, new HttpCallback<StopScreenSharingResponse>() {
                        @Override
                        public void onSuccess(@NonNull StopScreenSharingResponse object, @NonNull Response<StopScreenSharingResponse> response) {
                            getMedia().stopScreenCapturer();

                            isScreenshareOn = false;

                            final StopScreenShareAnswerEvent event = new StopScreenShareAnswerEvent(true);
                            mEventBus.post(event);
                            createVideoAnswer(object.getUserId(), object.getDescription(), object.getCandidates())
                                    .then(new PromiseExec<Boolean, Object>() {
                                        @Override
                                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                            //TODO check for possible call in the createVideoAnswer (callback ?)
                                            solver.resolve(true);
                                        }
                                    })
                                    .error(new ErrorPromise() {
                                        @Override
                                        public void onError(Throwable error) {
                                            error.printStackTrace();
                                            //TODO check for possible call in the createVideoAnswer (callback ?)
                                            solver.resolve(true);
                                        }
                                    });
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<StopScreenSharingResponse> response) {
                            HttpException.dumpErrorResponse(response);

                            getTwig().e(e);

                            isScreenshareOn = false;

                            StopScreenShareAnswerEvent event = new StopScreenShareAnswerEvent(false);
                            mEventBus.post(event);
                            solver.reject(e);
                        }
                    });
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                } catch (Exception exception) {
                    solver.reject(exception);
                }
            }
        });
    }

    /**
     * Start the video
     * resolve -> StartVideoAnswerEvent
     * reject -> PromisePermissionRefusedEventException
     * reject -> MediaException
     * <p>
     * If the app does not have the permission, will fire a PermissionRefusedEvent
     */
    public Promise<Boolean> startVideo() {
        //return new StartVideo(this).createPromise();
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    if (!Validate.hasCameraPermissions(getContext())) {
                        PermissionRefusedEvent event = new PermissionRefusedEvent(PermissionRefusedEvent.Permission.CAMERA);
                        getEventBus().post(event);
                        throw new PromisePermissionRefusedEventException(event);
                    }

                    if (!getMediaService().hasMedia()) {
                        Log.d(TAG, "startVideo: media is null");
                        throw new MediaEngineException("Media is null, invalid state");
                    }

                    if (isVideoOn()) {
                        Log.d(TAG, "startVideo: already started... please wait if you wanted to stop");
                        solver.resolve(true);//new StartVideoAnswerEvent(false, true));
                        return;
                    }

                    if (null == getDefaultCamera()) {
                        Log.d(TAG, "startVideo: unable to load video getDefaultCamera():=" + getDefaultCamera());
                        throw new MediaEngineException("DefaultCamera is null, invalid state");
                    }

                    ConferenceInformation information = getCurrentConferenceInformation();
                    if(null != information) {
                        information.setOwnVideoStarted(true);
                    }

                    getMediaService().getMedia().startVideo(getDefaultCamera());

                    final Call<StartVideoResponse> startVideo = mConferenceObservableProvider.getStartVideoObservable(getConferenceId(), VoxeetPreferences.id());
                    enqueue(startVideo, new AbstractConferenceSdkService.HttpCallback<StartVideoResponse>() {
                        @Override
                        public void onSuccess(@NonNull StartVideoResponse object, @NonNull Response<StartVideoResponse> response) {
                            //remove start video since it is not to be used
                            //media.startVideo(mDefaultCamera);

                            isVideoOn = true;

                            final StartVideoAnswerEvent event = new StartVideoAnswerEvent(true);
                            mEventBus.post(event);
                            createVideoAnswer(object.getUserId(), object.getDescription(), object.getCandidates())
                                    .then(new PromiseExec<Boolean, Object>() {
                                        @Override
                                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                            solver.resolve(true);
                                        }
                                    })
                                    .error(new ErrorPromise() {
                                        @Override
                                        public void onError(Throwable error) {
                                            error.printStackTrace();
                                            solver.resolve(true);
                                        }
                                    });
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<StartVideoResponse> response) {
                            HttpException.dumpErrorResponse(response);

                            if(null != information) {
                                information.setOwnVideoStarted(false);
                            }
                            mTwig.e(e);
                            isVideoOn = false;

                            StartVideoAnswerEvent event = new StartVideoAnswerEvent(false);
                            mEventBus.post(event);
                            solver.reject(e);
                        }
                    });
                    //explicitly keep the list of the different exception
                    //this methods throws
                } catch (PromisePermissionRefusedEventException exception) {
                    solver.reject(exception);
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                } catch (Exception exception) {
                    solver.reject(exception);
                }
            }
        });
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopVideo() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    String conferenceId = getCurrentConferenceId();

                    if (!isInConference()) {
                        Log.d(TAG, "stopVideo: not in conf");

                        if (hasMedia()) getMedia().stopVideo();
                        throw new NotInConferenceException();
                    }

                    if (!hasMedia()) {
                        Log.d(TAG, "startVideo: media is null");
                        throw new MediaEngineException("media is null");
                    }
                    //Validate.notNull(mConferenceId, "mConference Id");
                    //Validate.notNull(media, "media");

                    //we don't change the video state here since we don't want to override the possible feature of stopping
                    getMedia().stopVideo();

                    final Call<StopVideoResponse> stopVideo = mConferenceObservableProvider.getStopVideoObservable(conferenceId, VoxeetPreferences.id());
                    enqueue(stopVideo, new HttpCallback<StopVideoResponse>() {
                        @Override
                        public void onSuccess(@NonNull StopVideoResponse object, @NonNull Response<StopVideoResponse> response) {
                            getMedia().stopVideo();

                            isVideoOn = false;

                            final StopVideoAnswerEvent event = new StopVideoAnswerEvent(true);
                            mEventBus.post(event);
                            createVideoAnswer(object.getUserId(), object.getDescription(), object.getCandidates())
                                    .then(new PromiseExec<Boolean, Object>() {
                                        @Override
                                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                            //TODO check for possible call in the createVideoAnswer (callback ?)
                                            solver.resolve(true);
                                        }
                                    })
                                    .error(new ErrorPromise() {
                                        @Override
                                        public void onError(Throwable error) {
                                            error.printStackTrace();
                                            //TODO check for possible call in the createVideoAnswer (callback ?)
                                            solver.resolve(true);
                                        }
                                    });
                        }

                        @Override
                        public void onFailure(@NonNull Throwable t, @Nullable Response<StopVideoResponse> response) {
                            solver.reject(t);
                        }
                    });
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                } catch (Exception exception) {
                    solver.reject(exception);
                }
            }
        });
    }

    /**
     * Create a video answer and post it
     * <p>
     * resolve -> true -> everything went fine
     * resolve -> false -> on error occured
     * reject -> media exception
     *
     * @param userId
     * @param offerDescription
     * @param offerCandidates
     * @return
     */
    private Promise<Boolean> createVideoAnswer(final String userId, final OfferDescription offerDescription, final List<OfferCandidate> offerCandidates) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Log.d(TAG, "createVideoAnswer: " + userId + " " + offerDescription.getSdp() + " " + offerDescription.getType());
                Log.d("SDKMEDIA", "createVideoAnswer: " + userId + " " + offerDescription.getSdp() + " " + offerDescription.getType());

                try {
                    if (!hasMedia()) {
                        throw new MediaEngineException("media is null");
                    }
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                }

                SdpDescription description = new SdpDescription(offerDescription.getType(), offerDescription.getSdp());

                List<SdpCandidate> candidates = new ArrayList<>();
                if (offerCandidates != null) {
                    for (OfferCandidate candidate : offerCandidates) {
                        candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                    }
                }

                try {
                    getMedia().createAnswerForPeer(userId,
                            description.getSsrc(),
                            description,
                            candidates,
                            VoxeetPreferences.id().equalsIgnoreCase(userId),
                            new PendingPeerCallback() {
                                @Override
                                public void onMessage(@Nullable SdpMessage message) {
                                    Log.d(TAG, "onMessage: having a message " + message);
                                    answer(userId, message)
                                            .then(new PromiseExec<Integer, Object>() {
                                                @Override
                                                public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                    Log.d(TAG, "onSuccess: " + result);

                                                    //resolve true value : all clear
                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    if (error instanceof PromiseParticipantAddedErrorEventException)
                                                        mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                    else
                                                        error.printStackTrace();

                                                    //resolve false value : something happened
                                                    solver.resolve(false);
                                                }
                                            });
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @return a promise to resolve
     */
    @Override
    public Promise<ConferenceResponse> create() {
        return create(null, null, null);
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param metadata the MetadataHolder
     * @return a promise to resolve
     */
    public Promise<ConferenceResponse> create(@Nullable MetadataHolder metadata,
                                              @Nullable ParamsHolder params) {
        return create(null, metadata, params);
    }


    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceAlias create a given conference then join it
     * @return a promise to resolve
     */
    public Promise<ConferenceResponse> create(@Nullable final String conferenceAlias) {
        return create(conferenceAlias, null, null);
    }

    public Promise<ConferenceResponse> create(@Nullable String conferenceAlias,
                                              @Nullable MetadataHolder metadata,
                                              @Nullable ParamsHolder paramsholder) {
        Promise<ConferenceResponse> promise = new Promise<>(new PromiseSolver<ConferenceResponse>() {
            @Override
            public void onCall(@NonNull final Solver<ConferenceResponse> solver) {
                //TODO SET STATE FOR CONFERENCE ALIAS

                //setIsInConference(true);

                getEventBus().post(new ConferenceCreatingEvent(conferenceAlias, null));
                Log.d(TAG, "Attempting to create mConferene alias:=" + conferenceAlias);

                CreateConferenceParams params = new CreateConferenceParams()
                        .setMetadataHolder(metadata)
                        .setParamsHolder(paramsholder);
                //        .setStats(mEnableStats);

                if (null != conferenceAlias) params.setConferenceAlias(conferenceAlias);

                //now create the conference and will retry in case of issue...
                internalCreate(true, params, solver);
            }
        });

        if (mInstance.isSocketOpen()) {
            return promise;
        } else {
            return new Promise<ConferenceResponse>(new PromiseSolver<ConferenceResponse>() {
                @Override
                public void onCall(@NonNull Solver<ConferenceResponse> solver) {
                    mInstance.logCurrentlySelectedUserWithChain().then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver internal_solver) {
                            promise.then(new PromiseExec<ConferenceResponse, Object>() {
                                @Override
                                public void onCall(@Nullable ConferenceResponse result, @NonNull Solver<Object> internal_solver) {
                                    solver.resolve(result);
                                }
                            }).error(new ErrorPromise() {
                                @Override
                                public void onError(@NonNull Throwable error) {
                                    solver.reject(error);
                                }
                            });
                        }
                    }).error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            solver.reject(error);
                        }
                    });
                }
            });
        }
    }

    /**
     * Create a conference and retry if neede
     *
     * @param retry  retry or not
     * @param params the params to send
     * @param solver the solver to resolve
     */
    private void internalCreate(boolean retry, CreateConferenceParams params, Solver<ConferenceResponse> solver) {
        Call<ConferenceResponse> observable = mConferenceObservableProvider.getCreateConferenceObservable(params);
        enqueue(observable, new HttpCallback<ConferenceResponse>() {
            @Override
            public void onSuccess(@NonNull ConferenceResponse object, @NonNull Response<ConferenceResponse> response) {
                Log.d(TAG, "onNext: having conference");
                createOrSetConferenceWithParams(object.getConfId(), object.getConfAlias());

                getConferenceInformation(object.getConfId()).setConferenceState(ConferenceState.CREATED);
                ConferenceInformation information = getCurrentConferenceInformation();
                information.setConferenceType(ConferenceSimpleState.CONFERENCE);

                Log.d(TAG, "internalCreate onNext: join with := " + object.getConfId() + " " + object.getConfAlias());

                solver.resolve(object);
            }

            @Override
            public void onFailure(@NonNull Throwable e, @Nullable Response<ConferenceResponse> response) {
                HttpException.dumpErrorResponse(response);
                e.printStackTrace();
                mTwig.e(e);

                if (!retry) {
                    Log.d(TAG, "internalCreate onFailure: conference creation failed ! but no retry... quit...");
                    setIsInConference(false);

                    closeMedia();

                    mEventBus.post(new ConferenceCreatedError(handleError(e)));
                    solver.reject(e);
                } else {
                    Log.d(TAG, "internalCreate onFailure: conference creation failed ! but retry... now...");
                    internalCreate(false, params, solver);
                }
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceId the existing conference to join
     * @return a promise to resolve
     */
    public Promise<Boolean> join(@NonNull final String conferenceId) {
        Promise<Boolean> promise = new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                //locking
                joinLock();

                String currentConferenceId = getConferenceId();
                ConferenceInformation holder = getConferenceInformation(conferenceId);

                boolean init_done = null != holder &&
                        (ConferenceState.JOINING.equals(holder.getConferenceState())
                                || ConferenceState.JOINED.equals(holder.getConferenceState()));

                if (null != currentConferenceId && currentConferenceId.equals(conferenceId) && init_done) {
                    Log.d(TAG, "onCall: join the conference " + conferenceId + " // invalid call ! already calling it");
                    Log.d(TAG, "onCall: btw, the current " + conferenceId + " state is := " + holder.getConferenceState());

                    solver.resolve(true);
                    joinUnlock();
                    return;
                }

                //prevent issue with the join
                setIsInConference(true);

                DefaultConference conference = holder.getConference();

                getEventBus().post(new ConferencePreJoinedEvent(conferenceId, conference.getConferenceAlias()));
                Log.d(TAG, "Attempting to join conference alias:=" + conference.getConferenceAlias() + " conferenceId:=" + conferenceId);

                Log.d(TAG, "onNext: having conference");
                createOrSetConferenceWithParams(conference.getConferenceId(), conference.getConferenceAlias());

                getCurrentConferenceInformation().setConferenceState(ConferenceState.JOINING);
                ConferenceInformation information = getCurrentConferenceInformation();
                information.setConferenceType(ConferenceSimpleState.CONFERENCE);

                Log.d(TAG, "onNext: join with := " + conference.getConferenceId() + " " + conference.getConferenceAlias());

                joinUnlock();

                //fromthe onConferenceCreated deprecated method
                //createOrSetConferenceWithParams(conferenceId, conferenceAlias);
                //getConferenceInformation(conferenceId).setConferenceState(ConferenceState.CREATED);

                //getTwig().i("IConference created with id: " + mConferenceId);

                //mEventBus.post(new ConferenceCreationSuccess(conferenceId, conferenceAlias));
                mEventBus.post(new ConferenceCreationSuccess(conferenceId, conference.getConferenceAlias()));

                joinVoxeetConference(conferenceId).then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                        solver.resolve(false);
                    }
                });
                //solver.resolve(onConferenceCreated(conference.getConferenceAlias(), conference.getConferenceAlias(), null));


            }
        });

        if (mInstance.isSocketOpen()) {
            return promise;
        } else {
            return new Promise<Boolean>(new PromiseSolver<Boolean>() {
                @Override
                public void onCall(@NonNull Solver<Boolean> solver) {
                    Log.d(TAG, "onCall: joining but the socket is disconnected");
                    mInstance.logCurrentlySelectedUserWithChain().then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver internal_solver) {
                            Log.d(TAG, "onCall: connection ? " + result);
                            promise.then(new PromiseExec<Boolean, Object>() {
                                @Override
                                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                                    Log.d(TAG, "onCall: join ok? " + result);
                                    solver.resolve(result);
                                }
                            }).error(new ErrorPromise() {
                                @Override
                                public void onError(@NonNull Throwable error) {
                                    Log.d(TAG, "onError: join ko");
                                    solver.reject(error);
                                }
                            });
                        }
                    }).error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            solver.reject(error);
                        }
                    });
                }
            });
        }
    }


    protected Promise<ResumeConference> extendedJoin(final String conferenceId) {
        return new Promise<>(new PromiseSolver<ResumeConference>() {
            @Override
            public void onCall(@NonNull final Solver<ResumeConference> solver) {

                Call<ResumeConference> observable = mConferenceObservableProvider.joinConference(conferenceId, new DeviceEvent(DeviceType.ANDROID, isListenerMode()));
                enqueue(observable, new HttpCallback<ResumeConference>() {
                    @Override
                    public void onSuccess(@NonNull ResumeConference object, @NonNull Response<ResumeConference> response) {
                        solver.resolve(object);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable t, @Nullable Response<ResumeConference> response) {
                        HttpException.dumpErrorResponse(response);

                        solver.reject(t);
                    }
                });
            }
        });
    }


    protected Promise<Boolean> onConferenceCreated(@NonNull final String conferenceId,
                                                   @NonNull final String conferenceAlias,
                                                   @Nullable final NormalConferenceResponse other) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                createOrSetConferenceWithParams(conferenceId, conferenceAlias);
                getConferenceInformation(conferenceId).setConferenceState(ConferenceState.CREATED);

                getTwig().i("IConference created with id: " + mConferenceId);

                mEventBus.post(new ConferenceCreationSuccess(conferenceId, conferenceAlias));

                joinVoxeetConference(conferenceId).then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * resole -> true
     * reject -> network error or on created error
     *
     * @return
     */
    @Override
    public Promise<Boolean> demo() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                getTwig().i("Attempting to create demo mConference");

                final Call<DemoEvent> user = mConferenceObservableProvider.getCreateDemoObservable();
                enqueue(user, new HttpCallback<DemoEvent>() {
                    @Override
                    public void onSuccess(@NonNull DemoEvent object, @NonNull Response<DemoEvent> response) {
                        onCreateDemoSuccess(object)
                                .then(new PromiseExec<Boolean, Object>() {
                                    @Override
                                    public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                        ConferenceInformation information = getCurrentConferenceInformation();
                                        information.setConferenceType(ConferenceSimpleState.CONFERENCE);

                                        solver.resolve(true);
                                    }
                                })
                                .error(new ErrorPromise() {
                                    @Override
                                    public void onError(Throwable error) {
                                        solver.reject(error);
                                    }
                                });
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<DemoEvent> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e(e);

                        mEventBus.post(new ConferenceCreatedError(handleError(e)));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> list of ConferenceRefreshedUsers
     * reject -> network error
     *
     * @param ids a non null list of non null strings
     * @return a non null promise
     */
    @NonNull
    public Promise<List<ConferenceRefreshedEvent>> inviteUserInfos(final List<UserInfo> ids) {
        return new Promise<>(new PromiseSolver<List<ConferenceRefreshedEvent>>() {
            @Override
            public void onCall(@NonNull Solver<List<ConferenceRefreshedEvent>> solver) {
                //Warning = getConferenceUsers is returning a new non retained if not in a conference
                List<DefaultConferenceUser> users = getConferenceUsers();
                List<String> strings = new ArrayList<>();

                if (null != ids) {
                    for (UserInfo info : ids) {
                        if (null != info && info.getExternalId() != null) {
                            int i = 0;
                            while (i < users.size()) {
                                DefaultConferenceUser user = users.get(i);
                                if (null != user && null != user.getUserInfo()
                                        && info.getExternalId().equalsIgnoreCase(user.getUserInfo().getExternalId())) {
                                    user.updateIfNeeded(info.getName(), info.getAvatarUrl());
                                }
                                i++;
                            }
                            strings.add(info.getExternalId());
                        }
                    }
                }

                solver.resolve(invite(strings));
            }
        });
    }

    /**
     * resolve -> list of ConferenceRefreshedUsers
     * reject -> network error
     *
     * @param ids a non null list of non null strings
     * @return a non null promise
     */
    @NonNull
    @Override
    public Promise<List<ConferenceRefreshedEvent>> invite(final List<String> ids) {
        return new Promise<>(new PromiseSolver<List<ConferenceRefreshedEvent>>() {
            @Override
            public void onCall(@NonNull final Solver<List<ConferenceRefreshedEvent>> solver) {
                //remove the timeout
                removeTimeoutCallbacks();

                //a new one is now sent
                sendTimeoutCallbacks();

                boolean sdk = isSDK();
                List<String> voxeetIds = sdk ? null : new ArrayList<>();
                final List<String> externalIds = new ArrayList<>();

                externalIds.addAll(ids);
                if (!sdk) voxeetIds.addAll(ids);

                //send an error if you're not in a conference
                if (null == mConferenceId) {
                    try {
                        throw new IllegalStateException("You're not in a conference");
                    } catch (Exception e) {
                        mEventBus.post(new AddConferenceParticipantResultEvent(handleError(e), false));
                        solver.reject(e);
                        return;
                    }
                }

                final Call<ResponseBody> user = mConferenceObservableProvider.getInviteObservable(mConferenceId, new SdkConferenceInvitation(voxeetIds, externalIds));
                enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        List<ConferenceRefreshedEvent> list = new ArrayList<>();
                        mEventBus.post(new AddConferenceParticipantResultEvent(response.code() == 200));

                        try {
                            String body = response.body().string();
                            Log.d(TAG, "onNext: " + body);
                        } catch (IOException e) {
                            e.printStackTrace();
                        }

                        if (response.code() == 200) {
                            for (String userId : externalIds) {
                                if (!userId.equals(VoxeetPreferences.id())) {
                                    getTwig().i("Conference participant with id: " + userId + " invited");
                                    ConferenceRefreshedEvent event = new ConferenceRefreshedEvent(userId, updateConferenceParticipants(userId, ConferenceUserStatus.IN_PROGRESS));


                                    mEventBus.post(event);
                                    list.add(event);
                                }
                            }
                        }
                        solver.resolve(list);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e(e);

                        mEventBus.post(new AddConferenceParticipantResultEvent(handleError(e), false));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false response from server is an error
     * reject -> network error
     *
     * @return
     */
    @Override
    public Promise<Boolean> logout() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                String id = VoxeetPreferences.id();
                if (null != id) {
                    mTwig.i("Attempting to logout");

                    final Call<ResponseBody> user = mConferenceObservableProvider.getLogOutObservable(VoxeetPreferences.token());
                    enqueue(user, new HttpCallback<ResponseBody>() {
                        @Override
                        public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                            String message;
                            if (response.code() == 200) {
                                message = "Logout success";

                                getTwig().i(message);

                                VoxeetPreferences.onLogout();
                                getVoxeetSDK().closeSocket();
                                getVoxeetSDK().cleanUserSession(id);

                                mEventBus.post(new SdkLogoutSuccessEvent(message));
                                solver.resolve(true);
                            } else {
                                message = "Logout failed";

                                getTwig().e(message);

                                VoxeetPreferences.onLogout();
                                getVoxeetSDK().closeSocket();
                                getVoxeetSDK().cleanUserSession(id);

                                mEventBus.post(new SdkLogoutErrorEvent(message));
                                solver.resolve(false);
                            }
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                            HttpException.dumpErrorResponse(response);

                            getTwig().e(e);
                            String message = "Logout failed";
                            getTwig().e(message);

                            VoxeetPreferences.onLogout();
                            getVoxeetSDK().closeSocket();
                            getVoxeetSDK().cleanUserSession(id);

                            mEventBus.post(new SdkLogoutErrorEvent(handleError(e)));
                            solver.reject(e);
                        }
                    });
                } else {
                    final String message = "Already logged out";
                    getTwig().e(message);
                    mEventBus.post(new SdkLogoutSuccessEvent("Already logged out"));
                    solver.resolve(true);
                }
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false in case of error
     * reject -> dangerous issue
     *
     * @param conferenceId the mConference id
     * @param offset
     * @return
     */
    @Override
    public Promise<Boolean> replay(final String conferenceId, final long offset) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                isListenerMode = true;
                setListenerMode(false);

                final Call<ReplayConferenceEvent> user = mConferenceObservableProvider.getReplayObservable(conferenceId, new SdkConferenceReplayBody(offset));
                enqueue(user, new HttpCallback<ReplayConferenceEvent>() {
                    @Override
                    public void onSuccess(@NonNull ReplayConferenceEvent object, @NonNull Response<ReplayConferenceEvent> response) {
                        getTwig().i("Success server answer for replaying mConference with id: " +
                                conferenceId + "at offset: " + offset + " given := " + object.getConferenceId());

                        ConferenceInformation information = getConferenceInformation(object.getConferenceId());
                        information.setConferenceType(ConferenceSimpleState.REPLAY);

                        joinVoxeetConference(object.getConferenceId())
                                .then(new PromiseExec<Boolean, Object>() {
                                    @Override
                                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                                        solver.resolve(result);
                                    }
                                })
                                .error(new ErrorPromise() {
                                    @Override
                                    public void onError(Throwable error) {
                                        error.printStackTrace();
                                        solver.resolve(false);
                                    }
                                });
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ReplayConferenceEvent> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Failed to replay mConference");

                        mEventBus.post(new ReplayConferenceErrorEvent(handleError(e)));
                        e.printStackTrace();
                        solver.resolve(false);
                    }
                });

            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false in case of non 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> startRecording() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(mConferenceId, "mConference Id");

                final Call<ResponseBody> user = mConferenceObservableProvider.getStartRecordingObservable(mConferenceId);
                enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        if (response.code() == 200)
                            getTwig().i("IConference recording started ");

                        mEventBus.post(new StartRecordingResultEvent(response.code() == 200));
                        try {
                            Log.d(TAG, "onNext: " + object.string());
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                        solver.resolve(response.code() == 200);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Failed to start mConference recording");

                        mEventBus.post(new StartRecordingResultEvent(handleError(e), false));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false not a 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> stopRecording() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(mConferenceId, "mConference Id");

                final Call<ResponseBody> user = mConferenceObservableProvider.getStopRecordingObservable(mConferenceId);
                enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        if (response.code() == 200)
                            getTwig().i("Conference recording stopped");

                        try {
                            Log.d(TAG, "onNext: " + response.body().string());
                        } catch (IOException e) {
                            e.printStackTrace();
                        }

                        mEventBus.post(new StopRecordingResultEvent(response.code() == 200));
                        solver.resolve(response.code() == 200);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Failed to cancel mConference recording");

                        mEventBus.post(new StopRecordingResultEvent(handleError(e), false));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * Get the current mConference status
     * send events but also managed by the promise
     * <p>
     * resolve -> GetConferenceStatusEvent
     * reject -> error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<GetConferenceStatusEvent> getConferenceStatus(final String conferenceId) {
        return new Promise<>(new PromiseSolver<GetConferenceStatusEvent>() {
            @Override
            public void onCall(@NonNull final Solver<GetConferenceStatusEvent> solver) {
                final Call<GetConferenceStatusEvent> user = mConferenceObservableProvider.getConferenceStatusObservable(conferenceId);
                enqueue(user, new HttpCallback<GetConferenceStatusEvent>() {
                    @Override
                    public void onSuccess(@NonNull GetConferenceStatusEvent object, @NonNull Response<GetConferenceStatusEvent> response) {
                        try {
                            if (null != response && null != object.getConferenceUsers()) {
                                for (DefaultConferenceUser conferenceUser : object.getConferenceUsers()) {
                                    getTwig().e(conferenceUser.getUserId() + " / " + conferenceUser.getStatus());
                                }
                                getTwig().i("Dispatching mConference status result");
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }

                        mEventBus.post(object);
                        solver.resolve(object);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<GetConferenceStatusEvent> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Error while getting mConference history");
                        e.printStackTrace();

                        try {
                            GetConferenceStatusErrorEvent event = new GetConferenceStatusErrorEvent(handleError(e));
                            mEventBus.post(event);
                            throw new PromiseGetConferenceStatusErrorEventException(event, e);
                        } catch (PromiseGetConferenceStatusErrorEventException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }

    /**
     * resolve -> event
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<GetConferenceHistoryEvent> conferenceHistory(final String conferenceId) {
        return new Promise<>(new PromiseSolver<GetConferenceHistoryEvent>() {
            @Override
            public void onCall(@NonNull final Solver<GetConferenceHistoryEvent> solver) {
                final Call<List<HistoryConference>> user = mConferenceObservableProvider.getConferenceHistoryObservable(conferenceId);
                enqueue(user, new HttpCallback<List<HistoryConference>>() {
                    @Override
                    public void onSuccess(@NonNull List<HistoryConference> object, @NonNull Response<List<HistoryConference>> response) {
                        getTwig().i("Dispatching mConference history response");

                        GetConferenceHistoryEvent event = new GetConferenceHistoryEvent(object);
                        mEventBus.post(event);
                        solver.resolve(event);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<List<HistoryConference>> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Error while retrieving mConference history");

                        mEventBus.post(new GetConferenceHistoryErrorEvent(handleError(e)));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> none
     *
     * @return
     */
    @Override
    public Promise<Boolean> switchCamera() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(getMedia(), "media");

                getMedia().switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean isFrontCamera) {
                        getTwig().i("Successfully switched camera");

                        mEventBus.post(new CameraSwitchSuccessEvent(isFrontCamera));
                        solver.resolve(true);
                    }

                    @Override
                    public void onCameraSwitchError(String errorDescription) {
                        getTwig().e("Failed to switch camera " + errorDescription);

                        mEventBus.post(new CameraSwitchErrorEvent(errorDescription));
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<Boolean> subscribe(final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final Call<SubscribeConference> user = mConferenceObservableProvider.getSubscribeObservable(conferenceId);
                enqueue(user, new HttpCallback<SubscribeConference>() {
                    @Override
                    public void onSuccess(@NonNull SubscribeConference object, @NonNull Response<SubscribeConference> response) {
                        getTwig().i("You are no subscribed to this mConference's events");

                        mEventBus.post(new SubscribeConferenceEvent(object));
                        solver.resolve(true);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<SubscribeConference> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Failed to subscribe to this mConference's events");

                        mEventBus.post(new SubscribeConferenceErrorEvent(handleError(e)));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @return a valid promise
     */
    @Override
    public Promise<Boolean> unSubscribe() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Call<ResponseBody> call = mConferenceObservableProvider.getUnSubscribeObservable(mConferenceId);
                enqueue(call, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        getTwig().i("You are no longer subscribed to this mConference's events");

                        mEventBus.post(new UnSubscribeConferenceAnswerEvent(response.code() == 200));
                        solver.resolve(true);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Failed to unsubscribe from this mConference's events");

                        mEventBus.post(new UnSubscribeConferenceAnswerEvent(false, handleError(e)));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return
     */
    @Override
    public Promise<Boolean> subscribeForCall(final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Call<ResponseBody> call = mConferenceObservableProvider.getSubscribeForCallObservable(conferenceId);
                enqueue(call, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        getTwig().i("You are no subscribed to receive events when this conf starts");

                        mEventBus.post(new SubscribeForCallConferenceAnswerEvent(conferenceId));
                        solver.resolve(true);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Failed to subscribe to this mConference's start");

                        mEventBus.post(new SubscribeForCallConferenceErrorEvent(conferenceId, handleError(e)));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return
     */
    @Override
    public Promise<Boolean> unSubscribeFromCall(final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final Call<Response<ResponseBody>> user = mConferenceObservableProvider.getUnSubscribeFromCallObservable(conferenceId);
                enqueue(user, new HttpCallback<Response<ResponseBody>>() {
                    @Override
                    public void onSuccess(@NonNull Response<ResponseBody> object, @NonNull Response<Response<ResponseBody>> response) {
                        getTwig().i("You are no longer subscribed to this mConference's start");

                        mEventBus.post(new UnSubscribeFromConferenceAnswerEvent(conferenceId));
                        solver.resolve(true);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<Response<ResponseBody>> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Failed to unsubscribe from this mConference's start");

                        mEventBus.post(new UnsubscribeFromCallConferenceErrorEvent(conferenceId, handleError(e)));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     *
     * @param conferenceId the mConference id
     * @return a promise to catch
     */
    @Deprecated
    @NonNull
    public Promise<Boolean> joinUsingConferenceId(@NonNull final String conferenceId) {
        return joinVoxeetConference(conferenceId);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(@NonNull SocketConnectEvent event) {
        String conferenceId = getConferenceId();
        ConferenceInformation information = getCurrentConferenceInformation();

        if (null != conferenceId && null != information && ConferenceState.JOINED.equals(information.getConferenceState())) {
            if (isICERestartEnabled()) {
                Log.d(TAG, "onEvent: SocketConnectEvent Joined <3");
                final Call<ResponseBody> user = mConferenceObservableProvider.iceRestart(conferenceId);
                enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        Log.d(TAG, "onEvent: SocketConnectEvent Joined responded <3");
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        Log.d(TAG, "onEvent: SocketConnectEvent Joined error </3");
                        HttpException.dumpErrorResponse(response);
                    }
                });
            } else {
                Log.d(TAG, "onEvent: socket state opened while in conference but no isICERestartEnabled() = true. A reconnect may be longer");
            }
        } else {
            ConferenceState state = ConferenceState.DEFAULT;
            if (null != information) {
                state = information.getConferenceState();
            }
            Log.d(TAG, "onEvent: SocketConnectEvent not rejoined </3 " + state + " " + conferenceId);
        }
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     *
     * @param conferenceId the mConference id
     * @return a promise to catch
     */
    @Deprecated
    @NonNull
    @Override
    public Promise<Boolean> joinVoxeetConference(@NonNull final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                String existingConferenceId = getConferenceId();

                Log.d(TAG, "onCall: " + mInConference + " " + existingConferenceId + " " + conferenceId);
                if (mInConference && (null == existingConferenceId || !existingConferenceId.equals(conferenceId))) {
                    try {
                        throw new InConferenceException();
                    } catch (InConferenceException exception) {
                        solver.reject(exception);
                        return;
                    }
                }

                //remove the timeout, a new one should be started when user will be invited
                removeTimeoutCallbacks();

                Log.d(TAG, "joining " + conferenceId);

                setIsInConference(true);
                mConferenceId = conferenceId;
                final ConferenceInformation information = getCurrentConferenceInformation();
                if (null != information) {
                    information.setConferenceState(ConferenceState.JOINING);
                }

                AudioService service = mInstance.getAudioService();
                if (null != service) {
                    service.enableAec(true);
                    service.enableNoiseSuppressor(true);
                }

                boolean isSuccess = initMedia();

                if (!isSuccess) {
                    Log.d(TAG, "onCall: InitMedia failed... new state = left");
                    try {
                        setIsInConference(false);
                        if (null != information) {
                            information.setConferenceState(ConferenceState.LEFT);
                        }
                        closeMedia();
                        ConferenceJoinedError error = new ConferenceJoinedError(handleError(null));
                        mEventBus.post(error);
                        throw new PromiseConferenceJoinedErrorException(error, null);
                    } catch (PromiseConferenceJoinedErrorException exception) {
                        solver.reject(exception);
                    }
                    return;
                }

                mEventBus.post(new ConferencePreJoinedEvent(getConferenceId(), getAliasId()));

                final Call<ResumeConference> user = mConferenceObservableProvider.joinConference(conferenceId, new DeviceEvent(DeviceType.ANDROID, isListenerMode));

                enqueue(user, new HttpCallback<ResumeConference>() {
                    @Override
                    public void onSuccess(@NonNull ResumeConference object, @NonNull Response<ResumeConference> response) {
                        createOrSetConferenceWithParams(object.getConferenceId(),
                                object.getConferenceAlias());
                        initMedia();

                        ConferenceInformation information = getCurrentConferenceInformation();
                        if (null != information) {
                            information.setConferenceState(ConferenceState.JOINED);
                        }

                        onConferenceResumedInternal(object, solver);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResumeConference> response) {
                        HttpException.dumpErrorResponse(response);

                        try {
                            e.printStackTrace();
                            getTwig().e("Failed to Join mConference with id " + conferenceId);

                            setIsInConference(false);

                            //ConferenceInformation information = getCurrentConferenceInformation();
                            if (null != information) {
                                information.setConferenceState(ConferenceState.LEFT);
                            }
                            closeMedia();
                            ConferenceJoinedError error = new ConferenceJoinedError(handleError(e));
                            mEventBus.post(error);
                            throw new PromiseConferenceJoinedErrorException(error, e);
                        } catch (PromiseConferenceJoinedErrorException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }

    protected void onConferenceResumedInternal(ResumeConference response, Solver<Boolean> solver) {
        mConferenceId = response.getConferenceId();

        ConferenceInformation information = getCurrentConferenceInformation();

        //information should not be null in this case

        DefaultConference conference = information.getConference();

        conference.setConferenceId(response.getConferenceId());
        conference.setConferenceAlias(response.getConferenceId());

        if (response.getConferenceAlias() != null) {
            conference.setConferenceAlias(response.getConferenceAlias());
        }

        participantsToConferenceUsers(information, response.getParticipants());

        setIsInConference(true);

        setAudioRoute(AudioRoute.ROUTE_PHONE);

        getTwig().i("Joined mConference with id " + getConferenceId());

        mEventBus.post(new ConferenceJoinedSuccessEvent(getConferenceId(), getAliasId()));
        solver.resolve(true);
    }

    /**
     * resolve -> the event with the list
     * reject -> any error
     *
     * @return
     */
    @Override
    public Promise<ConferenceUsersInvitedEvent> getInvitedUsers() {
        return new Promise<>(new PromiseSolver<ConferenceUsersInvitedEvent>() {
            @Override
            public void onCall(@NonNull final Solver<ConferenceUsersInvitedEvent> solver) {
                final Call<GetConferenceStatusEvent> user = mConferenceObservableProvider.getConferenceStatusObservable(mConferenceId);
                enqueue(user, new HttpCallback<GetConferenceStatusEvent>() {
                    @Override
                    public void onSuccess(@NonNull GetConferenceStatusEvent object, @NonNull Response<GetConferenceStatusEvent> response) {
                        Iterable<DefaultConferenceUser> users = Iterables.filter(object.getConferenceUsers(), new Predicate<DefaultConferenceUser>() {
                            @Override
                            public boolean apply(DefaultConferenceUser input) {
                                return input.getStatus().equalsIgnoreCase(ConferenceUserStatus.CONNECTING.toString());
                            }
                        });

                        mTwig.i("Successfully retrieved connecting users");

                        ConferenceUsersInvitedEvent event = new ConferenceUsersInvitedEvent(users);
                        mEventBus.post(event);
                        solver.resolve(event);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<GetConferenceStatusEvent> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("Error while retrieving connecting users");

                        mEventBus.post(new ConferenceUsersInvitedEvent(null));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    @Override
    public Promise<Boolean> sendBroadcastMessage(@NonNull final String message) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    if (null == mConferenceId) {
                        throw new NotInConferenceException();
                    }
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                    return;
                }

                final Call<ResponseBody> user = mConferenceObservableProvider.broadcastMessage(mConferenceId, new BroadcastEvent(message));
                enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        getTwig().i("The broadcast message has been sent");

                        mEventBus.post(new SendBroadcastResultEvent(response.code() == 200));
                        solver.resolve(true);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        getTwig().e("The broadcast message has not been sent");

                        mEventBus.post(new SendBroadcastResultEvent(handleError(e), false));
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * This method attempts to leave the current mConference
     * <p>
     * The previous version used a boolean, the new method fingerprint is now void
     */
    @Override
    public Promise<Boolean> leave() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final String conferenceId = mConferenceId;
                final ConferenceInformation infos = getConferenceInformation(conferenceId);
                if (null != infos) infos.setConferenceState(ConferenceState.LEAVING);
                final String conferenceAlias = null != infos ? infos.getConference().getConferenceAlias() : "";

                getTwig().i("Attempting to leave mConference with mConference id " + conferenceId);

                closeMedia();

                final Call<ResponseBody> user = mConferenceObservableProvider.leaveConference(conferenceId);
                enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        removeTimeoutCallbacks();

                        getTwig().i("Conference left successfully");

                        int leftInConference = 0;
                        DefaultConference conference = getConference();
                        if (conference != null) {
                            leftInConference = getConference().getConferenceUsers().size() - 1;
                        }

                        //previously this call would have been in direct conflict with possible parallel create/join
                        //closeMedia();

                        if (null != infos) infos.setConferenceState(ConferenceState.LEFT);
                        if (response.code() == 200 || response.code() == 404) {
                            mEventBus.post(new ConferenceLeftSuccessEvent(conferenceId, conferenceAlias, leftInConference));
                            solver.resolve(true);
                        } else {
                            mEventBus.post(new ConferenceLeftError(conferenceId, conferenceAlias, response.code() + ""));
                            solver.resolve(false);
                        }
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        e.printStackTrace();
                        removeTimeoutCallbacks();

                        getTwig().e("Something went wrong while leaving the mConference");

                        //previously this call would have been in direct conflict with possible parallel create/join
                        //closeMedia();

                        if (null != infos) infos.setConferenceState(ConferenceState.LEFT);
                        mEventBus.post(new ConferenceLeftError(conferenceId, conferenceAlias, handleError(e)));
                        solver.resolve(false);
                    }
                });
            }
        });
    }


    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Event management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceStatsEvent event) {
        ConferenceStats stats = event.getEvent();
        if (stats.getConference_id() != null && stats.getConference_id().equals(getConferenceId())) {
            float mos = stats.getScore(VoxeetPreferences.id());

            mEventBus.post(new QualityIndicators(mos));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUpdatedEvent event) {
        //TODO maybe manage users here too ?
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceived invitation) {
        try {
            onEvent(invitation.getEvent());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceivedEvent invitation) {

        String conferenceId = invitation.getConferenceId();

        if (null == conferenceId && null != invitation.getConference()) {
            conferenceId = invitation.getConference().getConferenceId();
        }

        Log.d(TAG, "onEvent: current mConferenceId " + mConferenceId + " vs " + invitation.getConferenceId() + " vs " + conferenceId);

        if (null != mConferenceId && mConferenceId.equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for our conference, we prevent this");
            return;
        }

        //we update the information for this conference only
        final ConferenceInformation information = getConferenceInformation(conferenceId);

        DefaultConference conference = invitation.getConference();
        boolean is_own_user = invitation.getUserId().equals(VoxeetPreferences.id());
        ConferenceType type = ConferenceType.fromId(conference.getConferenceType());

        if (isLive() && null != getConferenceId() && getConferenceId().equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for the same conference, invalidate the call");
            return;
        }

        if (!is_own_user || ConferenceType.SCHEDULED.equals(type)) {
            InvitationReceivedEvent.UserInviter inviter = invitation.getInviter();
            List<DefaultInvitation> invitations = invitation.getInvitations();

            if (null == information) {
                Log.d(TAG, "onEvent: INVALID INFORMATION FOR THE CONFERENCE");
            }

            if (null != inviter && null != inviter.externalId && null != inviter.userId) {
                safeUserManagementStarted();
                Log.d(TAG, "onEvent: Invitation with inviter " + inviter.userId + " " + inviter.externalId);
                UserInfo info = createUserInfo(inviter.nickName, inviter.externalId, inviter.externalAvatarUrl);

                DefaultConferenceUser inviter_user = new DefaultConferenceUser(inviter.userId,
                        "", info);
                information.getLastInvitationReceived().add(inviter_user);
                information.getUserIdsCached().put(inviter.userId, inviter.externalId);
                safeUserManagementEnded();
            } else {
                Log.d(TAG, "onEvent: Invitation with invalid inviter");
            }

            final String finalConferenceId = conferenceId;
            getConferenceStatus(conference.getConferenceId())
                    .then(new PromiseExec<GetConferenceStatusEvent, Object>() {
                        @Override
                        public void onCall(@Nullable GetConferenceStatusEvent result, @NonNull Solver<Object> internal_solver) {
                            Log.d(TAG, "onSuccess: " + result);

                            //removed the check for live Event since now it is able to have multiple

                            if (result.getConferenceUsers().size() > 0) {
                                Log.d(TAG, "onEvent: users " + result.getConferenceUsers().size());

                                DefaultConferenceUser foundUser = null;
                                String foundExternalId = null;

                                List<DefaultConferenceUser> merged_list = new ArrayList<>();
                                //add every user from the mConference
                                merged_list.addAll(result.getConferenceUsers());
                                //add every use from the invitation
                                merged_list.addAll(getLastInvitationUsers());

                                Iterable<DefaultConferenceUser> list = Iterables.filter(merged_list, new Predicate<DefaultConferenceUser>() {
                                    @Override
                                    public boolean apply(@Nullable DefaultConferenceUser input) {
                                        return !currentUserOrEmpty().equals(input.getUserId());
                                    }
                                });

                                for (DefaultConferenceUser conferenceUser : list) {
                                    String userId = conferenceUser.getUserId();
                                    if (null == foundExternalId) {
                                        String externalId = conferenceUser.getUserInfo().getExternalId();
                                        String cachedExternalId = information.getUserIdsCached().get(userId);
                                        Log.d(TAG, "onEvent: " + userId + " " + externalId + " " + cachedExternalId);

                                        foundUser = conferenceUser;
                                        if (null != userId && (null == externalId || userId.equals(externalId))) {
                                            //should be different than null
                                            externalId = cachedExternalId;
                                            if (null != externalId) {
                                                foundExternalId = cachedExternalId;
                                            }
                                        }
                                    }
                                }

                                if (foundUser != null && null == foundExternalId) {
                                    foundExternalId = foundUser.getUserId();
                                    Log.d(TAG, "externalId is null, setting it to userId");
                                }

                                Map<String, String> infos = new HashMap<>();

                                infos.put(VoxeetIntentFactory.INVITER_ID, foundUser.getUserId());
                                infos.put(VoxeetIntentFactory.INVITER_NAME, foundUser.getUserInfo().getName());
                                infos.put(VoxeetIntentFactory.NOTIF_TYPE, result.getType());
                                infos.put(VoxeetIntentFactory.INVITER_EXTERNAL_ID, foundExternalId);
                                infos.put(VoxeetIntentFactory.INVITER_URL, foundUser.getUserInfo().getAvatarUrl());
                                infos.put(VoxeetIntentFactory.CONF_ID, result.getConferenceId());


                                getTwig().i("Starting default incoming view " + finalConferenceId + " " + result.getConferenceId());

                                Intent intent = VoxeetIntentFactory.buildFrom(mContext, VoxeetPreferences.getDefaultActivity(), infos);
                                if (intent != null)
                                    mContext.startActivity(intent);

                                mEventBus.post(new IncomingCallEvent());
                            }
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(Throwable error) {
                            error.printStackTrace();
                        }
                    });
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(OwnConferenceStartedEvent ownConferenceStartedEvent) {
        if (null != ownConferenceStartedEvent.event().getConferenceInfos()) {
            mConferenceId = ownConferenceStartedEvent.event().getConferenceInfos().getConferenceId();
        }

        DefaultConference conference = getConference();
        getTwig().i("Own mConference started :: " + mConferenceId + " " + conference);

        conference.setConferenceInfos(ownConferenceStartedEvent.event().getConferenceInfos());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final StartVideoAnswerEvent event) {
        Log.d(TAG, "onEvent: start video event " + event.isSuccess() + " " + event.message());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(RenegociationUpdate event) {
        RenegociationEndedEvent renego = event.getEvent();

        Log.d(TAG, "onEvent: Renegociation " + renego.getConferenceId()
                + " " + renego.getType() + " " + renego.getAnswerReceived()
                + " " + renego.getOfferSent());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final OfferCreatedEvent event) {
        Log.d(TAG, "onEvent: OfferCreatedEvent " + event.message());
        try {
            handleAnswer(event.offer())
                    .then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onCall: answer called, result is " + aBoolean + " " + event.offer().getUserId() + " " + VoxeetPreferences.id());
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(Throwable error) {
                            error.printStackTrace();
                        }
                    });
        } catch (MediaEngineException e) {
            e.printStackTrace();
            getTwig().e(e);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserAddedEvent event) {
        if (!mConferenceId.equals(event.getEvent().getConferenceId())) {
            Log.d(TAG, "onEvent: USER ADDED FOR ANOTHER CONFERENCE");
        }

        DefaultConference conference = getOrSetConference(event.getEvent().getConferenceId());
        ConferenceUserAdded content = event.getEvent();

        //if (null == conference) {
        //    Log.d(TAG, "onEvent: ERROR THE CONFERENCE IS NULL, PLEASE CHECK WORKFLOW");
        //    createOrSetConferenceWithParams(event.getEvent().getConferenceId(),
        //            event.getEvent().getConferenceAlias());
        //    conference = getConference();
        //}

        if (null != content) {
            Log.d(TAG, "onEvent: ParticipantAdded " + content + " " + content.getUserId() + " " + conference.getConferenceUsers().size());

            DefaultConferenceUser user = findUserById(content.getUserId());

            if (null == user) {
                user = new DefaultConferenceUser(content.getUserId(), null);
                user.setIsOwner(false);
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));

                conference.getConferenceUsers().add(user);
            }

            if (null == user.getUserInfo()) {
                Log.d(TAG, "onEvent: existing user info are invalid, refreshing");
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));
            }

            user.updateIfNeeded(content.getName(), content.getAvatarUrl());

            ConferenceUserStatus status = ConferenceUserStatus.fromString(content.getStatus());
            if (null != status) {
                user.setConferenceStatus(status);
                user.setStatus(content.getStatus());
            }

            mTwig.i("from Event ConferenceUserAddedEvent, user joined with id := " + user.getUserId() + " " + (user.getUserInfo() != null ? user.getUserInfo().getExternalId() : "no external id"));

            MediaStream stream = null;
            if (hasMedia()) {
                HashMap<String, MediaStream> streams = getMapOfStreams();
                if (streams.containsKey(user.getUserId())) {
                    stream = streams.get(user.getUserId());
                }
            }

            //update the current conference state specificly from the users point of view
            updateConferenceFromUsers();
            mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
        }

        //in a previous version in the v1.0, the createOffer was forced right here
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantUpdatedEvent event) {
        DefaultConferenceUser user = findUserById(event.getUserId());
        final ConferenceUserStatus status = ConferenceUserStatus.valueOf(event.getStatus());

        if (user != null) {
            startTransactionConferenceUser();
            user.setStatus(event.getStatus());
            commitTransactionConferenceUser();
        } else {
            getTwig().i("Not in mConference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
        }

        //update the current conference state specificly from the users point of view
        updateConferenceFromUsers();

        switch (status) {
            case CONNECTING:
                if (null != user && !user.getUserId().equals(VoxeetPreferences.id())) {
                    mTwig.i("Cancelling timeout timer from user connecting");
                    removeTimeoutCallbacks();
                }
                break;
            case LEFT:
            case DECLINE:
                if (hasMedia() && user != null) {
                    getTwig().i("In mConference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
                    getMedia().removePeer(event.getUserId());
                }
                break;
            default:
                getTwig().i("status not managed updated to " + event.getStatus());

        }

        //check if the timeout runnable was not triggered
        boolean timeout_managed = timeoutRunnable != null && timeoutRunnable.isTriggered();

        if (!timeout_managed) {
            switch (status) {
                case DECLINE:
                    if (!event.getUserId().equals(VoxeetPreferences.id())) {
                        getTwig().i("Conference user with id: " + event.getUserId() + " declined the call");

                        mEventBus.post(new ConferenceUserCallDeclinedEvent(event.getConfId(), event.getUserId(), event.getStatus()));
                    }
                    break;
                case LEFT:
                    onUserLeft(event.getUserId());
                    break;
                default:
                    MediaStream stream = null;
                    if (hasMedia() && null != user) {
                        HashMap<String, MediaStream> streams = getMapOfStreams();
                        if (streams.containsKey(user.getUserId())) {
                            stream = streams.get(user.getUserId());
                        }
                    }

                    if (null != user) {
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    }
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantAddedEvent event) {
        DefaultConferenceUser user = event.getUser();

        if (user != null) {
            mTwig.i("Conference user joined with id: " + user.getUserId());

            MediaStream stream = null;
            if (hasMedia()) {
                HashMap<String, MediaStream> streams = getMapOfStreams();
                if (streams.containsKey(user.getUserId())) {
                    stream = streams.get(user.getUserId());
                }
            }

            updateConferenceFromUsers();
            mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceDestroyedPushEvent event) {
        if (null == mConferenceId || mConferenceId.equals(event.getPush().getConferenceId())) {
            closeMedia();

            getTwig().i("Conference has ended");
            mEventBus.post(event.getPush());
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceEndedEvent event) {
        if (null == mConferenceId || mConferenceId.equals(event.getEvent().getConferenceId())) {
            closeMedia();

            getTwig().i("Conference has ended");

            mEventBus.post(event.getEvent());
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final QualityUpdatedEvent event) {
        for (DefaultConferenceUser conferenceUser : event.getEvent().getUser()) {
            DefaultConferenceUser user = findUserById(conferenceUser.getUserId());
            if (user != null) {
                mTwig.i("Quality updated for " + user.getUserId());

                startTransactionConferenceUser();
                user.setQuality(conferenceUser.getQuality());
                commitTransactionConferenceUser();

                mEventBus.post(new ConferenceUserQualityUpdatedEvent(user));
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final RecordingStatusUpdate event) {
        String conferenceId = event.getEvent().getConferenceId();
        Log.d(TAG, "onEvent: eevnt for " + conferenceId);

        DefaultConference conference = getConferenceInformation(conferenceId).getConference();
        RecordingStatusUpdateEvent recordingStatusUpdateEvent = event.getEvent();
        DefaultConferenceUser user = findUserById(recordingStatusUpdateEvent.getUserId());
        if (user != null) {
            RecordingStatus status = RecordingStatus.valueOf(recordingStatusUpdateEvent.getRecordingStatus());
            if (status == RecordingStatus.RECORDING) {
                isRecording = true;
                conference.setStartRecordTimestamp(new Date(recordingStatusUpdateEvent.getTimeStamp()));
                conference.setRecordingStatus(RecordingStatus.RECORDING);
                conference.setRecordingUser(recordingStatusUpdateEvent.getUserId());
                user.setIsRecordingOwner(true);
            } else {
                isRecording = false;
                conference.setStartRecordTimestamp(null);
                conference.setRecordingStatus(RecordingStatus.NOT_RECORDING);
                conference.setRecordingUser(null);
                user.setIsRecordingOwner(false);
            }
        }

        getTwig().i("Conference's recording status has changed to: " + event.getEvent().getRecordingStatus());

        mEventBus.post(recordingStatusUpdateEvent);
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Protected management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    //new DefaultConference()
    //new DefaultConferenceRealm()
    protected abstract DefaultConference createConference();

    //mConference.getConferenceUsers().add(new DefaultConferenceUser(offer.getUserId(), offer.getDevice(), new UserInfo("Sophie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png")));
    protected abstract DefaultConferenceUser createConferenceUser(String userId, String device, UserInfo userInfo);

    protected abstract UserInfo createUserInfo(String userName, String externalId, String avatarUrl);

    protected Twig getTwig() {
        return mTwig;
    }

    protected void setConference(@NonNull DefaultConference conference) {
        mConferenceId = conference.getConferenceId();
        ConferenceInformation information = getCurrentConferenceInformation();
        information.setConference(conference);
    }

    protected void setConferenceAlias(String alias) {
        DefaultConference conference = getConference();
        conference.setConferenceAlias(alias);
        //null ? throw error B)
    }

    public EventBus getEventBus() {
        return mEventBus;
    }

    protected ConferenceListener getConferenceListener() {
        return mListener;
    }

    protected MediaSDK getMedia() {
        return getMediaService().getMedia();
    }

    protected abstract void startTransactionConferenceUser();

    protected abstract void commitTransactionConferenceUser();

    /**
     * Return true if the current implementation is SDK
     * false otherwise
     *
     * @return the state of this implementation
     */
    protected abstract boolean isSDK();

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected Promise<Integer> answer(final String peer, final SdpMessage message) {
        return new Promise<>(new PromiseSolver<Integer>() {
            @Override
            public void onCall(@NonNull final Solver<Integer> solver) {
                try {
                    if (null == mConferenceId) {
                        throw new NotInConferenceException();
                    }
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                    return;
                }

                Log.d("SDKMEDIA", "answer: peer := " + peer + " message := " + message);

                final Call<ResponseBody> user = mConferenceObservableProvider.answerConference(mConferenceId, peer, message.getDescription());
                enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        if (response.code() != 200) {
                            getTwig().i("Offer created for " + peer + " : KO");

                            try {
                                ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(response.code() + "");
                                throw new PromiseParticipantAddedErrorEventException(event);
                            } catch (PromiseParticipantAddedErrorEventException exception) {
                                solver.reject(exception);
                            }
                        } else {
                            getTwig().i("Offer created for " + peer + " : OK");
                            solver.resolve(response.code());
                        }
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        getTwig().e("Offer created for " + peer + " : KO");

                        HttpException.dumpErrorResponse(response);
                        try {
                            ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(handleError(e));
                            throw new PromiseParticipantAddedErrorEventException(event);
                        } catch (PromiseParticipantAddedErrorEventException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }


    /**
     * resolve -> true everything went fine
     * resolve -> false an error occured
     * reject -> media exception
     *
     * @param offer
     * @return a promise with the result of the call
     * @throws MediaEngineException
     */
    protected Promise<Boolean> handleAnswer(final OfferCreated offer) throws MediaEngineException {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                if (!hasMedia()) {
                    try {
                        throw new MediaEngineException("handleAnswer media is null");
                    } catch (MediaEngineException exception) {
                        solver.reject(exception);
                    }
                }


                final String conferenceId = offer.getConferenceId();

                if (!conferenceId.equals(mConferenceId)) {
                    Log.d(TAG, "onCall: CONFERENCE IS NOT THE SAME ! ANSWER SHALL BE DISCARDED");
                }

                SdpDescription description = new SdpDescription(offer.getDescription().getType(), offer.getDescription().getSdp());

                List<SdpCandidate> candidates = new ArrayList<>();
                for (OfferCandidate candidate : offer.getCandidates()) {
                    Log.d(TAG, "onCall: sdpcandidate " + candidate.getMid() + " " + candidate.getmLine());
                    candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                }

                String userId = offer.getUserId();//isSDK() ? offer.getExternalId() : offer.getUserId();
                Log.d(TAG, "handleAnswer: offer := " + offer.getUserId() + " " + offer.getExternalId());
                Log.d("SDKMEDIA", "handleAnswer: " + offer.getUserId() + " " + offer.getExternalId() + " " + description.getSsrc());


                try {
                    //TODO in the callback, check for the conferenceId being the same !
                    getMedia().createAnswerForPeer(userId,
                            description.getSsrc(),
                            description,
                            candidates,
                            offer.isMaster(),
                            new PendingPeerCallback() {
                                @Override
                                public void onMessage(@Nullable SdpMessage message) {
                                    try {
                                        if (null == mConferenceId) {
                                            Log.d(TAG, "onMessage: INVALID CONFERENCE ID WHEN OFFER IS RECEIVED");
                                            mConferenceId = offer.getConferenceId();
                                        }
                                        DefaultConference conference = getConference();

                                        DefaultConferenceUser user = null;

                                        for (DefaultConferenceUser in_conf : conference.getConferenceUsers()) {
                                            if (in_conf.getUserId() != null && in_conf.getUserId().equals(offer.getUserId())) {
                                                user = in_conf;
                                            }
                                        }

                                        UserInfo infos;
                                        if (offer.getUserId().contains("11111"))
                                            infos = createUserInfo("Julie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_1.png");
                                        else if (offer.getUserId().contains("22222"))
                                            infos = createUserInfo("Sophie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png");
                                        else if ((offer.getUserId().contains("33333")))
                                            infos = createUserInfo("Mike", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_3.png");
                                        else
                                            infos = createUserInfo(offer.getName(), offer.getExternalId(), offer.getAvatarUrl());

                                        if (user == null) {
                                            user = createConferenceUser(offer.getUserId(), offer.getDevice(), infos);
                                            conference.getConferenceUsers().add(user);
                                        } else {
                                            user.setUserInfo(infos);
                                        }


                                        setUserPosition(offer.getUserId(), 0, 0.5);

                                        answer(offer.getUserId(), message)
                                                .then(new PromiseExec<Integer, Object>() {
                                                    @Override
                                                    public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                        Log.d(TAG, "onSuccess: " + result);

                                                        solver.resolve(true);
                                                    }
                                                })
                                                .error(new ErrorPromise() {
                                                    @Override
                                                    public void onError(Throwable error) {
                                                        if (error instanceof PromiseParticipantAddedErrorEventException)
                                                            mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                        else
                                                            error.printStackTrace();

                                                        solver.resolve(false);
                                                    }
                                                });
                                    } catch (Exception e) {
                                        Log.d(TAG, "onMessage: unlockPeerOperation" + e.getMessage());
                                        solver.resolve(false);
                                    }
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }
    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected void safeUserManagementStarted() {

    }

    protected void safeUserManagementEnded() {

    }

    protected void safeConferenceManagementStarted() {

    }

    protected void safeConferenceManagementEnded() {

    }

    protected DefaultConferenceUser updateConferenceParticipants(String userId, ConferenceUserStatus status) {
        DefaultConferenceUser user = findUserById(userId);
        DefaultConference conference = getConference();

        if (null != user && null != conference) {
            startTransactionConferenceUser();
            user.setIsOwner(conference.getOwnerProfile() == null || conference.getOwnerProfile().getUserId().equals(user.getProfile().getUserId()))
                    .setConferenceStatus(status)
                    .setQuality(ConferenceQuality.ULTRA_HIGH.StringValue());
            commitTransactionConferenceUser();

            return user;
        }
        return null;
    }

    /**
     * @return false if permission refused -> cancel ! true otherwise
     */
    protected boolean initMedia() {
        Validate.notNull(mContext, "mContext");
        Validate.notNull(VoxeetPreferences.id(), "user id");

        if (!hasMedia()) {
            try {
                if (!Validate.hasMicrophonePermissions(context)) {
                    getTwig().i("the app does not seem to have mic permission, disabling mic");
                    mute(true);
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                }

                getMediaService().createMedia(context, VoxeetPreferences.id(), mediaStreamListener, cameraEventsHandler,
                        isVideoOn(), Validate.hasMicrophonePermissions(context));
                getAudioService().setSpeakerMode(isDefaultOnSpeaker);
                getAudioService().requestAudioFocus();

                if (isDefaultMute) {
                    getMedia().mute();
                } else {
                    getMedia().unMute();
                }
            } catch (MediaEngineException e) {
                //TODO check for excception as a "false"
                getTwig().e(e);
            }
        }

        return true;
    }

    protected void closeMedia() {
        if (hasMedia()) {
            MediaSDK media = getMedia();
            Conference conference = getConference();
            ConferenceInformation information = getCurrentConferenceInformation();

            if(null != information) {
                information.setOwnVideoStarted(false);
            }

            try {
                if (null != conference) {
                    for (DefaultConferenceUser user : conference.getConferenceUsers())
                        if (user != null && user.getUserId() != null)
                            media.removePeer(user.getUserId());
                }
            } catch (Exception ex) {
                mTwig.e(ex);
            }

            getAudioService().abandonAudioFocusRequest();

            getMediaService().releaseMedia();
            getTwig().i("Microphone value restored to true");

            isVideoOn = false;
            mTwig.i("Video value restored to false");

            isListenerMode = false;
            getTwig().i("Listener mode set back to false");
        }

        getTwig().i("Media closed");
        setIsInConference(false);
        mConferenceId = null;
    }

    /**
     * Remove any previous timeout callback
     * <p>
     * To be used when it is mandatory to remove such behaviour
     * for instance, when joining a new mConference
     * <p>
     * It happens since we do not want to manage every edge case :
     * - decline
     * - network error
     * - etc...
     */
    private void removeTimeoutCallbacks() {
        if (timeoutRunnable != null) {
            timeoutRunnable.setCanceled(true);
            handler.removeCallbacks(timeoutRunnable);
        }
    }

    private void sendTimeoutCallbacks() {
        if (mTimeOutTimer != -1) {

            timeoutRunnable = new TimeoutRunnable(this,
                    getTwig(),
                    mEventBus,
                    mTimeOutTimer);

            getTwig().i("scheduling timer to leave the mConference in " + mTimeOutTimer);

            handler.postDelayed(timeoutRunnable, mTimeOutTimer);
        }
    }

    private void onUserLeft(@NonNull String peer) {
        DefaultConferenceUser user = findUserById(peer);
        if (user != null) {
            startTransactionConferenceUser();
            user.setConferenceStatus(ConferenceUserStatus.LEFT);

            if (getConferenceUsers().contains(user)) getConferenceUsers().remove(user);

            getTwig().i("Conference user left with id: " + user.getUserId());
            commitTransactionConferenceUser();

            if (mapOfStreams.containsKey(user.getUserId())) {
                mapOfStreams.remove(user.getUserId());
            }

            if (mapOfScreenShareStreams.containsKey(user.getUserId())) {
                mapOfScreenShareStreams.remove(user.getUserId());
            }

            //refresh the conference state users
            updateConferenceFromUsers();
            mEventBus.post(new ConferenceUserLeftEvent(user));
        } else {
            Log.d(TAG, "run: unknown user in stream left");
        }
    }

    protected VoxeetSdkTemplate getVoxeetSDK() {
        return mSDK;
    }

    public void setStatEnabled(boolean stats) {
        mEnableStats = stats;
    }

    private AudioService getAudioService() {
        return getVoxeetSDK().getAudioService();
    }

    private void createOrSetConferenceWithParams(@NonNull String conferenceId, @Nullable String conferenceAlias) {
        mConferenceId = conferenceId;
        Log.d(TAG, "createOrSetConferenceWithParams: set conference id := " + mConferenceId);
        DefaultConference conference = getCurrentConferenceInformation().getConference();

        conference.setConferenceId(mConferenceId);
        conference.setConferenceAlias(conferenceAlias);
    }

    @Nullable
    public ConferenceInformation getCurrentConferenceInformation() {
        return getConferenceInformation(mConferenceId);
    }

    @Nullable
    private ConferenceInformation getConferenceInformation(@Nullable String conferenceId) {
        if (null == conferenceId) {
            return null;
        }
        return mConferenceInformationHolder.getInformation(conferenceId);
    }

    private void updateConferenceFromUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            ConferenceState state = information.getState();
            Log.d(TAG, "updateConferenceFromUsers: updating conference status from users " + state);
            switch (state) {
                case JOINED:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceState.FIRST_PARTICIPANT);
                    break;
                case FIRST_PARTICIPANT:
                    if (!hasParticipants())
                        information.setConferenceState(ConferenceState.NO_MORE_PARTICIPANT);
                    break;
                case NO_MORE_PARTICIPANT:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceState.FIRST_PARTICIPANT);
                    break;
                default:
                    //nothing
            }
            state = information.getState();
            Log.d(TAG, "updateConferenceFromUsers: new state of conference from users " + state);
        }
    }

    public boolean hasParticipants() {
        List<DefaultConferenceUser> users = getConferenceUsers();

        for (DefaultConferenceUser user : users) {
            ConferenceUserStatus status = user.getConferenceStatus();
            if (null != user.getUserId() && !user.getUserId().equals(VoxeetPreferences.id())
                    && ConferenceUserStatus.ON_AIR.equals(status)) {
                return true;
            }
        }
        return false;
    }

    private <T> void enqueue(Call<T> call, final HttpCallback<T> callback) {
        call.enqueue(new Callback<T>() {
            @Override
            public void onResponse(Call<T> call, Response<T> response) {
                if (response.isSuccessful() && null != response.body()) {
                    callback.onSuccess(response.body(), response);
                } else {
                    callback.onFailure(HttpException.throwResponse(response), response);
                }
            }

            @Override
            public void onFailure(Call<T> call, Throwable t) {
                callback.onFailure(t, null);
            }
        });
    }

    public void setICERestartEnabled(boolean new_state) {
        isICERestartEnabled = new_state;
    }

    public boolean isICERestartEnabled() {
        return isICERestartEnabled;
    }

    @NonNull
    public Context getContext() {
        return mContext;
    }

    @Nullable
    public String getDefaultCamera() {
        return mDefaultCamera;
    }

    public interface HttpCallback<T> {
        void onSuccess(@NonNull T object, @NonNull Response<T> response);

        void onFailure(@NonNull Throwable t, @Nullable Response<T> response);
    }

    private boolean hasMedia() {
        return null != getMedia();
    }

    @NonNull
    private MediaService getMediaService() {
        return mInstance.getMediaService();
    }

    private void participantsToConferenceUsers(@Nullable ConferenceInformation information,
                                               @Nullable List<DefaultParticipant> participants) {
        if (null != information && null != participants) {
            Map<String, String> cache = information.getUserIdsCached();
            for (DefaultParticipant participant : participants) {
                cache.put(participant.getExternalId(), participant.getExternalId());
            }

            DefaultConference conference = information.getConference();
            for (DefaultParticipant participant : participants) {
                updateParticipantToConferenceUsers(conference, participant);
            }
        }
    }

    /**
     * Transform any given participant to conference user
     * <p>
     * Get the DefaultConferenceUser or null
     *
     * @param conference  the valid conference to map the user
     * @param participant the participant to map
     * @return true if the user has not been added
     */
    private boolean updateParticipantToConferenceUsers(@NonNull DefaultConference conference,
                                                       @NonNull DefaultParticipant participant) {
        DefaultConferenceUser user = findUserById(conference, participant.getParticipantId());
        boolean found = null != user;

        if (null == user) {
            user = new DefaultConferenceUser(participant.getParticipantId(),
                    participant.getDeviceType(),
                    new UserInfo(participant.getName(),
                            participant.getExternalId(),
                            participant.getAvatarUrl()));
            conference.getConferenceUsers().add(user);
            found = false;
        } else {
            user.updateIfNeeded(participant.getName(),
                    participant.getAvatarUrl());
        }
        //TODO check for any user modification > name, avatarUrl

        ConferenceUserStatus status = ConferenceUserStatus.fromString(participant.getStatus());
        user.setConferenceStatus(status);

        return !found;
    }

    private void joinLock() {
        try {
            joinLock.lock();
        } catch (Exception e) {

        }
    }

    private void joinUnlock() {
        try {
            if (joinLock.isLocked())
                joinLock.unlock();
        } catch (Exception e) {

        }
    }

}
