package voxeet.com.sdk.core.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.audio.AudioRoute;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.List;

import voxeet.com.sdk.audio.SoundManager;
import voxeet.com.sdk.core.AbstractVoxeetService;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;
import voxeet.com.sdk.events.success.ConferencePreJoinedEvent;
import voxeet.com.sdk.json.ConferenceDestroyedPush;
import voxeet.com.sdk.utils.AudioType;

public class AudioService extends AbstractVoxeetService {

    private static SoundManager sSoundManager;

    /**
     * Instantiate a new Audio Service
     *
     * @param instance the voxeet parent instance
     */
    public AudioService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        AudioService.preInitSounds(instance.getApplicationContext());
        instance.getEventBus().register(this);
    }

    @Nullable
    public static SoundManager getSoundManager() {
        return sSoundManager;
    }

    public static boolean preInitSounds(@NonNull Context applicationContext) {
        if(null == sSoundManager) {
            sSoundManager = new SoundManager(applicationContext);
            return true;
        }

        return false;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferencePreJoinedEvent event) {
        sSoundManager.onConferencePreJoinedEvent();
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPush event) {
        sSoundManager.onConferenceDestroyedPush();
    }


    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return sSoundManager.getAvailableRoutes();
    }

    public AudioRoute currentRoute() {
        return sSoundManager.currentRoute();
    }

    public boolean setAudioRoute(@NonNull AudioRoute route) {
        return sSoundManager.setAudioRoute(route);
    }

    public void setSpeakerMode(boolean isSpeaker) {
        sSoundManager.setSpeakerMode(isSpeaker);
    }

    public boolean isSpeakerOn() {
        return sSoundManager.isSpeakerOn();
    }

    public void acquireLocks() {
        sSoundManager.acquireLocks();
    }

    public void releaseLocks() {
        sSoundManager.releaseLocks();
    }

    public void setInVoiceCallSoundType() {
        sSoundManager.setInVoiceCallSoundType();
    }

    public void resetDefaultSoundType() {
        sSoundManager.resetDefaultSoundType();
    }

    public void abandonAudioFocusRequest() {
        sSoundManager.abandonAudioFocusRequest();
    }

    public void requestAudioFocus() {
        sSoundManager.requestAudioFocus();
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return sSoundManager.setSound(type, assetName);
    }

    public void playSoundType(@NonNull AudioType type) {
        sSoundManager.playSoundType(type);
    }

    public void stopSoundType(@NonNull AudioType audioType) {
        sSoundManager.stopSoundType(audioType);
    }

    public void stop() {
        sSoundManager.stop();
    }
}
