package voxeet.com.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaSDK;
import com.voxeet.android.media.stats.LocalStats;

import java.util.HashMap;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

import voxeet.com.sdk.core.AbstractVoxeetService;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;
import voxeet.com.sdk.core.services.localstats.LocalStatsHandler;
import voxeet.com.sdk.core.services.localstats.LocalStatsUserInfo;
import voxeet.com.sdk.models.impl.DefaultConference;
import voxeet.com.sdk.models.impl.DefaultConferenceUser;

public class LocalStatsService extends AbstractVoxeetService implements Runnable {
    public final static int NEXT_LOOP_FETCH_MS = 500;

    private ReentrantLock mLock;
    private VoxeetSdkTemplate mInstance;

    private LocalStatsHandler mHandler;
    private HashMap<String, HashMap<String, LocalStatsUserInfo>> mConferencesHolder;

    public LocalStatsService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        mConferencesHolder = new HashMap<>();
        mHandler = null;
        mLock = new ReentrantLock();
        mInstance = instance;
    }

    @Nullable
    public LocalStats getLocalStats(@NonNull String peer) {
        MediaSDK media = mInstance.getMediaService().getMedia();
        if (null != media) return media.getLocalStats(peer);
        return null;
    }


    public boolean startAutoFetch() {
        if (null == mHandler) {
            mHandler = new LocalStatsHandler(this, NEXT_LOOP_FETCH_MS);
            mHandler.startAutoFetch();
            return true;
        } else {
            return false;
        }
    }

    public boolean stopAutoFetch() {
        if (null != mHandler) {
            mHandler.stopAutoFetch();
            mHandler = null;
            return true;
        }
        return false;
    }

    @Override
    public void run() {
        //List<LocalStats> stats = new ArrayList<>();
        DefaultConference conference = mInstance.getConferenceService().getConference();

        if (null != conference) {
            List<DefaultConferenceUser> users = conference.getConferenceUsers();
            for (int i = 0; i < users.size(); i++) {
                DefaultConferenceUser user = users.get(i);
                LocalStats stat = getLocalStats(user.getUserId());
                getUserInfo(conference.getConferenceId(), user.getUserId()).addFromLocalStats(stat);
            }
        } else {
            //no conference ?
        }
    }

    /**
     * Retrieve the in-memory holder of the conference information specifically designed for the stats
     *
     * @param conferenceId the conferenceId - not the alias
     * @return a non null HashMap instance
     */
    @NonNull
    private HashMap<String, LocalStatsUserInfo> getUserStatsHolder(@NonNull String conferenceId) {
        HashMap<String, LocalStatsUserInfo> holder = mConferencesHolder.get(conferenceId);

        if (null == holder) {
            holder = new HashMap<>();
            mConferencesHolder.put(conferenceId, holder);
        }
        return holder;
    }

    /**
     * Retrieve the in-memory holder of the user specifically designed for the user
     *
     * @param conferenceId the conferenceId - not the alias
     * @param userId       the userId - not the externalId
     * @return a valid user info holder
     */
    public LocalStatsUserInfo getUserInfo(@NonNull String conferenceId, @NonNull String userId) {
        HashMap<String, LocalStatsUserInfo> holder = getUserStatsHolder(conferenceId);
        LocalStatsUserInfo info = holder.get(userId);

        if (null == info) {
            info = new LocalStatsUserInfo(userId);
            holder.put(userId, info);
        }
        return info;
    }
}
