package voxeet.com.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import retrofit2.Response;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.schedulers.Schedulers;
import voxeet.com.sdk.core.AbstractVoxeetService;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.network.ISdkVideoPresentationRService;
import voxeet.com.sdk.core.services.abstracts.AbstractPresentationService;
import voxeet.com.sdk.events.success.VideoPresentationPausedEvent;
import voxeet.com.sdk.events.success.VideoPresentationPlayEvent;
import voxeet.com.sdk.events.success.VideoPresentationSeekEvent;
import voxeet.com.sdk.events.success.VideoPresentationStartedEvent;
import voxeet.com.sdk.events.success.VideoPresentationStoppedEvent;
import voxeet.com.sdk.json.VideoPresentationPaused;
import voxeet.com.sdk.json.VideoPresentationPlay;
import voxeet.com.sdk.json.VideoPresentationSeek;
import voxeet.com.sdk.json.VideoPresentationStarted;
import voxeet.com.sdk.json.VideoPresentationStopped;
import voxeet.com.sdk.models.FilePresentationConverted;

/**
 *
 */
public class SDKVideoPresentationService extends AbstractPresentationService<ISdkVideoPresentationRService> {

    private List<Solver<VideoPresentationStarted>> mCacheStartedSolvers;
    private List<Solver<VideoPresentationStopped>> mCacheStoppedSolvers;
    private List<Solver<VideoPresentationPaused>> mCachePausedSolvers;
    private List<Solver<VideoPresentationPlay>> mCachePlaySolvers;
    private List<Solver<VideoPresentationSeek>> mCacheSeekSolvers;

    /**
     * Instantiates a new User service.
     *
     * @param instance the parent instance
     */
    public SDKVideoPresentationService(VoxeetSdkTemplate instance) {
        super(instance, ISdkVideoPresentationRService.class);

        mCachePausedSolvers = new ArrayList<>();
        mCacheStartedSolvers = new ArrayList<>();
        mCacheStoppedSolvers = new ArrayList<>();
        mCachePausedSolvers = new ArrayList<>();
        mCachePlaySolvers = new ArrayList<>();
        mCacheSeekSolvers = new ArrayList<>();
        registerEventBus();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationStartedEvent event) {
        tryUnlock(event.getEvent(), mCacheStartedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationStoppedEvent event) {
        tryUnlock(event.getEvent(), mCacheStoppedSolvers);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationPausedEvent event) {
        tryUnlock(event.getEvent(), mCachePausedSolvers);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationPlayEvent event) {
        tryUnlock(event.getEvent(), mCachePlaySolvers);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationSeekEvent event) {
        tryUnlock(event.getEvent(), mCacheSeekSolvers);
    }

    public Promise<VideoPresentationStarted> startVideoPresentation(final String url) {
        return new Promise<>(new PromiseSolver<VideoPresentationStarted>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationStarted> solver) {
                ISdkVideoPresentationRService.VideoPresentationUrl holder = new ISdkVideoPresentationRService.VideoPresentationUrl(url);
                consumeInternalCall(solver, mCacheStartedSolvers,
                        internalCall(getService().startVideoPresentation(getConferenceId(), holder)));
            }
        });
    }

    public Promise<VideoPresentationStopped> stopVideoPresentation() {
        return new Promise<>(new PromiseSolver<VideoPresentationStopped>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationStopped> solver) {

                consumeInternalCall(solver, mCacheStoppedSolvers,
                        internalCall(getService().stopVideoPresentation(getConferenceId())));
            }
        });
    }


    public Promise<VideoPresentationPlay> playVideoPresentation() {
        return new Promise<>(new PromiseSolver<VideoPresentationPlay>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationPlay> solver) {

                consumeInternalCall(solver, mCachePlaySolvers,
                        internalCall(getService().playVideoPresentation(getConferenceId())));
            }
        });
    }

    public Promise<VideoPresentationPaused> pauseVideoPresentation(final long timestamp) {
        return new Promise<>(new PromiseSolver<VideoPresentationPaused>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationPaused> solver) {
                ISdkVideoPresentationRService.VideoPresentationSeek body_sent = new ISdkVideoPresentationRService.VideoPresentationSeek(timestamp);

                consumeInternalCall(solver, mCachePausedSolvers,
                        internalCall(getService().pauseVideoPresentation(getConferenceId(), body_sent)));
            }
        });
    }

    public Promise<VideoPresentationSeek> seekVideoPresentation(final long timestamp) {
        return new Promise<>(new PromiseSolver<VideoPresentationSeek>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationSeek> solver) {

                ISdkVideoPresentationRService.VideoPresentationSeek body_sent = new ISdkVideoPresentationRService.VideoPresentationSeek(timestamp);

                consumeInternalCall(solver, mCacheSeekSolvers,
                        internalCall(getService().seekVideoPresentation(getConferenceId(), body_sent)));
            }
        });
    }

}