package voxeet.com.sdk.core;

import android.annotation.SuppressLint;
import android.app.Application;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.firebase.iid.FirebaseInstanceId;

import java.io.IOException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.Authenticator;
import okhttp3.ConnectionPool;
import okhttp3.Cookie;
import okhttp3.CookieJar;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.Route;
import okhttp3.logging.HttpLoggingInterceptor;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Retrofit;
import voxeet.com.sdk.converter.JacksonConverterFactory;
import voxeet.com.sdk.core.http.RefreshTokenBody;
import voxeet.com.sdk.core.network.IVoxeetRService;
import voxeet.com.sdk.core.preferences.VoxeetPreferences;
import voxeet.com.sdk.json.GrandTypeEvent;
import voxeet.com.sdk.json.UserInfo;
import voxeet.com.sdk.models.TokenResponse;
import voxeet.com.sdk.models.UserTokenResponse;
import voxeet.com.sdk.networking.DeviceType;
import voxeet.com.sdk.utils.Auth64;

/**
 * VoxeetHttp manages every network call made by the SDK
 * <p>
 * It will identify, re-identify and make the calls
 */
public class VoxeetHttp {

    private static final String TAG = VoxeetHttp.class.getSimpleName();
    private static final String HEADER_NAME_AUTHORIZATION = "Authorization";
    private static final String HEADER_NAME_XTOKEN = "X-Token";
    private final VoxeetSdkTemplate sdk;

    /**
     * The Service.
     */
    private @NonNull
    IVoxeetRService service;

    /**
     * The current application id obtained from Voxeet Developer Portal
     * Note : if non null, password must be also non null
     */
    @Nullable
    private final String appId;
    /**
     * The current application id obtained from Voxeet Developer Portal
     * Note : if non null, appId must be also non null
     */
    @Nullable
    private final String password;


    /**
     * The current tokenAccess set by the SDK initialization
     * Note : if non null, tokenRefresh must be also non null
     */
    @Nullable
    private final String tokenAccess;
    /**
     * The current tokenRefresh callback set by the SDK initialization
     * Note : if non null, tokenAccess must be also non null
     * Callbable means that if any network call is done, it must be done synchronically
     * <p>
     * it is made sure by VoxeetHttp that the call will not block the UI interface
     */
    @Nullable
    private final Callable<String> tokenRefresh;

    /**
     * The Token response.
     */
    @Nullable
    private TokenResponse tokenResponse;

    private Application appContext;

    private String serverUrl;

    private String serverPort;

    /**
     * The Retrofit.
     */
    public Retrofit retrofit;

    private OkHttpClient client;

    private TokenResponseProvider tokenResponseProvider;

    private VoxeetServiceListener listener;

    private boolean debug = false;

    private final HashMap<String, List<Cookie>> cookieStore = new HashMap<>();
    private UserTokenResponse mUserTokenResponse;
    private UserInfo mUserInfo;
    private String TOKEN_RESET = "";
    private boolean isAuthenticating = false;
    private OkHttpClient clientIdentify;
    private Retrofit retrofitIdentify;

    /**
     * Instantiates a new Voxeet http.
     *
     * @param appContext   the Application Context
     * @param sdk          the current sdk instance
     * @param appId        the app id
     * @param password     the password
     * @param listener     the listener
     * @param tokenAccess  the token access
     * @param tokenRefresh the token refresher
     */
    private VoxeetHttp(Application appContext, VoxeetSdkTemplate sdk,
                       String serverUrl, String serverPort,
                       String appId, String password,
                       VoxeetServiceListener listener,
                       final String tokenAccess, final Callable<String> tokenRefresh,
                       boolean debug) {
        this.sdk = sdk;
        this.tokenResponse = null;
        this.serverUrl = serverUrl;
        this.serverPort = serverPort;
        this.appContext = appContext;

        this.tokenAccess = tokenAccess;
        this.tokenRefresh = tokenRefresh;

        this.debug = debug;

        this.listener = listener;

        this.appId = appId;

        this.password = password;

        this.initClient();

        this.initRetrofit();

        this.initService();

        //set 1 provider according to the 2 different cases which be made available
        if (null != appId && null != password) {
            Log.d(TAG, "VoxeetHttp: initializing using appId/password");
            tokenResponseProvider = new TokenResponseProvider() {
                @Nullable
                @Override
                public TokenResponse refreshTokenResponse() {
                    Log.d(VoxeetHttp.TAG, "refreshTokenResponse: calling refreshing token");
                    TokenResponse previous = tokenResponse;
                    TokenResponse new_token_response = null;
                    Call<TokenResponse> token = null;

                    String tokenizedCredentials = Auth64.serialize(VoxeetHttp.this.appId, VoxeetHttp.this.password);


                    try {
                        if (null != previous && null != previous.getRefreshToken()) {
                            token = service.refreshToken(tokenizedCredentials, new RefreshTokenBody(previous.getRefreshToken()));
                        }
                        if (null != token) {
                            new_token_response = token.execute().body();
                        }
                    } catch (Exception e) {
                        Log.d(TAG, "authenticate: error on refresh token " + e.getMessage());
                        e.printStackTrace();
                    }

                    if (null != new_token_response && null != new_token_response.getAccessToken()) {
                        return new_token_response;
                    }

                    try {
                        token = service.getToken(tokenizedCredentials, new GrandTypeEvent("client_credentials"));
                        return token.execute().body();
                    } catch (Exception e) {
                        Log.d(VoxeetHttp.TAG, "refreshTokenResponse: second token attempt error");

                        e.printStackTrace();
                        return null;
                    }
                }
            };
        } else if (null != tokenAccess && null != tokenRefresh) {
            Log.d(TAG, "VoxeetHttp: initializing using access/refresh");
            //create a temporary tokenResponse object which will hold the values
            tokenResponse = new TokenResponse(tokenAccess, null);

            //in such case, the user's servers are here to manage oauth
            tokenResponseProvider = new TokenResponseProvider() {
                @Nullable
                @Override
                public TokenResponse refreshTokenResponse() {
                    try {
                        String tokenAccess = tokenRefresh.call();

                        return new TokenResponse(tokenAccess, null);
                    } catch (Exception e) {
                        e.printStackTrace();
                        return null;
                    }
                }
            };
        }
    }

    /**
     * Overrode retrofit client.
     */
    private void initClient() {
        HttpLoggingInterceptor interceptor = new HttpLoggingInterceptor();
        interceptor.setLevel(HttpLoggingInterceptor.Level.BODY);


        CookieJar cookieJar = new CookieJar() {

            @Override
            public void saveFromResponse(HttpUrl url, List<Cookie> cookies) {
                String base = url.scheme() + "://" + url.host() + ":" + url.port();

                cookieStore.put(base, Lists.newArrayList(Iterables.filter(cookies, new Predicate<Cookie>() {
                    @Override
                    public boolean apply(Cookie input) {
                        return input != null && !input.value().equalsIgnoreCase("deleted") && new Date().before(new Date(input.expiresAt()));
                    }
                })));
            }

            @Override
            public List<Cookie> loadForRequest(HttpUrl url) {
                String base = url.scheme() + "://" + url.host() + ":" + url.port();

                List<Cookie> cookies = cookieStore.get(base);

                return cookies != null ? cookies : new ArrayList<Cookie>();
            }
        };

        OkHttpClient.Builder builderIdentify = new OkHttpClient.Builder()
                .cookieJar(cookieJar)

                .addNetworkInterceptor(new Interceptor() {
                    @Override
                    public Response intercept(Chain chain) throws IOException {
                        Request.Builder builder = chain.request()
                                .newBuilder();

                        if (null != tokenResponse) {
                            builder.removeHeader(HEADER_NAME_AUTHORIZATION)
                                    .addHeader(HEADER_NAME_AUTHORIZATION, Auth64.serialize(appId, password))
                                    .addHeader(HEADER_NAME_AUTHORIZATION, "bearer " + TOKEN_RESET + tokenResponse.getAccessToken());
                        }

                        return chain.proceed(builder.build());
                    }
                })
                .authenticator(new Authenticator() {
                    @Override
                    public Request authenticate(Route route, Response response) throws IOException {
                        return reauthenticate(route, response);
                    }
                })
                .connectionPool(new ConnectionPool(5, 45, TimeUnit.SECONDS))
                .retryOnConnectionFailure(true)
                .readTimeout(5, TimeUnit.SECONDS)
                .writeTimeout(5, TimeUnit.SECONDS);

        OkHttpClient.Builder builder = new OkHttpClient.Builder()
                .cookieJar(cookieJar)
                .addNetworkInterceptor(new Interceptor() {
                    @Override
                    public Response intercept(Chain chain) throws IOException {
                        Log.d(TAG, "intercept: ----------------------------");
                        try {
                            return execute(chain);
                        } catch (Exception exception) {
                            Log.d(TAG, "intercept: executing failed time");
                            if (exception instanceof IOException) throw exception;
                        }

                        Request.Builder builder = chain.request()
                                .newBuilder();
                        return chain.proceed(builder.build());
                    }
                })
                .authenticator(new Authenticator() {
                    @Override
                    public Request authenticate(Route route, Response response) throws IOException {
                        Log.d(TAG, "authenticate: reauthenticating");
                        return reauthenticate(route, response);
                    }
                })
                .connectionPool(new ConnectionPool(5, 45, TimeUnit.SECONDS))
                .retryOnConnectionFailure(true)
                .readTimeout(5, TimeUnit.SECONDS)
                .writeTimeout(5, TimeUnit.SECONDS);

        if (this.debug) {
            builder = builder.hostnameVerifier(new HostnameVerifier() {
                @Override
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            });
            builder = builder.addInterceptor(interceptor);
            builderIdentify = builderIdentify.addInterceptor(interceptor);
        }

        client = builder.build();
        clientIdentify = builderIdentify.build();
    }

    private void initRetrofit() {
        retrofit = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltServerUrl())
                .client(client)
                .build();


        retrofitIdentify = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltServerUrl())
                .client(clientIdentify)
                .build();
    }

    private void initService() {
        service = retrofitIdentify.create(IVoxeetRService.class);
    }

    /**
     * Identify. Similar to a login
     *
     * @param userInfo the user info
     */
    @SuppressLint("HardwareIds")
    void identify(final UserInfo userInfo) {
        identifyChain(userInfo).then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onCall: identify finished " + result);
            }
        }).error(new ErrorPromise() {
            @Override
            public void onError(@NonNull Throwable error) {
                error.printStackTrace();
            }
        });
    }

    Promise<Boolean> identifyChain(final UserInfo userInfo) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull Solver<Boolean> final_solver) {
                mUserInfo = userInfo;

                VoxeetPreferences.setExternalUserInfo(mUserInfo);

                userInfo.setDeviceIdentifier(Settings.Secure.getString(appContext.getContentResolver(), Settings.Secure.ANDROID_ID));

                if (FirebaseController.getInstance().isTokenUploadAllowed()) {
                    String token = FirebaseController.getInstance().getToken();

                    if (token != null) {
                        sdk.getTwig().i("FirebaseInstanceId.getInstance().getAccessToken(): " + FirebaseInstanceId.getInstance().getToken());
                        userInfo.setDevicePushToken(token);
                    } else {
                        sdk.getTwig().e("FirebaseInstanceId.getInstance().getAccessToken() returned an IllegalStateException, you have an issue with your project configuration (google-services.json for instance)");
                    }

                } else
                    sdk.getTwig().i("FirebaseApp is not initialized. Make sure to call FirebaseApp.initializeApp(Context) " +
                            "before initializing the VoxeetSDK if you are planning on using FCM.");

                userInfo.setDeviceType(DeviceType.ANDROID);

                retrieveTokenResponse().then(new PromiseExec<TokenResponse, UserTokenResponse>() {
                    @Override
                    public void onCall(@Nullable TokenResponse result, @NonNull Solver<UserTokenResponse> solver) {
                        sdk.getTwig().i("validated access with the servers ; authenticating user... " + result);

                        tokenResponse = result;
                        solver.resolve(retrieveUserTokenResponse(userInfo));
                    }
                }).then(new PromiseExec<UserTokenResponse, Object>() {
                    @Override
                    public void onCall(@Nullable UserTokenResponse result, @NonNull Solver<Object> solver) {
                        mUserTokenResponse = result;
                        sdk.getTwig().i("Successful login with id " + result.getId());

                        VoxeetPreferences.setId(result.getId());
                        VoxeetPreferences.saveLoginCookie(result.getUserToken());

                        listener.onIdentifySuccess(result, tokenResponse);

                        List<Cookie> cookies = getCookies(getBuiltServerUrl());
                        if (cookies != null && cookies.size() > 0)
                            VoxeetPreferences.saveLoginCookie(cookies.get(0).toString());
                        final_solver.resolve(true);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        error.printStackTrace();
                        if (error instanceof UnknownHostException) {
                            listener.onNetworkError((UnknownHostException) error);
                        } else {
                            listener.onIdentifyError(error.getMessage());
                        }
                        final_solver.reject(error);
                    }
                });
            }
        });
    }

    /**
     * Gets cookies.
     *
     * @param url the url
     * @return the cookies
     */
    private List<Cookie> getCookies(String url) {
        for (Object o : cookieStore.entrySet()) {
            Map.Entry pair = (Map.Entry) o;
            if (((String) pair.getKey()).contains(url)) {
                return cookieStore.get(pair.getKey());
            }
        }
        return null;
    }

    /**
     * Create cookie.
     *
     * @param value the value
     * @return the string
     */
    private String createCookie(String value) {
        String url = getBuiltServerUrl();

        Cookie cookie = Cookie.parse(HttpUrl.parse(url), value);
        if (cookie != null && cookie.toString().length() > 0) {
            cookieStore.put(url, Collections.singletonList(cookie));
            return cookie.value();
        }
        return null;
    }

    /**
     * Gets client.
     *
     * @return the client
     */

    public OkHttpClient getClient() {
        return client;
    }

    /**
     * Retrieve cookie.
     *
     * @return the string
     */
    public String retrieveCookie() {
        String cookie;
        if ((cookie = VoxeetPreferences.getLoginCookie()) != null && cookie.length() > 0) {
            cookie = createCookie(cookie);
        } else {
            List<Cookie> cookies = getCookies(getBuiltServerUrl());
            if (cookies != null && cookies.size() > 0) {
                cookie = cookies.get(0).value();
                VoxeetPreferences.saveLoginCookie(cookies.get(0).toString());
            }
        }
        return cookie;
    }

    Retrofit getRetrofit() {
        return retrofit;
    }

    String getToken() {
        return mUserTokenResponse != null ? mUserTokenResponse.getUserToken() : null;
    }

    String getJwtToken() {
        return mUserTokenResponse != null ? mUserTokenResponse.getJwtUserToken() : null;
    }

    /**
     * The interface Voxeet service listener.
     */
    public interface VoxeetServiceListener {

        /**
         * On identify success.
         *
         * @param response      the response
         * @param tokenResponse the token response
         */
        void onIdentifySuccess(UserTokenResponse response, TokenResponse tokenResponse);

        /**
         * On identify error.
         *
         * @param error the error
         */
        void onIdentifyError(String error);


        /**
         * On network error
         *
         * @param error the error
         */
        void onNetworkError(UnknownHostException error);
    }

    String getBuiltServerUrl() {

        String port = serverPort;
        if (!port.startsWith(":")) port = ":" + port;
        return serverUrl + port;
    }

    /**
     * Retrieve a valid TokenResponse from the servers
     * <p>
     * It will use on of the two available method available :
     * - resolve the injected tokenAccess
     * - resolve a token response obtained from Voxeet servers
     * <p>
     * Note : don't forget to call .then().error()
     *
     * @return a promise to resolve and manage the error
     */
    @NonNull
    private Promise<TokenResponse> retrieveTokenResponse() {
        return new Promise<>(new PromiseSolver<TokenResponse>() {
            @Override
            public void onCall(@NonNull final Solver<TokenResponse> solver) {
                if (null != tokenAccess && null != tokenRefresh) {
                    solver.resolve(new TokenResponse(tokenAccess, null));
                } else if (null != appId && null != password) {
                    Call<TokenResponse> token = service.getToken(Auth64.serialize(appId, password), new GrandTypeEvent("client_credentials"));
                    token.enqueue(new Callback<TokenResponse>() {
                        @Override
                        public void onResponse(Call<TokenResponse> call, retrofit2.Response<TokenResponse> response) {
                            Log.d(TAG, "onResponse: " + response.body());
                            solver.resolve(response.body());
                        }

                        @Override
                        public void onFailure(Call<TokenResponse> call, Throwable t) {
                            Log.d(TAG, "onFailure: " + t);
                            t.printStackTrace();
                            solver.reject(t);
                        }
                    });
                } else {
                    try {
                        throw new Exception("Invalid mode, issue with SDK compilation?");
                    } catch (Exception e) {
                        solver.reject(e);
                    }
                }
            }
        });
    }


    /**
     * Retrieve the current user token information
     * Note that :
     * - The current SDK must be logged in when called (call retrieveTokenResponse or response from 403)
     * - it will resolve a valid object
     * - it will reject in case of network error
     * <p>
     * Note : don't forget to call .then().error()
     *
     * @param userInfo valid user info to identify the user
     * @return a promise to resolve and manage the error
     */
    @NonNull
    private Promise<UserTokenResponse> retrieveUserTokenResponse(@NonNull final UserInfo userInfo) {
        return new Promise<>(new PromiseSolver<UserTokenResponse>() {
            @Override
            public void onCall(@NonNull final Solver<UserTokenResponse> solver) {
                final Call<UserTokenResponse> user = service.identify(userInfo);
                user.enqueue(new Callback<UserTokenResponse>() {
                    @Override
                    public void onResponse(Call<UserTokenResponse> call, retrofit2.Response<UserTokenResponse> response) {
                        solver.resolve(response.body());
                    }

                    @Override
                    public void onFailure(Call<UserTokenResponse> call, Throwable t) {
                        solver.reject(t);
                    }
                });
            }
        });
    }

    public final static class Builder {

        private Application _application;
        private VoxeetSdkTemplate _sdk;
        private String _server_url;
        private String _server_port;
        private String _app_id;
        private String _password;
        private String _tokenAccess;
        private Callable<String> _tokenRefresh;
        private VoxeetServiceListener _listener;
        private boolean _debug;

        public Builder() {

        }

        public Builder setApplication(Application application) {
            this._application = application;
            return this;
        }

        public Builder setVoxeetSDK(VoxeetSdkTemplate sdk) {
            this._sdk = sdk;
            return this;
        }

        public Builder setServerUrl(String serverUrl) {
            this._server_url = serverUrl;
            return this;
        }

        public Builder setServerPort(String serverPort) {
            this._server_port = serverPort;
            return this;
        }

        public Builder setAppId(String appId) {
            this._app_id = appId;
            return this;
        }

        public Builder setPassword(String password) {
            this._password = password;
            return this;
        }

        public Builder setVoxeetServiceListener(VoxeetServiceListener listener) {
            this._listener = listener;
            return this;
        }

        public Builder setDebug(boolean debug) {
            this._debug = debug;
            return this;
        }

        public Builder setTokenAccess(String tokenAccess) {
            this._tokenAccess = tokenAccess;
            return this;
        }

        public Builder setTokenRefresh(Callable<String> tokenRefresh) {
            this._tokenRefresh = tokenRefresh;
            return this;
        }

        public VoxeetHttp build() {
            return new VoxeetHttp(this._application,
                    _sdk,
                    _server_url,
                    _server_port,
                    _app_id,
                    _password,
                    _listener,
                    _tokenAccess,
                    _tokenRefresh,
                    _debug);
        }
    }

    /**
     * Provide a valid TokenAccess holder object
     */
    private interface TokenResponseProvider {
        @Nullable
        TokenResponse refreshTokenResponse();
    }

    public void FORCE_TOKEN_RESET() {
        TOKEN_RESET = "resetting";
    }

    @Nullable
    private Response execute(Interceptor.Chain chain) throws IOException {
        try {
            Request.Builder builder = chain.request()
                    .newBuilder();

            if (null != tokenResponse) {
                builder//.removeHeader(HEADER_NAME_AUTHORIZATION)
                        .addHeader(HEADER_NAME_AUTHORIZATION, "bearer " + TOKEN_RESET + tokenResponse.getAccessToken());
            }

            if (null != mUserTokenResponse) {
                builder.removeHeader(HEADER_NAME_XTOKEN)
                        .addHeader(HEADER_NAME_XTOKEN, mUserTokenResponse.getJwtUserToken());
            }

            return chain.proceed(builder.build());
        } catch (Exception e) {
            Log.d(TAG, "execute: ISSUE WHILE EXECUTING " + chain.call().request().toString());
            if (e instanceof IOException) throw e;
            e.printStackTrace();
            return null;
        }
    }

    private Request reauthenticate(Route route, Response response) {
        isAuthenticating = true;

        TOKEN_RESET = "";

        Request.Builder builder = response.request().newBuilder();

        TokenResponse tokenResponse = tokenResponseProvider.refreshTokenResponse();

        if (null != tokenResponse) {
            VoxeetHttp.this.tokenResponse = tokenResponse;
        } else {
            isAuthenticating = false;
            throw new IllegalStateException("Invalid credentials");
        }

        if (null != mUserInfo && null == mUserTokenResponse) {
            try {
                UserTokenResponse userTokenResponse = service.identify(mUserInfo).execute().body();
                if (null != userTokenResponse) {
                    mUserTokenResponse = userTokenResponse;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if (null != tokenResponse) {
            builder.removeHeader(HEADER_NAME_AUTHORIZATION)
                    .addHeader(HEADER_NAME_AUTHORIZATION, Auth64.serialize(appId, password))
                    .addHeader(HEADER_NAME_AUTHORIZATION, "bearer " + tokenResponse.getAccessToken());
        }

        if (null != mUserTokenResponse) {
            builder.removeHeader(HEADER_NAME_XTOKEN)
                    .addHeader(HEADER_NAME_XTOKEN, mUserTokenResponse.getJwtUserToken());
        } else {
            Log.d(TAG, "authenticate: no jwt token");
        }

        isAuthenticating = false;
        return builder.build();
    }
}