package voxeet.com.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.network.ISdkFilePresentationRService;
import voxeet.com.sdk.core.services.abstracts.AbstractPresentationService;
import voxeet.com.sdk.events.error.HttpException;
import voxeet.com.sdk.events.success.FileConvertedEvent;
import voxeet.com.sdk.events.success.FilePresentationStartedEvent;
import voxeet.com.sdk.events.success.FilePresentationStoppedEvent;
import voxeet.com.sdk.events.success.FilePresentationUpdatedEvent;
import voxeet.com.sdk.json.FileConverted;
import voxeet.com.sdk.json.FilePresentationStarted;
import voxeet.com.sdk.json.FilePresentationStopped;
import voxeet.com.sdk.json.FilePresentationUpdated;
import voxeet.com.sdk.models.FilePresentationConverted;
import voxeet.com.sdk.models.impl.DefaultFile;

/**
 *
 */
public class SDKFilePresentationService extends AbstractPresentationService<ISdkFilePresentationRService> {
    private static final String TAG = SDKFilePresentationService.class.getSimpleName();

    private HashMap<String, Solver<FilePresentationConverted>> mCacheSolvers;
    private HashMap<String, Solver<FilePresentationStarted>> mCacheStartedSolvers;
    private HashMap<String, Solver<FilePresentationStopped>> mCacheStoppedSolvers;
    private HashMap<String, Solver<FilePresentationUpdated>> mCacheUpdatedSolvers;

    /**
     * Instantiates a new User service.
     *
     * @param instance the parent instance
     */
    public SDKFilePresentationService(VoxeetSdkTemplate instance) {
        super(instance, ISdkFilePresentationRService.class);

        mCacheSolvers = new HashMap<>();
        mCacheStartedSolvers = new HashMap<>();
        mCacheStoppedSolvers = new HashMap<>();
        mCacheUpdatedSolvers = new HashMap<>();
        registerEventBus();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationStartedEvent event) {
        tryUnlock(event.getEvent().getFileId(), event.getEvent(), mCacheStartedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationStoppedEvent event) {
        tryUnlock(event.getEvent().getFileId(), event.getEvent(), mCacheStoppedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationUpdatedEvent event) {
        tryUnlock(event.getEvent().getFileId(), event.getEvent(), mCacheUpdatedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FileConvertedEvent event) {
        if (null != event) {
            FileConverted converted = event.getEvent();
            List<DefaultFile> list = converted.getFiles();

            for (DefaultFile file : list) {
                Log.d(TAG, "onEvent: " + file.getName());
                String key = findSolverFor(file.getName());
                Log.d(TAG, "onEvent: " + key);
                if (null != key) {
                    FilePresentationConverted temp = new FilePresentationConverted(
                            file.getName(),
                            file.getFileId(),
                            file.getSize(),
                            file.getNbImageConverted());

                    tryUnlock(key, temp, mCacheSolvers);
                }
            }
        }
    }

    /**
     * Get the url for an image
     *
     * @param fileId     the remote file id
     * @param pageNumber the requested page number
     * @return a formatted string
     */
    public String getImage(String fileId, int pageNumber) {
        return String.format("%s/v1/files/%s/converted/%d?token=%s",
                getURLRoot(getVoxeetSdkInstance()), fileId, pageNumber, getInternalJwtToken(getVoxeetSdkInstance()));
    }

    /**
     * Get the url for a thumbnail
     *
     * @param fileId     the remote file id
     * @param pageNumber the requested page number
     * @return a formatted string
     */
    public String getThumbnail(String fileId, int pageNumber) {
        return String.format("%s/v1/files/%s/converted/%d/thumbnail?token=%s",
                getURLRoot(getVoxeetSdkInstance()), fileId, pageNumber, getInternalJwtToken(getVoxeetSdkInstance()));
    }

    @Nullable
    private String findSolverFor(@NonNull String name) {
        for (String value : mCacheSolvers.keySet())
            if (name.indexOf(value) == 0) return value;
        return null;
    }

    public Promise<FilePresentationConverted> convertFile(final File file) {
        return new Promise<>(new PromiseSolver<FilePresentationConverted>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationConverted> solver) {
                final String uuid = UUID.randomUUID().toString();

                String appended_name = uuid + file.getName();
                RequestBody requestFile = RequestBody.create(MediaType.parse("multipart/form-data"), file);

                // MultipartBody.Part is used to send also the actual file name
                MultipartBody.Part body = MultipartBody.Part.createFormData("file", appended_name, requestFile);


                // finally, execute the request
                getService().convertFile(requestFile, body)
                        .enqueue(new Callback<ResponseBody>() {
                            @Override
                            public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
                                //will wait for conversion result, so we put into cache...
                                if(response.isSuccessful()) {
                                    mCacheSolvers.put(uuid, solver);
                                } else {
                                    solver.reject(HttpException.throwResponse(response));
                                }
                            }

                            @Override
                            public void onFailure(Call<ResponseBody> call, Throwable e) {
                                Log.d(TAG, "onError: " + e);
                                solver.reject(e);
                            }
                        });
            }
        });
    }

    public Promise<FilePresentationStarted> startPresentation(@NonNull final FilePresentationConverted body) {
        return startPresentation(body, 0);
    }

    public Promise<FilePresentationStarted> startPresentation(@NonNull final FilePresentationConverted body,
                                                              final int position) {
        return new Promise<>(new PromiseSolver<FilePresentationStarted>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationStarted> solver) {
                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(body.getFileId(), body.getName(), position, body.getNbImageConverted());

                consumeInternalCall(solver, body.getFileId(), mCacheStartedSolvers,
                        internalCall(getService().startFilePresentation(getConferenceId(), body_sent)));
            }
        });
    }

    public Promise<FilePresentationStopped> stopPresentation(@NonNull final String fileId) {
        return new Promise<>(new PromiseSolver<FilePresentationStopped>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationStopped> solver) {
                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(fileId);

                consumeInternalCall(solver, fileId, mCacheStoppedSolvers,
                        internalCall(getService().startFilePresentation(getConferenceId(), body_sent)));
            }
        });
    }

    public Promise<FilePresentationUpdated> updatePresentation(@NonNull final String fileId,
                                                               final int position) {
        return new Promise<>(new PromiseSolver<FilePresentationUpdated>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationUpdated> solver) {

                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(fileId, position);

                consumeInternalCall(solver, fileId, mCacheUpdatedSolvers,
                        internalCall(getService().startFilePresentation(getConferenceId(), body_sent)));
            }
        });
    }

}