package voxeet.com.sdk.core.services;

import android.content.Context;
import android.support.annotation.NonNull;

import com.voxeet.android.media.MediaPowerManager;
import com.voxeet.android.media.audio.AudioManager;
import com.voxeet.android.media.audio.AudioRoute;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;

import voxeet.com.sdk.core.AbstractVoxeetService;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;
import voxeet.com.sdk.events.success.ConferencePreJoinedEvent;
import voxeet.com.sdk.json.ConferenceDestroyedPush;
import voxeet.com.sdk.utils.AudioType;
import voxeet.com.sdk.utils.SoundPool;
import voxeet.com.sdk.utils.Validate;

/**
 * Created by kevinleperf on 11/07/2018.
 */

public class AudioService extends AbstractVoxeetService {
    private final AudioManager mAudioManager;
    private final MediaPowerManager mMediaPowerManager;
    private final Context mContext;
    private final android.media.AudioManager mSystemAudioService;
    private final int mSoundMode;
    private int mSoundType = Integer.MIN_VALUE;
    private SoundPool _sound_pool;
    private HashMap<AudioType, String> mSounds;

    /**
     * Instantiate a new Audio Service
     *
     * @param instance the voxeet parent instance
     */
    public AudioService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        mSoundMode = android.media.AudioManager.STREAM_VOICE_CALL;

        mAudioManager = new AudioManager(instance.getApplicationContext());
        mMediaPowerManager = new MediaPowerManager(instance.getApplicationContext(),
                currentRoute());

        mContext = instance.getApplicationContext();
        instance.getEventBus().register(this);
        mSystemAudioService = (android.media.AudioManager) mContext.getSystemService(Context.AUDIO_SERVICE);

        mSounds = new HashMap<>();
        configure();
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return mAudioManager.availableRoutes();
    }

    public AudioRoute currentRoute() {
        return mAudioManager.outputRoute();
    }

    public boolean setAudioRoute(@NonNull AudioRoute route) {
        Validate.runningOnUiThread();
        mAudioManager.setOutputRoute(route);
        return true;
    }

    public void setSpeakerMode(boolean isSpeaker) {
        mAudioManager.setSpeakerMode(isSpeaker);
    }

    public boolean isSpeakerOn() {
        return AudioRoute.ROUTE_SPEAKER.equals(mAudioManager.outputRoute());
    }

    public void acquireLocks() {
        mMediaPowerManager.acquire();
    }

    public void releaseLocks() {
        mMediaPowerManager.release();
    }

    public void setInVoiceCallSoundType() {
        mAudioManager.forceVolumeControlStream(mSoundMode);

    }

    public void resetDefaultSoundType() {
        mAudioManager.forceVolumeControlStream(mAudioManager.getDefaultSoundStreamType());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferencePreJoinedEvent event) {
        mSoundType = getUiSoundsStreamType();
        forceVolumeControlStream(mSoundMode);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPush event) {
        if (Integer.MIN_VALUE != mSoundType) {
            forceVolumeControlStream(mSoundType);
        }
    }

    private int getUiSoundsStreamType() {
        try {
            Method method = mSystemAudioService.getClass().getDeclaredMethod("getUiSoundsStreamType");

            Object result = method.invoke(mSystemAudioService);
            return (int) result;
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
        return android.media.AudioManager.STREAM_SYSTEM;
    }

    private void forceVolumeControlStream(int type) {
        try {
            Method method = mSystemAudioService.getClass().getDeclaredMethod("forceVolumeControlStream", int.class);

            method.invoke(mSystemAudioService, type);
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
    }

    public void abandonAudioFocusRequest() {
        mAudioManager.abandonAudioFocusRequest();
    }

    public void requestAudioFocus() {
        mAudioManager.requestAudioFocus();
    }

    private SoundPool getSoundPool() {
        if (_sound_pool == null) {
            _sound_pool = new SoundPool(mContext, mSoundMode);
        }
        return _sound_pool;
    }

    private void configure() {
        mSounds.put(AudioType.RING, "out.mp3");
        mSounds.put(AudioType.HANGUP, "leave.mp3");

        setSound(AudioType.RING, mSounds.get(AudioType.RING));
        setSound(AudioType.HANGUP, mSounds.get(AudioType.HANGUP));
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return getSoundPool().release(type).setShortResource(type, assetName);
    }

    public void playSoundType(@NonNull AudioType type) {
        getSoundPool().playShortResource(type, mSounds.get(type));
    }

    public void stopSoundType(@NonNull AudioType audioType) {
        getSoundPool().stop(audioType);
    }


    public void stop() {
        getSoundPool().stop();
    }
}
