package voxeet.com.sdk.core.network.websocket;

import android.content.Context;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.neovisionaries.ws.client.WebSocket;
import com.neovisionaries.ws.client.WebSocketAdapter;
import com.neovisionaries.ws.client.WebSocketCloseCode;
import com.neovisionaries.ws.client.WebSocketException;
import com.neovisionaries.ws.client.WebSocketExtension;
import com.neovisionaries.ws.client.WebSocketFactory;
import com.neovisionaries.ws.client.WebSocketFrame;
import com.neovisionaries.ws.client.WebSocketState;
import com.voxeet.kernel.BuildConfig;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import voxeet.com.sdk.utils.DeviceStateUtils;

public class WebSocketProxy implements Runnable {

    private final static Handler HANDLER = new Handler();
    private final static int TIMEOUT = 5000;
    private static final int MAXIMUM_RETRY = 5;
    private static final long SOCKET_RETRY_MAX = 60 * 1000; //60s
    private static final long CLOSE_VOXEET_AFTER_MILLISECS = 2000;

    private WebSocketProxyState mState;
    private Context mContext;
    private ExecutorService mExecutorService;
    private String mSocketUrl;
    private int mCount = 0;
    private boolean shouldRetry = true;
    private WebSocketAdapter mAdapter;
    private WebSocketAdapter mAdapterListener;
    private List<SocketListener> mListeners;
    private long mSocketCurrentRetryDelay;
    private String mUserToken;
    private String mJwtUserToken;
    private WebSocketFactory mFactory;
    private com.neovisionaries.ws.client.WebSocket mWebSocket;
    private ConnectListener mConnectListener;
    private boolean isCanceled = false;

    public WebSocketProxy(Context context, WebSocketAdapter adapter, String socketUrl) {
        mState = WebSocketProxyState.CLOSED;
        mSocketUrl = socketUrl;
        mFactory = new WebSocketFactory();
        mFactory.setConnectionTimeout(TIMEOUT);

        mContext = context;
        mExecutorService = Executors.newSingleThreadExecutor();


        mListeners = new ArrayList<>();
        mCount = 0;
        mAdapterListener = adapter;

        mAdapter = new WebSocketAdapter() {

            @Override
            public void onConnected(WebSocket websocket, Map<String, List<String>> headers) throws Exception {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                mWebSocket = websocket;

                websocket.sendPing();
                websocket.setPingInterval(30000);

                super.onConnected(websocket, headers);

                mState = WebSocketProxyState.CONNECTED;

                //reset values to let the app reconnect on error
                mCount = 0;
                shouldRetry = true;

                attemptConnectListenerConnected(websocket);

                for (SocketListener listener : mListeners) {
                    listener.onConnect(websocket);
                }
            }

            @Override
            public void onTextMessage(WebSocket websocket, String message) throws Exception {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                for (SocketListener listener : mListeners) {
                    listener.onTextMessage(message);
                }
            }

            @Override
            public void onError(WebSocket websocket, WebSocketException cause) throws Exception {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                super.onError(websocket, cause);

                for (SocketListener listener : mListeners) {
                    listener.onError(cause);
                }
            }

            @Override
            public void onDisconnected(WebSocket websocket, WebSocketFrame serverCloseFrame, WebSocketFrame clientCloseFrame, boolean closedByServer) throws Exception {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                super.onDisconnected(websocket, serverCloseFrame, clientCloseFrame, closedByServer);

                if (shouldRetry) {
                    if (mCount < MAXIMUM_RETRY) {

                        mState = WebSocketProxyState.CONNECTING;
                        mCount++;
                        HANDLER.postDelayed(WebSocketProxy.this, mSocketCurrentRetryDelay);
                    } else {
                        mState = WebSocketProxyState.CLOSED;
                        attemptConnectListenerDisconnected();
                        WebSocketProxy.this.onDisconnected();
                    }
                } else {
                    mState = WebSocketProxyState.CLOSED;
                    WebSocketProxy.this.onDisconnected();
                }

            }

            @Override
            public void onStateChanged(WebSocket websocket, WebSocketState newState) throws Exception {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                super.onStateChanged(websocket, newState);

                for (SocketListener listener : mListeners) {
                    listener.onStateChanged(newState);
                }
            }
        };
    }

    private void attemptConnectListenerDisconnected() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (mConnectListener != null) {
            mConnectListener.onConnectError(new Exception("Socket disconnected" + mWebSocket));
            mConnectListener = null;
        }
    }

    private void attemptConnectListenerConnected(WebSocket webSocket) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (mConnectListener != null) {
            mConnectListener.onConnect(webSocket);
            mConnectListener = null;
        }
    }

    private void initWebSocket(@Nullable String jwtUserToken) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        String socketUrl = mSocketUrl;

        if(null != jwtUserToken) {
            if(socketUrl.indexOf("?") < 0) socketUrl += "?Token=" + jwtUserToken;
            else socketUrl += "Token=" + jwtUserToken;
        }

        try {
            mWebSocket = mFactory.createSocket(socketUrl);
        } catch (IOException e) {
            mWebSocket = null;
        }
    }

    public void connect(@NonNull String userToken, @NonNull String jwtUserToken, @Nullable ConnectListener connectListener) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        mConnectListener = connectListener;
        mUserToken = userToken;
        mJwtUserToken = jwtUserToken;

        if (DeviceStateUtils.isNetworkAvailable(mContext)) {
            mState = WebSocketProxyState.CONNECTING;
            initWebSocket(jwtUserToken);

            if(mWebSocket != null) {
                mWebSocket.setPingInterval(30000);
                mWebSocket.addExtension(WebSocketExtension.PERMESSAGE_DEFLATE);
                mWebSocket.clearHeaders();
                if (mUserToken != null) {
                    mWebSocket.addHeader("Voxeet-Token", mUserToken.replaceAll("\"", ""));
                }
                mWebSocket.addListener(mAdapter);
                if (mAdapterListener != null) {
                    mWebSocket.addListener(mAdapterListener);
                }

                // Connect to the server asynchronously.
                mWebSocket.connect(mExecutorService);
            }
        } else {
            mState = WebSocketProxyState.CLOSED;
            if(mConnectListener != null) {
                mConnectListener.onNoNetwork();
            }
        }
    }

    private void onDisconnected() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        for (SocketListener listener : mListeners) {
            listener.onDisconnected();
        }
    }

    public void addListener(@NonNull SocketListener listener) {
        if (!mListeners.contains(listener)) {
            mListeners.add(listener);
        }
    }

    public void removeListener(@NonNull SocketListener listener) {
        if (mListeners.contains(listener)) {
            mListeners.remove(listener);
        }
    }

    @Override
    public void run() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (DeviceStateUtils.isNetworkAvailable(mContext)) { // No point trying to reconnect if no network is available
            connect(mUserToken, mJwtUserToken, mConnectListener);
        } else {

            mSocketCurrentRetryDelay = Math.min(mSocketCurrentRetryDelay * 2, SOCKET_RETRY_MAX);

            HANDLER.postDelayed(this, mSocketCurrentRetryDelay);
        }
    }

    public boolean isOpen() {
        return mWebSocket != null && mWebSocket.isOpen();
    }

    public boolean sendPing() {
        return mWebSocket != null && mWebSocket.sendPing().getState() == WebSocketState.OPEN;
    }

    public void disconnect() {
        mState = WebSocketProxyState.CLOSED;
        //prevent reconnection
        mCount = MAXIMUM_RETRY;
        shouldRetry = false;

        //disconnect
        if(null != mWebSocket) {
            mWebSocket.sendClose();
            mWebSocket.disconnect(WebSocketCloseCode.NORMAL, null, CLOSE_VOXEET_AFTER_MILLISECS);
        }

        for (SocketListener listener : mListeners) {
            listener.onClose();
        }

        mExecutorService.shutdown();
    }

    public void removeListeners() {
        mAdapterListener = null;
        mListeners.clear();
        mAdapter = null;
    }

    public WebSocketProxyState getState() {
        return mState;
    }

    public WebSocket getWebSocket() {
        return mWebSocket;
    }

    private void checkCancel() {
        if (isCanceled && isOpen()) {
            disconnect();
        }
    }

    public void cancel() {
        removeListeners();
        isCanceled = true;
    }
}
