package voxeet.com.sdk.core.abs;

import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaSDK;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.audio.AudioRoute;
import com.voxeet.android.media.peer.PendingPeerCallback;
import com.voxeet.android.media.peer.SdpCandidate;
import com.voxeet.android.media.peer.SdpDescription;
import com.voxeet.android.media.peer.SdpMessage;
import com.voxeet.android.media.video.Camera2Enumerator;
import com.voxeet.kernel.R;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBase;
import org.webrtc.VideoRenderer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Response;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.schedulers.Schedulers;
import voxeet.com.sdk.audio.MessagingEnvironment;
import voxeet.com.sdk.core.ConferenceSimpleState;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.abs.information.ConferenceInformation;
import voxeet.com.sdk.core.abs.information.ConferenceInformationHolder;
import voxeet.com.sdk.core.preferences.VoxeetPreferences;
import voxeet.com.sdk.core.services.AudioService;
import voxeet.com.sdk.core.services.SdkConferenceService;
import voxeet.com.sdk.core.services.TimeoutRunnable;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;
import voxeet.com.sdk.events.error.CameraSwitchErrorEvent;
import voxeet.com.sdk.events.error.ConferenceCreatedError;
import voxeet.com.sdk.events.error.ConferenceJoinedError;
import voxeet.com.sdk.events.error.ConferenceLeftError;
import voxeet.com.sdk.events.error.GetConferenceHistoryErrorEvent;
import voxeet.com.sdk.events.error.GetConferenceStatusErrorEvent;
import voxeet.com.sdk.events.error.ParticipantAddedErrorEvent;
import voxeet.com.sdk.events.error.PermissionRefusedEvent;
import voxeet.com.sdk.events.error.ReplayConferenceErrorEvent;
import voxeet.com.sdk.events.error.SdkLogoutErrorEvent;
import voxeet.com.sdk.events.error.SubscribeConferenceErrorEvent;
import voxeet.com.sdk.events.error.SubscribeForCallConferenceErrorEvent;
import voxeet.com.sdk.events.error.UnsubscribeFromCallConferenceErrorEvent;
import voxeet.com.sdk.events.promises.InConferenceException;
import voxeet.com.sdk.events.promises.NotInConferenceException;
import voxeet.com.sdk.events.promises.PromiseConferenceJoinedErrorException;
import voxeet.com.sdk.events.promises.PromiseDeclineConferenceResultEventException;
import voxeet.com.sdk.events.promises.PromiseGetConferenceStatusErrorEventException;
import voxeet.com.sdk.events.promises.PromiseParticipantAddedErrorEventException;
import voxeet.com.sdk.events.promises.PromisePermissionRefusedEventException;
import voxeet.com.sdk.events.success.AddConferenceParticipantResultEvent;
import voxeet.com.sdk.events.success.CameraSwitchSuccessEvent;
import voxeet.com.sdk.events.success.ConferenceCreationSuccess;
import voxeet.com.sdk.events.success.ConferenceDestroyedPushEvent;
import voxeet.com.sdk.events.success.ConferenceEndedEvent;
import voxeet.com.sdk.events.success.ConferenceJoinedSuccessEvent;
import voxeet.com.sdk.events.success.ConferenceLeftSuccessEvent;
import voxeet.com.sdk.events.success.ConferencePreJoinedEvent;
import voxeet.com.sdk.events.success.ConferenceRefreshedEvent;
import voxeet.com.sdk.events.success.ConferenceStatsEvent;
import voxeet.com.sdk.events.success.ConferenceUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUserAddedEvent;
import voxeet.com.sdk.events.success.ConferenceUserCallDeclinedEvent;
import voxeet.com.sdk.events.success.ConferenceUserJoinedEvent;
import voxeet.com.sdk.events.success.ConferenceUserLeftEvent;
import voxeet.com.sdk.events.success.ConferenceUserQualityUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUserUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUsersInvitedEvent;
import voxeet.com.sdk.events.success.DeclineConferenceResultEvent;
import voxeet.com.sdk.events.success.GetConferenceHistoryEvent;
import voxeet.com.sdk.events.success.GetConferenceStatusEvent;
import voxeet.com.sdk.events.success.IncomingCallEvent;
import voxeet.com.sdk.events.success.InvitationReceived;
import voxeet.com.sdk.events.success.OfferCreatedEvent;
import voxeet.com.sdk.events.success.OwnConferenceStartedEvent;
import voxeet.com.sdk.events.success.ParticipantAddedEvent;
import voxeet.com.sdk.events.success.ParticipantUpdatedEvent;
import voxeet.com.sdk.events.success.QualityIndicators;
import voxeet.com.sdk.events.success.QualityUpdatedEvent;
import voxeet.com.sdk.events.success.RecordingStatusUpdate;
import voxeet.com.sdk.events.success.RenegociationEndedEvent;
import voxeet.com.sdk.events.success.RenegociationUpdate;
import voxeet.com.sdk.events.success.ReplayConferenceEvent;
import voxeet.com.sdk.events.success.ResumeConference;
import voxeet.com.sdk.events.success.ScreenStreamAddedEvent;
import voxeet.com.sdk.events.success.ScreenStreamRemovedEvent;
import voxeet.com.sdk.events.success.SdkLogoutSuccessEvent;
import voxeet.com.sdk.events.success.SendBroadcastResultEvent;
import voxeet.com.sdk.events.success.StartRecordingResultEvent;
import voxeet.com.sdk.events.success.StartScreenShareAnswerEvent;
import voxeet.com.sdk.events.success.StartVideoAnswerEvent;
import voxeet.com.sdk.events.success.StopRecordingResultEvent;
import voxeet.com.sdk.events.success.StopScreenShareAnswerEvent;
import voxeet.com.sdk.events.success.StopVideoAnswerEvent;
import voxeet.com.sdk.events.success.SubscribeConferenceEvent;
import voxeet.com.sdk.events.success.SubscribeForCallConferenceAnswerEvent;
import voxeet.com.sdk.events.success.UnSubscribeConferenceAnswerEvent;
import voxeet.com.sdk.events.success.UnSubscribeFromConferenceAnswerEvent;
import voxeet.com.sdk.factories.VoxeetIntentFactory;
import voxeet.com.sdk.json.BroadcastEvent;
import voxeet.com.sdk.json.ConferenceStats;
import voxeet.com.sdk.json.ConferenceUserAdded;
import voxeet.com.sdk.json.CreateConferenceParams;
import voxeet.com.sdk.json.DeviceEvent;
import voxeet.com.sdk.json.InvitationReceivedEvent;
import voxeet.com.sdk.json.OfferCreated;
import voxeet.com.sdk.json.RecordingStatusUpdateEvent;
import voxeet.com.sdk.json.SdkConferenceInvitation;
import voxeet.com.sdk.json.SdkConferenceReplayBody;
import voxeet.com.sdk.json.StartScreenSharingResponse;
import voxeet.com.sdk.json.StartVideoResponse;
import voxeet.com.sdk.json.StopScreenSharingResponse;
import voxeet.com.sdk.json.StopVideoResponse;
import voxeet.com.sdk.json.UserInfo;
import voxeet.com.sdk.models.ConferenceQuality;
import voxeet.com.sdk.models.ConferenceResponse;
import voxeet.com.sdk.models.ConferenceType;
import voxeet.com.sdk.models.ConferenceUserStatus;
import voxeet.com.sdk.models.HistoryConference;
import voxeet.com.sdk.models.NormalConferenceResponse;
import voxeet.com.sdk.models.OfferCandidate;
import voxeet.com.sdk.models.OfferDescription;
import voxeet.com.sdk.models.RecordingStatus;
import voxeet.com.sdk.models.SubscribeConference;
import voxeet.com.sdk.models.abs.Conference;
import voxeet.com.sdk.models.abs.ConferenceUser;
import voxeet.com.sdk.models.impl.DefaultConference;
import voxeet.com.sdk.models.impl.DefaultConferenceUser;
import voxeet.com.sdk.models.impl.DefaultInvitation;
import voxeet.com.sdk.models.impl.DefaultParticipant;
import voxeet.com.sdk.networking.DeviceType;
import voxeet.com.sdk.utils.ConferenceListener;
import voxeet.com.sdk.utils.Twig;
import voxeet.com.sdk.utils.Validate;

/**
 * Conference SDK Service
 * <p>
 * Warning : always implements the error for each promises you may call to start resolve them
 * You can also use the execute method but in case of Exception, you will trigger errors...
 *
 * @see SdkConferenceService
 */
public abstract class AbstractConferenceSdkService<T, COP extends AbstractConferenceSdkObservableProvider<T, DemoEvent>, DemoEvent>
        extends voxeet.com.sdk.core.AbstractVoxeetService<T>
        implements SdkConferenceService, ConferenceListener {

    private final static String TAG = AbstractConferenceSdkService.class.getSimpleName();
    private VoxeetSdkTemplate mSDK;
    private ConferenceListener mListener;
    private AbstractConferenceSdkObservableProvider mConferenceObservableProvider;
    private Twig mTwig;

    private EventBus mEventBus;
    private MediaSDK media;
    private EglBase mEglBase = EglBase.create();
    private final MessagingEnvironment mEnvironment;
    private boolean isListenerMode = false;
    private boolean isVideoOn = false; // default camera setting
    private boolean isScreenshareOn = false;
    private boolean isRecording = false;
    private boolean mInConference = false;
    private long mTimeOutTimer = -1;
    private HashMap<String, MediaStream> mapOfStreams = new HashMap<>();
    private HashMap<String, MediaStream> mapOfScreenShareStreams = new HashMap<>();


    private String mDefaultCamera;
    private boolean mEnableStats;
    private boolean isDefaultOnSpeaker;

    private String mConferenceId = null;
    private ConferenceInformationHolder mConferenceInformationHolder = new ConferenceInformationHolder();

    protected Context mContext;

    private @Nullable
    TimeoutRunnable timeoutRunnable = null;

    private MediaEngine.StreamListener mediaStreamListener = new MediaEngine.StreamListener() {
        @Override
        public void onStreamAdded(final String peer, final MediaStream stream) {
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    getTwig().i("New mConference user joined with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());

                    mapOfStreams.put(peer, stream);
                    DefaultConferenceUser user = updateConferenceParticipants(peer, ConferenceUserStatus.ON_AIR);
                    if (user != null) {
                        getTwig().i("New mConference user joined with id: " + user.getUserId());

                        if (!peer.equalsIgnoreCase(VoxeetPreferences.id()) && mTimeOutTimer != -1) {
                            mTwig.i("Cancelling timeout timer");
                            removeTimeoutCallbacks();
                        }

                        mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
                    }
                }
            });
        }

        @Override
        public void onStreamUpdated(final String peer, final MediaStream stream) {
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    getTwig().i("Conference user updated with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());
                    getTwig().i("stream updated having video ? " + (stream.videoTracks().size() > 0) + " screenshare := " + stream.isScreenShare());

                    if (stream.isScreenShare()) mapOfScreenShareStreams.put(peer, stream);
                    else mapOfStreams.put(peer, stream);

                    DefaultConferenceUser user = findUserById(peer);
                    if (user != null) {
                        getTwig().i("Conference user updated with id: " + user.getUserId());

                        mEventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    }
                }
            });
        }

        @Override
        public void onStreamRemoved(final String peer) {
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    onUserLeft(peer);
                }
            });
        }

        @Override
        public void onScreenStreamAdded(final String peer, final MediaStream stream) {
            getTwig().i("(for ScreenShare) Conference user updated with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());
            getTwig().i("screenshare stream updated having video ? " + stream.videoTracks().size() + " :: has stream inside ? " + (stream.videoTracks().size() > 0));

            mapOfScreenShareStreams.put(peer, stream);
            mEventBus.post(new ScreenStreamAddedEvent(peer, stream));
        }

        @Override
        public void onScreenStreamRemoved(String peer) {
            getTwig().i("Screen share stream removed: " + peer);

            mapOfScreenShareStreams.remove(peer);
            mEventBus.post(new ScreenStreamRemovedEvent(peer));
        }

        @Override
        public void onShutdown() {

        }
    };

    public AbstractConferenceSdkService(VoxeetSdkTemplate instance, COP conference_observable_provider, long timeout, @NonNull ServiceProviderHolder<T> holder) {
        super(instance, holder);

        mEnableStats = false;

        mDefaultCamera = new Camera2Enumerator(instance.getApplicationContext()).getNameOfFrontFacingDevice();
        mConferenceObservableProvider = conference_observable_provider;
        mConferenceObservableProvider.setRetrofitInstantiatedProvider(getService());
        mListener = instance;
        mSDK = instance;
        mTimeOutTimer = timeout;
        mTwig = instance.getTwig();
        mEventBus = EventBus.getDefault();
        mContext = instance.getApplicationContext();
        setDefaultBuiltInSpeaker(true);
        mEnvironment = new MessagingEnvironment.Builder(mContext)
                //.setServerHost(R.string.connectionManagerHost_prod)
                .setServerHost(R.string.session_manager_url)
                .setServerPort(R.string.connectionManagerPort)
                .setStunHost(R.string.stunHost)
                .setStunPort(R.string.stunPort)
                .build();


        register();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     * @deprecated Please use {@link #mute(boolean)} ()} instead.
     */
    @Deprecated
    public boolean muteConference(boolean mute) {
        return mute(mute);
    }

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     */
    @Override
    public boolean mute(boolean mute) {
        if (null != media) {
            if (!mute && media.isMuted()) {
                if (!Validate.hasMicrophonePermissions(context)) {
                    getTwig().i("No permission for mic... please check it");
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                } else {
                    getTwig().i("Conference unmuted");
                    getAudioService().setInVoiceCallSoundType();

                    media.unMute();
                }
            } else if (mute) {
                getTwig().i("Conference muted");

                getAudioService().setInVoiceCallSoundType();
                media.mute();
            }
        }
        return true;
    }

    @Override
    public boolean isMuted() {
        return media != null && media.isMuted();
    }

    @Override
    public boolean isUserMuted(String userId) {
        DefaultConferenceUser user = findUserById(userId);
        return user != null && user.isMuted();
    }

    @Override
    public void setListenerMode(boolean isListener) {
        if (null != media) {
            if (media.isMuted()) {
                media.unMute();
            } else {
                media.mute();
            }
            getAudioService().setInVoiceCallSoundType();
        }
    }

    @Override
    public boolean attachMediaStream(MediaStream stream, VideoRenderer.Callbacks render) {
        if (media != null) media.attachMediaStream(render, stream);
        return null != media;
    }

    @Override
    public boolean unAttachMediaStream(MediaStream stream, VideoRenderer.Callbacks render) {
        if (media != null) media.unattachMediaStream(render, stream);
        return null != media;
    }

    @Override
    public void register() {
        registerEventBus();
    }

    @Override
    public void unregister() {
        unRegisterEventBus();
    }

    public void setDefaultCamera(String cameraName) {
        mDefaultCamera = cameraName;
    }

    @Override
    public void toggleVideo() {
        Promise<Boolean> promise = isVideoOn() ? stopVideo() : startVideo();
        promise
                .then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                        Log.d(TAG, "onSuccess: toggleVideo " + result);
                    }
                })
                .error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                    }
                });
    }

    public void toggleScreenShare() {
        if (isScreenShareOn()) {
            stopScreenShare()
                    .then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onSuccess: toggleScreenShare " + result);
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            Log.d(TAG, "onError: toggleScreenShare");
                            error.printStackTrace();
                        }
                    });
        } else {
            getVoxeetSDK().getScreenShareService().sendRequestStartScreenShare();
        }
    }

    @Override
    public String getCurrentConferenceId() {
        Validate.runningOnUiThread();
        return mConferenceId;
    }

    @Override
    public int getConferenceRoomSize() {
        Conference conference = getCurrentConferenceInformation().getConference();
        return conference.getConferenceRoomSize();
    }

    @Override
    public String currentSpeaker() {
        Conference conference = getConference();
        if (media == null || conference == null || conference.getConferenceUsers() == null) {
            return VoxeetPreferences.id();
        } else {
            String currentSpeaker = null;
            for (DefaultConferenceUser user : conference.getConferenceUsers()) {
                if (user.getUserId() != null) {
                    int peerVuMeter = media.getPeerVuMeter(user.getUserId());
                    if (currentSpeaker == null || (peerVuMeter > 100 && media.getPeerVuMeter(currentSpeaker) < peerVuMeter))
                        currentSpeaker = user.getUserId();
                }
            }
            return currentSpeaker;
        }
    }

    @NonNull
    private String currentUserOrEmpty() {
        String currentUserId = VoxeetPreferences.id();
        return null != currentUserId ? currentUserId : "";
    }

    @Override
    public int getSdkPeerVuMeter(String peerId) {
        Validate.runningOnUiThread();
        return media != null ? media.getPeerVuMeter(peerId) : 0;
    }

    @Override
    public DefaultConferenceUser findUserById(final String userId) {

        Log.d(TAG, "findUserById: looking for USER " + userId);
        Validate.notNull(userId, "user id");

        Conference conference = getConference();
        if (conference != null && conference.getConferenceUsers() != null) {

            return Iterables.find(conference.getConferenceUsers(), new Predicate<DefaultConferenceUser>() {
                @Override
                public boolean apply(DefaultConferenceUser input) {
                    return userId.equalsIgnoreCase(input.getUserId());
                }
            }, null);
        }

        return null;
    }

    @Override
    public String getAliasId() {
        DefaultConference conference = getConference();
        return null != conference ? conference.getConferenceAlias() : null;
    }

    @Override
    public String getConferenceId() {
        return mConferenceId;
    }

    @Deprecated
    public List<AudioRoute> getAvailableRoutes() {
        return getAudioService().getAvailableRoutes();
    }

    @Deprecated
    public AudioRoute currentRoute() {
        return getAudioService().currentRoute();
    }

    @Override
    public EglBase.Context getEglContext() {
        return mEglBase.getEglBaseContext();
    }

    @Override
    public long getTimeout() {
        return mTimeOutTimer;
    }

    @Override
    public void toggleRecording() {
        Promise<Boolean> promise = isRecording ? stopRecording() : startRecording();
        promise
                .then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                        Log.d(TAG, "onSuccess: toggle done " + result);
                    }
                })
                .error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                    }
                });
    }

    @Override
    public boolean muteUser(@NonNull String userId, boolean shouldMute) {
        Validate.runningOnUiThread();

        DefaultConferenceUser user = findUserById(userId);
        if (user != null) {
            getTwig().i("Setting mute property for mConference participant with id " + userId + " to " + shouldMute);

            startTransactionConferenceUser();
            user.setMuted(shouldMute);
            media.changePeerGain(userId, shouldMute ? MUTE_FACTOR : UNMUTE_FACTOR);
            commitTransactionConferenceUser();
        }
        return user != null;
    }

    public boolean isVideoOn() {
        return isVideoOn;
    }

    public boolean isScreenShareOn() {
        return isScreenshareOn;
    }

    @Deprecated
    public boolean setAudioRoute(AudioRoute route) {
        return getAudioService().setAudioRoute(route);
    }

    public boolean setDefaultBuiltInSpeaker(boolean default_state) {
        isDefaultOnSpeaker = default_state;
        return true;
    }

    @Override
    public boolean setTimeOut(long timeout) {
        getTwig().i("Timeout set to " + timeout);
        mTimeOutTimer = timeout;
        return true;
    }

    @Override
    public boolean setUserPosition(final String userId, final double angle, final double distance) {
        if (media != null) media.changePeerPosition(userId, angle, distance);
        return null != media;
    }

    /**
     * Get the current conference type
     *
     * @return a ConferenceType, ConferenceType.NONE is the default value
     */
    @NonNull
    public ConferenceSimpleState getConferenceType() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information || null == mConferenceId) return ConferenceSimpleState.NONE;
        return information.getConferenceType();
    }

    /**
     * Get the current mConference object
     *
     * @return a nullable object
     */
    @Nullable
    public DefaultConference getConference() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return null;

        return information.getConference();
    }

    @NonNull
    private DefaultConference getOrSetConference(@NonNull String conferenceId) {
        return getCurrentConferenceInformation().getConference();
    }

    /**
     * Get the current state of joining
     * in conjunction of the getConference() it is easy to check the current state :
     * <p>
     * mInConference() + getConference() non null = mConference joined and on air
     * mInConference() + getConference() null = joining mConference with the answer from server for now
     * !mInConference() = not currently joining, joined mConference (or left)
     *
     * @return if in mConference or "attempting" to join
     */
    public boolean isInConference() {
        return mInConference;
    }

    protected void setIsInConference(boolean status) {
        mInConference = status;
    }

    public int getPeerVuMeter(String peerId) {
        Validate.runningOnUiThread();

        if (media != null) return media.getPeerVuMeter(peerId);
        return 0;
    }

    @Nullable
    @Override
    public List<DefaultConferenceUser> getConferenceUsers() {
        DefaultConference conference = getConference();
        if (null == conference) return null;
        return conference.getConferenceUsers();
    }

    @NonNull
    public List<DefaultConferenceUser> getLastInvitationUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return new ArrayList<>();
        return information.getLastInvitationReceived();
    }

    @Override
    public boolean isLive() {
        return null != getConference();
    }

    @Override
    public boolean isListenerMode() {
        return isListenerMode;
    }

    @Nullable
    public ConferenceUser getUser(final String userId) {
        Conference conference = getConference();
        if (null != conference && conference.getConferenceUsers() != null) {
            return Iterables.find(conference.getConferenceUsers(), new Predicate<ConferenceUser>() {
                @Override
                public boolean apply(ConferenceUser input) {
                    return userId.equals(input.getUserId());
                }
            }, null);
        }
        return null;
    }

    /**
     * Deprecated since it will possibly be removed in the near future
     *
     * @return the map of streams available
     */
    @Deprecated
    @NonNull
    public HashMap<String, MediaStream> getMapOfStreams() {
        return mapOfStreams;
    }

    /**
     * Deprecated since it will possibly be removed in the near future
     *
     * @return the map of screenshare streams available
     */
    @Deprecated
    @NonNull
    public HashMap<String, MediaStream> getMapOfScreenShareStreams() {
        return mapOfScreenShareStreams;
    }

    /**
     * Give the ability to cancel the timeout in some cases
     */
    public AbstractConferenceSdkService<T, COP, DemoEvent> cancelTimeout() {
        removeTimeoutCallbacks();

        return this;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected abstract Promise<Boolean> onCreateDemoSuccess(DemoEvent response);

    /**
     * Join a conference with a conference
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     *
     * @param conferenceId id of the conference to join as a listener
     * @return a promise to resolve
     */
    @Override
    public Promise<Boolean> listenConference(@NonNull String conferenceId) {
        isListenerMode = true;
        getTwig().i("Listener mode set to true");

        //init the conference with the proper conferenceId and alias to null
        createOrSetConferenceWithParams(conferenceId, null);

        return joinVoxeetConference(conferenceId);
    }

    /**
     * resolve -> the result event
     * reject -> network error
     *
     * @param conferenceId the conference id to decline
     * @return a promise to resolve
     */
    @Override
    public Promise<DeclineConferenceResultEvent> decline(final String conferenceId) {
        return new Promise<>(new PromiseSolver<DeclineConferenceResultEvent>() {
            @Override
            public void onCall(@NonNull final Solver<DeclineConferenceResultEvent> solver) {
                final Observable<Response<ResponseBody>> user = mConferenceObservableProvider.getDeclineObservable(conferenceId);
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e(e);

                                DeclineConferenceResultEvent event = new DeclineConferenceResultEvent(handleError(e), false);
                                mEventBus.post(event);
                                solver.reject(new PromiseDeclineConferenceResultEventException(event));
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                getTwig().i("IConference declined with id: " + conferenceId);

                                DeclineConferenceResultEvent event = new DeclineConferenceResultEvent(response.code() == 200);
                                mEventBus.post(event);
                                solver.resolve(event);
                            }
                        });
            }
        });
    }

    public void sendRequestStartScreenShare() {
        getVoxeetSDK().getScreenShareService().sendRequestStartScreenShare();
    }

    public void onUserCanceledScreenShare() {
        //call before any specific code so stopScreenCapturer() should do nothing /!\
        media.stopScreenCapturer();
        isScreenshareOn = false;
    }

    public Promise<Boolean> startScreenShare(final Intent intent,
                                             final int width,
                                             final int height) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    if (isScreenShareOn()) {
                        solver.resolve(true);
                    }

                    if (null == media) {
                        Log.d(TAG, "startScreenShare: media is null");
                        throw new MediaEngineException("Media is null, invalid state");
                    }

                    isScreenshareOn = true;
                    media.startScreenCapturer(intent, width, height);

                    final Observable<StartScreenSharingResponse> startScreenShare = mConferenceObservableProvider.getStartScreenShareObservable(mConferenceId, VoxeetPreferences.id());
                    startScreenShare.subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                            .subscribe(new Subscriber<StartScreenSharingResponse>() {
                                @Override
                                public void onCompleted() {

                                }

                                @Override
                                public void onError(Throwable e) {
                                    mTwig.e(e);
                                    isScreenshareOn = false;

                                    StartScreenShareAnswerEvent event = new StartScreenShareAnswerEvent(false);
                                    mEventBus.post(event);
                                    solver.reject(e);
                                }

                                @Override
                                public void onNext(StartScreenSharingResponse response) {
                                    isScreenshareOn = true;

                                    final StartScreenShareAnswerEvent event = new StartScreenShareAnswerEvent(true);
                                    mEventBus.post(event);
                                    createVideoAnswer(response.getUserId(), response.getDescription(), response.getCandidates())
                                            .then(new PromiseExec<Boolean, Object>() {
                                                @Override
                                                public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                                    Log.d(TAG, "onCall: createVideoAnswer := having result " + aBoolean);
                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    error.printStackTrace();
                                                    solver.resolve(true);
                                                }
                                            });
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                }
            }
        });
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopScreenShare() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    String conferenceId = getCurrentConferenceId();

                    if (!isInConference()) {
                        Log.d(TAG, "stopScreenShare: not in conf");

                        if (null != media) media.stopScreenCapturer();
                        throw new NotInConferenceException();
                    }

                    if (null == media) {
                        Log.d(TAG, "stopScreenShare: media is null");
                        throw new MediaEngineException("media is null");
                    }
                    //Validate.notNull(mConferenceId, "mConference Id");
                    //Validate.notNull(media, "media");

                    media.stopScreenCapturer();

                    final Observable<StopScreenSharingResponse> stopVideo = mConferenceObservableProvider.getStopScreenShareObservable(conferenceId, VoxeetPreferences.id());
                    stopVideo.subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                            .subscribe(new Subscriber<StopScreenSharingResponse>() {
                                @Override
                                public void onCompleted() {
                                }

                                @Override
                                public void onError(Throwable e) {
                                    getTwig().e(e);

                                    isScreenshareOn = false;

                                    StopScreenShareAnswerEvent event = new StopScreenShareAnswerEvent(false);
                                    mEventBus.post(event);
                                    solver.reject(e);
                                }

                                @Override
                                public void onNext(StopScreenSharingResponse response) {
                                    media.stopScreenCapturer();

                                    isScreenshareOn = false;

                                    final StopScreenShareAnswerEvent event = new StopScreenShareAnswerEvent(true);
                                    mEventBus.post(event);
                                    createVideoAnswer(response.getUserId(), response.getDescription(), response.getCandidates())
                                            .then(new PromiseExec<Boolean, Object>() {
                                                @Override
                                                public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                                    //TODO check for possible call in the createVideoAnswer (callback ?)
                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    error.printStackTrace();
                                                    //TODO check for possible call in the createVideoAnswer (callback ?)
                                                    solver.resolve(true);
                                                }
                                            });
                                }
                            });
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                } catch (Exception exception) {
                    solver.reject(exception);
                }
            }
        });
    }

    /**
     * Start the video
     * resolve -> StartVideoAnswerEvent
     * reject -> PromisePermissionRefusedEventException
     * reject -> MediaException
     * <p>
     * If the app does not have the permission, will fire a PermissionRefusedEvent
     */
    public Promise<Boolean> startVideo() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    if (!Validate.hasCameraPermissions(mContext)) {
                        PermissionRefusedEvent event = new PermissionRefusedEvent(PermissionRefusedEvent.Permission.CAMERA);
                        mEventBus.post(event);
                        throw new PromisePermissionRefusedEventException(event);
                    }

                    if (null == media) {
                        Log.d(TAG, "startVideo: media is null");
                        throw new MediaEngineException("Media is null, invalid state");
                    }
                    if (isVideoOn()) {
                        Log.d("SDKMEDIA", "startVideo: already started... please wait if you wanted to stop");
                        solver.resolve(true);//new StartVideoAnswerEvent(false, true));
                        return;
                    }

                    media.startVideo(mDefaultCamera);

                    final Observable<StartVideoResponse> startVideo = mConferenceObservableProvider.getStartVideoObservable(mConferenceId, VoxeetPreferences.id());
                    startVideo.subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                            .subscribe(new Subscriber<StartVideoResponse>() {
                                @Override
                                public void onCompleted() {
                                }

                                @Override
                                public void onError(Throwable e) {
                                    mTwig.e(e);
                                    isVideoOn = false;

                                    StartVideoAnswerEvent event = new StartVideoAnswerEvent(false);
                                    mEventBus.post(event);
                                    solver.reject(e);
                                }

                                @Override
                                public void onNext(StartVideoResponse response) {
                                    //remove start video since it is not to be used
                                    //media.startVideo(mDefaultCamera);

                                    isVideoOn = true;

                                    final StartVideoAnswerEvent event = new StartVideoAnswerEvent(true);
                                    mEventBus.post(event);
                                    createVideoAnswer(response.getUserId(), response.getDescription(), response.getCandidates())
                                            .then(new PromiseExec<Boolean, Object>() {
                                                @Override
                                                public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    error.printStackTrace();
                                                    solver.resolve(true);
                                                }
                                            });
                                }
                            });
                    //explicitly keep the list of the different exception
                    //this methods throws
                } catch (PromisePermissionRefusedEventException exception) {
                    solver.reject(exception);
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                } catch (Exception exception) {
                    solver.reject(exception);
                }
            }
        });
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopVideo() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    String conferenceId = getCurrentConferenceId();

                    if (!isInConference()) {
                        Log.d(TAG, "stopVideo: not in conf");

                        if (null != media) media.stopVideo();
                        throw new NotInConferenceException();
                    }

                    if (null == media) {
                        Log.d(TAG, "startVideo: media is null");
                        throw new MediaEngineException("media is null");
                    }
                    //Validate.notNull(mConferenceId, "mConference Id");
                    //Validate.notNull(media, "media");

                    media.stopVideo();

                    final Observable<StopVideoResponse> stopVideo = mConferenceObservableProvider.getStopVideoObservable(conferenceId, VoxeetPreferences.id());
                    stopVideo.subscribeOn(Schedulers.io())
                            .observeOn(AndroidSchedulers.mainThread())
                            .subscribe(new Subscriber<StopVideoResponse>() {
                                @Override
                                public void onCompleted() {
                                }

                                @Override
                                public void onError(Throwable e) {
                                    getTwig().e(e);

                                    StopVideoAnswerEvent event = new StopVideoAnswerEvent(false);
                                    mEventBus.post(event);
                                    solver.reject(e);
                                }

                                @Override
                                public void onNext(StopVideoResponse response) {
                                    media.stopVideo();

                                    isVideoOn = false;

                                    final StopVideoAnswerEvent event = new StopVideoAnswerEvent(true);
                                    mEventBus.post(event);
                                    createVideoAnswer(response.getUserId(), response.getDescription(), response.getCandidates())
                                            .then(new PromiseExec<Boolean, Object>() {
                                                @Override
                                                public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                                    //TODO check for possible call in the createVideoAnswer (callback ?)
                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    error.printStackTrace();
                                                    //TODO check for possible call in the createVideoAnswer (callback ?)
                                                    solver.resolve(true);
                                                }
                                            });
                                }
                            });
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                } catch (Exception exception) {
                    solver.reject(exception);
                }
            }
        });
    }

    /**
     * Create a video answer and post it
     * <p>
     * resolve -> true -> everything went fine
     * resolve -> false -> on error occured
     * reject -> media exception
     *
     * @param userId
     * @param offerDescription
     * @param offerCandidates
     * @return
     */
    private Promise<Boolean> createVideoAnswer(final String userId, final OfferDescription offerDescription, final List<OfferCandidate> offerCandidates) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Log.d(TAG, "createVideoAnswer: " + userId + " " + offerDescription.getSdp() + " " + offerDescription.getType());
                Log.d("SDKMEDIA", "createVideoAnswer: " + userId + " " + offerDescription.getSdp() + " " + offerDescription.getType());

                try {
                    if (null == media) {
                        throw new MediaEngineException("media is null");
                    }
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                }

                SdpDescription description = new SdpDescription(offerDescription.getType(), offerDescription.getSdp());

                List<SdpCandidate> candidates = new ArrayList<>();
                if (offerCandidates != null) {
                    for (OfferCandidate candidate : offerCandidates) {
                        candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                    }
                }

                try {
                    media.createAnswerForPeer(userId,
                            description.getSsrc(),
                            description,
                            candidates,
                            VoxeetPreferences.id().equalsIgnoreCase(userId),
                            new PendingPeerCallback() {
                                @Override
                                public void onMessage(@Nullable SdpMessage message) {
                                    Log.d(TAG, "onMessage: having a message " + message);
                                    answer(userId, message)
                                            .then(new PromiseExec<Integer, Object>() {
                                                @Override
                                                public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                    Log.d(TAG, "onSuccess: " + result);

                                                    //resolve true value : all clear
                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    if (error instanceof PromiseParticipantAddedErrorEventException)
                                                        mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                    else
                                                        error.printStackTrace();

                                                    //resolve false value : something happened
                                                    solver.resolve(false);
                                                }
                                            });
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @return
     */
    @Override
    public Promise<Boolean> create() {
        return join(null);
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceAlias create a given conference then join it
     * @return the promise
     */
    @Override
    public Promise<Boolean> join(@Nullable final String conferenceAlias) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Log.d(TAG, "Attempting to join mConferene alias:=" + conferenceAlias);

                CreateConferenceParams params = new CreateConferenceParams()
                        .setStats(mEnableStats);
                if (null != conferenceAlias) params.setConferenceAlias(conferenceAlias);

                Observable<ConferenceResponse> observable = mConferenceObservableProvider.getCreateConferenceObservable(params);
                observable.subscribeOn(Schedulers.newThread())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<ConferenceResponse>() {
                            @Override
                            public void onCompleted() {
                            }

                            @Override
                            public void onError(Throwable e) {
                                e.printStackTrace();
                                mTwig.e(e);

                                setIsInConference(false);

                                closeMedia();

                                mEventBus.post(new ConferenceCreatedError(handleError(e)));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(ConferenceResponse response) {
                                Log.d(TAG, "onNext: having conference");
                                createOrSetConferenceWithParams(response.getConfId(), response.getConfAlias());

                                ConferenceInformation information = getCurrentConferenceInformation();
                                information.setConferenceType(ConferenceSimpleState.CONFERENCE);

                                Log.d(TAG, "onNext: join with := " + response.getConfId() + " " + response.getConfAlias());
                                solver.resolve(onConferenceCreated(response.getConfId(), response.getConfAlias(), null));
                            }
                        });
            }
        });
    }


    protected Promise<ResumeConference> extendedJoin(final String conferenceId) {
        return new Promise<>(new PromiseSolver<ResumeConference>() {
            @Override
            public void onCall(@NonNull final Solver<ResumeConference> solver) {

                Observable<ResumeConference> observable = mConferenceObservableProvider.joinConference(conferenceId, new DeviceEvent(DeviceType.ANDROID, isListenerMode()));
                observable.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<ResumeConference>() {

                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(ResumeConference resumeConference) {
                                solver.resolve(resumeConference);
                            }
                        });
            }
        });
    }


    protected Promise<Boolean> onConferenceCreated(@NonNull final String conferenceId,
                                                   @NonNull final String conferenceAlias,
                                                   @Nullable final NormalConferenceResponse other) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                createOrSetConferenceWithParams(conferenceId, conferenceAlias);

                getTwig().i("IConference created with id: " + mConferenceId);

                mEventBus.post(new ConferenceCreationSuccess(conferenceId, conferenceAlias));

                joinVoxeetConference(conferenceId).then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * resole -> true
     * reject -> network error or on created error
     *
     * @return
     */
    @Override
    public Promise<Boolean> demo() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                getTwig().i("Attempting to create demo mConference");

                final Observable<DemoEvent> user = mConferenceObservableProvider.getCreateDemoObservable();
                user.subscribeOn(Schedulers.newThread())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<DemoEvent>() {
                            @Override
                            public void onCompleted() {
                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e(e);

                                mEventBus.post(new ConferenceCreatedError(handleError(e)));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(final DemoEvent response) {
                                onCreateDemoSuccess(response)
                                        .then(new PromiseExec<Boolean, Object>() {
                                            @Override
                                            public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                                ConferenceInformation information = getCurrentConferenceInformation();
                                                information.setConferenceType(ConferenceSimpleState.CONFERENCE);

                                                solver.resolve(true);
                                            }
                                        })
                                        .error(new ErrorPromise() {
                                            @Override
                                            public void onError(Throwable error) {
                                                solver.reject(error);
                                            }
                                        });
                            }
                        });
            }
        });
    }

    /**
     * resolve -> list of ConferenceRefreshedUsers
     * reject -> network error
     *
     * @param ids a non null list of non null strings
     * @return a non null promise
     */
    @NonNull
    @Override
    public Promise<List<ConferenceRefreshedEvent>> invite(final List<String> ids) {
        return new Promise<>(new PromiseSolver<List<ConferenceRefreshedEvent>>() {
            @Override
            public void onCall(@NonNull final Solver<List<ConferenceRefreshedEvent>> solver) {
                //remove the timeout
                removeTimeoutCallbacks();

                //a new one is now sent
                sendTimeoutCallbacks();

                boolean sdk = isSDK();
                List<String> voxeetIds = sdk ? null : new ArrayList<String>();
                final List<String> externalIds = new ArrayList<>();

                externalIds.addAll(ids);
                if (!sdk) voxeetIds.addAll(ids);

                final rx.Observable<Response<ResponseBody>> user = mConferenceObservableProvider.getInviteObservable(mConferenceId, new SdkConferenceInvitation(voxeetIds, externalIds));
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {
                                //Do nothing
                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e(e);

                                mEventBus.post(new AddConferenceParticipantResultEvent(handleError(e), false));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                List<ConferenceRefreshedEvent> list = new ArrayList<>();
                                mEventBus.post(new AddConferenceParticipantResultEvent(response.code() == 200));

                                try {
                                    String body = response.body().string();
                                    Log.d(TAG, "onNext: " + body);
                                } catch (IOException e) {
                                    e.printStackTrace();
                                }

                                if (response.code() == 200) {
                                    for (String userId : externalIds) {
                                        if (!userId.equals(VoxeetPreferences.id())) {
                                            getTwig().i("Conference participant with id: " + userId + " invited");
                                            ConferenceRefreshedEvent event = new ConferenceRefreshedEvent(userId, updateConferenceParticipants(userId, ConferenceUserStatus.IN_PROGRESS));


                                            mEventBus.post(event);
                                            list.add(event);
                                        }
                                    }
                                }
                                solver.resolve(list);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false response from server is an error
     * reject -> network error
     *
     * @return
     */
    @Override
    public Promise<Boolean> logout() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                if (null != VoxeetPreferences.id()) {
                    mTwig.i("Attempting to logout");

                    final Observable<Response<ResponseBody>> user = mConferenceObservableProvider.getLogOutObservable(VoxeetPreferences.token());
                    user.subscribeOn(Schedulers.newThread())
                            .observeOn(AndroidSchedulers.mainThread())
                            .subscribe(new Subscriber<Response<ResponseBody>>() {
                                @Override
                                public void onCompleted() {
                                }

                                @Override
                                public void onError(Throwable e) {
                                    getTwig().e(e);
                                    String message = "Logout failed";
                                    getTwig().e(message);

                                    VoxeetPreferences.onLogout();
                                    getVoxeetSDK().closeSocket();

                                    mEventBus.post(new SdkLogoutErrorEvent(handleError(e)));
                                    solver.reject(e);
                                }

                                @Override
                                public void onNext(Response<ResponseBody> response) {
                                    String message;
                                    if (response.code() == 200) {
                                        message = "Logout success";

                                        getTwig().i(message);

                                        VoxeetPreferences.onLogout();
                                        getVoxeetSDK().closeSocket();

                                        mEventBus.post(new SdkLogoutSuccessEvent(message));
                                        solver.resolve(true);
                                    } else {
                                        message = "Logout failed";

                                        getTwig().e(message);

                                        VoxeetPreferences.onLogout();
                                        getVoxeetSDK().closeSocket();

                                        mEventBus.post(new SdkLogoutErrorEvent(message));
                                        solver.resolve(false);
                                    }
                                }
                            });
                } else {
                    final String message = "Already logged out";
                    getTwig().e(message);
                    mEventBus.post(new SdkLogoutSuccessEvent("Already logged out"));
                    solver.resolve(true);
                }
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false in case of error
     * reject -> dangerous issue
     *
     * @param conferenceId the mConference id
     * @param offset
     * @return
     */
    @Override
    public Promise<Boolean> replay(final String conferenceId, final long offset) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                isListenerMode = true;
                setListenerMode(false);

                final rx.Observable<ReplayConferenceEvent> user = mConferenceObservableProvider.getReplayObservable(conferenceId, new SdkConferenceReplayBody(offset));
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<ReplayConferenceEvent>() {
                            @Override
                            public void onCompleted() {
                                //Do nothing
                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Failed to replay mConference");

                                mEventBus.post(new ReplayConferenceErrorEvent(handleError(e)));
                                e.printStackTrace();
                                solver.resolve(false);
                            }

                            @Override
                            public void onNext(ReplayConferenceEvent response) {
                                getTwig().i("Success server answer for replaying mConference with id: " +
                                        conferenceId + "at offset: " + offset + " given := " + response.getConferenceId());

                                ConferenceInformation information = getConferenceInformation(response.getConferenceId());
                                information.setConferenceType(ConferenceSimpleState.REPLAY);

                                joinVoxeetConference(response.getConferenceId())
                                        .then(new PromiseExec<Boolean, Object>() {
                                            @Override
                                            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                                                solver.resolve(result);
                                            }
                                        })
                                        .error(new ErrorPromise() {
                                            @Override
                                            public void onError(Throwable error) {
                                                error.printStackTrace();
                                                solver.resolve(false);
                                            }
                                        });
                            }
                        });

            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false in case of non 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> startRecording() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(mConferenceId, "mConference Id");

                final Observable<Response<ResponseBody>> user = mConferenceObservableProvider.getStartRecordingObservable(mConferenceId);
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Failed to start mConference recording");

                                mEventBus.post(new StartRecordingResultEvent(handleError(e), false));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                if (response.code() == 200)
                                    getTwig().i("IConference recording started ");

                                mEventBus.post(new StartRecordingResultEvent(response.code() == 200));
                                try {
                                    Log.d(TAG, "onNext: " + response.body().string());
                                } catch (IOException e) {
                                    e.printStackTrace();
                                }
                                solver.resolve(response.code() == 200);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false not a 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> stopRecording() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(mConferenceId, "mConference Id");

                final Observable<Response<ResponseBody>> user = mConferenceObservableProvider.getStopRecordingObservable(mConferenceId);
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Failed to cancel mConference recording");

                                mEventBus.post(new StopRecordingResultEvent(handleError(e), false));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                if (response.code() == 200)
                                    getTwig().i("Conference recording stopped");

                                try {
                                    Log.d(TAG, "onNext: " + response.body().string());
                                } catch (IOException e) {
                                    e.printStackTrace();
                                }

                                mEventBus.post(new StopRecordingResultEvent(response.code() == 200));
                                solver.resolve(response.code() == 200);
                            }
                        });
            }
        });
    }

    /**
     * Get the current mConference status
     * send events but also managed by the promise
     * <p>
     * resolve -> GetConferenceStatusEvent
     * reject -> error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<GetConferenceStatusEvent> getConferenceStatus(final String conferenceId) {
        return new Promise<>(new PromiseSolver<GetConferenceStatusEvent>() {
            @Override
            public void onCall(@NonNull final Solver<GetConferenceStatusEvent> solver) {
                final Observable<GetConferenceStatusEvent> user = mConferenceObservableProvider.getConferenceStatusObservable(conferenceId);
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<GetConferenceStatusEvent>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Error while getting mConference history");

                                try {
                                    GetConferenceStatusErrorEvent event = new GetConferenceStatusErrorEvent(handleError(e));
                                    mEventBus.post(event);
                                    throw new PromiseGetConferenceStatusErrorEventException(event, e);
                                } catch (PromiseGetConferenceStatusErrorEventException exception) {
                                    solver.reject(exception);
                                }
                            }

                            @Override
                            public void onNext(GetConferenceStatusEvent response) {
                                for (DefaultConferenceUser conferenceUser : response.getConferenceUsers()) {
                                    getTwig().e(conferenceUser.getUserId() + " / " + conferenceUser.getStatus());
                                }
                                getTwig().i("Dispatching mConference status result");

                                mEventBus.post(response);
                                solver.resolve(response);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> event
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<GetConferenceHistoryEvent> conferenceHistory(final String conferenceId) {
        return new Promise<>(new PromiseSolver<GetConferenceHistoryEvent>() {
            @Override
            public void onCall(@NonNull final Solver<GetConferenceHistoryEvent> solver) {
                final Observable<List<HistoryConference>> user = mConferenceObservableProvider.getConferenceHistoryObservable(conferenceId);
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<List<HistoryConference>>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Error while retrieving mConference history");

                                mEventBus.post(new GetConferenceHistoryErrorEvent(handleError(e)));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(List<HistoryConference> response) {
                                getTwig().i("Dispatching mConference history response");

                                GetConferenceHistoryEvent event = new GetConferenceHistoryEvent(response);
                                mEventBus.post(event);
                                solver.resolve(event);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> none
     *
     * @return
     */
    @Override
    public Promise<Boolean> switchCamera() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(media, "media");

                media.switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean isFrontCamera) {
                        getTwig().i("Successfully switched camera");

                        mEventBus.post(new CameraSwitchSuccessEvent(isFrontCamera));
                        solver.resolve(true);
                    }

                    @Override
                    public void onCameraSwitchError(String errorDescription) {
                        getTwig().e("Failed to switch camera " + errorDescription);

                        mEventBus.post(new CameraSwitchErrorEvent(errorDescription));
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    @Override
    public Promise<Boolean> subscribe(final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final Observable<SubscribeConference> user = mConferenceObservableProvider.getSubscribeObservable(conferenceId);
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<SubscribeConference>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Failed to subscribe to this mConference's events");

                                mEventBus.post(new SubscribeConferenceErrorEvent(handleError(e)));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(SubscribeConference response) {
                                getTwig().i("You are no subscribed to this mConference's events");

                                mEventBus.post(new SubscribeConferenceEvent(response));
                                solver.resolve(true);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @return a valid promise
     */
    @Override
    public Promise<Boolean> unSubscribe() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                mConferenceObservableProvider.getUnSubscribeObservable(mConferenceId)
                        .subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Failed to unsubscribe from this mConference's events");

                                mEventBus.post(new UnSubscribeConferenceAnswerEvent(false, handleError(e)));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                getTwig().i("You are no longer subscribed to this mConference's events");

                                mEventBus.post(new UnSubscribeConferenceAnswerEvent(response.code() == 200));
                                solver.resolve(true);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return
     */
    @Override
    public Promise<Boolean> subscribeForCall(final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                mConferenceObservableProvider.getSubscribeForCallObservable(conferenceId)
                        .subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<SubscribeConference>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Failed to subscribe to this mConference's start");

                                mEventBus.post(new SubscribeForCallConferenceErrorEvent(conferenceId, handleError(e)));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(SubscribeConference response) {
                                getTwig().i("You are no subscribed to receive events when this conf starts");

                                mEventBus.post(new SubscribeForCallConferenceAnswerEvent(conferenceId));
                                solver.resolve(true);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return
     */
    @Override
    public Promise<Boolean> unSubscribeFromCall(final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final Observable<Response<ResponseBody>> user = mConferenceObservableProvider.getUnSubscribeFromCallObservable(conferenceId);
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Failed to unsubscribe from this mConference's start");

                                mEventBus.post(new UnsubscribeFromCallConferenceErrorEvent(conferenceId, handleError(e)));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                getTwig().i("You are no longer subscribed to this mConference's start");

                                mEventBus.post(new UnSubscribeFromConferenceAnswerEvent(conferenceId));
                                solver.resolve(true);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     *
     * @param conferenceId the mConference id
     * @return a promise to catch
     */
    @NonNull
    public Promise<Boolean> joinUsingConferenceId(@NonNull final String conferenceId) {
        return joinVoxeetConference(conferenceId);
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     *
     * @param conferenceId the mConference id
     * @return a promise to catch
     */
    @Deprecated
    @NonNull
    @Override
    public Promise<Boolean> joinVoxeetConference(@NonNull final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                if (mInConference) {
                    try {
                        throw new InConferenceException();
                    } catch (InConferenceException exception) {
                        solver.reject(exception);
                        return;
                    }
                }

                //remove the timeout, a new one should be started when user will be invited
                removeTimeoutCallbacks();

                Log.d(TAG, "joining " + conferenceId);

                setIsInConference(true);
                mConferenceId = conferenceId;

                initMedia();

                final Observable<ResumeConference> user = mConferenceObservableProvider.joinConference(conferenceId, new DeviceEvent(DeviceType.ANDROID, isListenerMode));
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<ResumeConference>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                try {
                                    e.printStackTrace();
                                    getTwig().e("Failed to Join mConference with id " + conferenceId);

                                    setIsInConference(false);

                                    closeMedia();

                                    ConferenceJoinedError error = new ConferenceJoinedError(handleError(e));
                                    mEventBus.post(error);
                                    throw new PromiseConferenceJoinedErrorException(error, e);
                                } catch (PromiseConferenceJoinedErrorException exception) {
                                    solver.reject(exception);
                                }
                            }

                            @Override
                            public void onNext(ResumeConference response) {
                                createOrSetConferenceWithParams(response.getConferenceId(),
                                        response.getConferenceAlias());

                                onConferenceResumedInternal(response, solver);
                            }
                        });
            }
        });
    }

    protected void onConferenceResumedInternal(ResumeConference response, Solver<Boolean> solver) {
        mConferenceId = response.getConferenceId();

        ConferenceInformation information = getCurrentConferenceInformation();

        //information should not be null in this case

        DefaultConference conference = information.getConference();

        conference.setConferenceId(response.getConferenceId());
        conference.setConferenceAlias(response.getConferenceId());

        if (response.getConferenceAlias() != null) {
            conference.setConferenceAlias(response.getConferenceAlias());
        }

        List<DefaultParticipant> participants = response.getParticipants();

        if (null != participants) {
            Map<String, String> cache = information.getUserIdsCached();
            for (DefaultParticipant participant : participants) {
                cache.put(participant.getExternalId(), participant.getExternalId());
            }
        }

        setIsInConference(true);

        setAudioRoute(AudioRoute.ROUTE_PHONE);

        getTwig().i("Joined mConference with id " + getConferenceId());


        mEventBus.post(new ConferencePreJoinedEvent(getConferenceId(), getAliasId()));

        mEventBus.post(new ConferenceJoinedSuccessEvent(getConferenceId(), getAliasId()));
        solver.resolve(true);
    }

    /**
     * resolve -> the event with the list
     * reject -> any error
     *
     * @return
     */
    @Override
    public Promise<ConferenceUsersInvitedEvent> getInvitedUsers() {
        return new Promise<>(new PromiseSolver<ConferenceUsersInvitedEvent>() {
            @Override
            public void onCall(@NonNull final Solver<ConferenceUsersInvitedEvent> solver) {
                final Observable<GetConferenceStatusEvent> user = mConferenceObservableProvider.getConferenceStatusObservable(mConferenceId);
                user.subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<GetConferenceStatusEvent>() {
                            @Override
                            public void onCompleted() {

                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Error while retrieving connecting users");

                                mEventBus.post(new ConferenceUsersInvitedEvent(null));
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(GetConferenceStatusEvent response) {
                                Iterable<DefaultConferenceUser> users = Iterables.filter(response.getConferenceUsers(), new Predicate<DefaultConferenceUser>() {
                                    @Override
                                    public boolean apply(DefaultConferenceUser input) {
                                        return input.getStatus().equalsIgnoreCase(ConferenceUserStatus.CONNECTING.toString());
                                    }
                                });

                                mTwig.i("Successfully retrieved connecting users");

                                ConferenceUsersInvitedEvent event = new ConferenceUsersInvitedEvent(users);
                                mEventBus.post(event);
                                solver.resolve(event);
                            }
                        });
            }
        });
    }

    @Override
    public Promise<Boolean> sendBroadcastMessage(@NonNull final String message) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    if (null == mConferenceId) {
                        throw new NotInConferenceException();
                    }
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                    return;
                }

                final Observable<Response<ResponseBody>> user = mConferenceObservableProvider.broadcastMessage(mConferenceId, new BroadcastEvent(message));
                user.subscribeOn(Schedulers.newThread())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {
                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("The broadcast message has not been sent");

                                mEventBus.post(new SendBroadcastResultEvent(handleError(e), false));
                                solver.resolve(false);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                getTwig().i("The broadcast message has been sent");

                                mEventBus.post(new SendBroadcastResultEvent(response.code() == 200));
                                solver.resolve(true);
                            }
                        });
            }
        });
    }

    /**
     * This method attempts to leave the current mConference
     * <p>
     * The previous version used a boolean, the new method fingerprint is now void
     */
    @Override
    public Promise<Boolean> leave() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                //TODO the conference in the request must be set here to prevent overwrite call
                //remove the timeout, a new one should be started when user will be invited
                //in the next conferece
                //removeTimeoutCallbacks();
                String conferenceId = mConferenceId;

                getTwig().i("Attempting to leave mConference with mConference id " + conferenceId);

                //TODO proper close call
                //FIX the issue with SurfaceTexture and overlay
                closeMedia();

                final Observable<Response<ResponseBody>> user = mConferenceObservableProvider.leaveConference(conferenceId);
                user.subscribeOn(Schedulers.newThread())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {

                            private boolean success = false;

                            @Override
                            public void onCompleted() {
                            }

                            @Override
                            public void onError(Throwable e) {
                                e.printStackTrace();
                                removeTimeoutCallbacks();

                                getTwig().e("Something went wrong while leaving the mConference");

                                closeMedia();

                                if (!success) {
                                    mEventBus.post(new ConferenceLeftError(handleError(e)));
                                }
                                solver.resolve(false);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                removeTimeoutCallbacks();

                                getTwig().i("Conference left successfully");

                                int leftInConference = 0;
                                DefaultConference conference = getConference();
                                if (conference != null) {
                                    leftInConference = getConference().getConferenceUsers().size() - 1;
                                }

                                closeMedia();

                                if (response.code() == 200 || response.code() == 404) {
                                    mEventBus.post(new ConferenceLeftSuccessEvent(leftInConference));
                                    success = true; //when having 404, the above onError is sent !
                                    solver.resolve(true);
                                } else {
                                    mEventBus.post(new ConferenceLeftError(response.code() + ""));
                                    solver.resolve(false);
                                }
                            }
                        });
            }
        });
    }


    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Event management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceStatsEvent event) {
        ConferenceStats stats = event.getEvent();
        if (stats.getConference_id() != null && stats.getConference_id().equals(getConferenceId())) {
            float mos = stats.getScore(VoxeetPreferences.id());

            mEventBus.post(new QualityIndicators(mos));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUpdatedEvent event) {
        //TODO maybe manage users here too ?
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceived invitation) {
        try {
            onEvent(invitation.getEvent());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceivedEvent invitation) {

        String conferenceId = invitation.getConferenceId();
        if (null == conferenceId) conferenceId = invitation.getConferenceId();

        if (null == conferenceId && null != invitation.getConference()) {
            conferenceId = invitation.getConference().getConferenceId();
        }

        Log.d(TAG, "onEvent: current mConferenceId " + mConferenceId + " vs " + invitation.getConferenceId() + " vs " + conferenceId);

        //we update the information for this conference only
        final ConferenceInformation information = getConferenceInformation(conferenceId);

        DefaultConference conference = invitation.getConference();
        boolean is_own_user = invitation.getUserId().equals(VoxeetPreferences.id());
        ConferenceType type = ConferenceType.fromId(conference.getConferenceType());

        if (!is_own_user || ConferenceType.SCHEDULED.equals(type)) {
            InvitationReceivedEvent.UserInviter inviter = invitation.getInviter();
            List<DefaultInvitation> invitations = invitation.getInvitations();

            if (null == information) {
                Log.d(TAG, "onEvent: INVALID INFORMATION FOR THE CONFERENCE");
            }

            if (null != inviter && null != inviter.externalId && null != inviter.userId) {
                safeUserManagementStarted();
                Log.d(TAG, "onEvent: Invitation with inviter " + inviter.userId + " " + inviter.externalId);
                UserInfo info = createUserInfo(inviter.nickName, inviter.externalId, inviter.externalAvatarUrl);

                DefaultConferenceUser inviter_user = new DefaultConferenceUser(inviter.userId,
                        "", info);
                information.getLastInvitationReceived().add(inviter_user);
                information.getUserIdsCached().put(inviter.userId, inviter.externalId);
                safeUserManagementEnded();
            } else {
                Log.d(TAG, "onEvent: Invitation with invalid inviter");
            }

            final String finalConferenceId = conferenceId;
            getConferenceStatus(conference.getConferenceId())
                    .then(new PromiseExec<GetConferenceStatusEvent, Object>() {
                        @Override
                        public void onCall(@Nullable GetConferenceStatusEvent result, @NonNull Solver<Object> internal_solver) {
                            Log.d(TAG, "onSuccess: " + result);

                            //removed the check for live Event since now it is able to have multiple

                            if (result.getConferenceUsers().size() > 0) {
                                Log.d(TAG, "onEvent: users " + result.getConferenceUsers().size());

                                DefaultConferenceUser foundUser = null;
                                String foundExternalId = null;

                                List<DefaultConferenceUser> merged_list = new ArrayList<>();
                                //add every user from the mConference
                                merged_list.addAll(result.getConferenceUsers());
                                //add every use from the invitation
                                merged_list.addAll(getLastInvitationUsers());

                                Iterable<DefaultConferenceUser> list = Iterables.filter(merged_list, new Predicate<DefaultConferenceUser>() {
                                    @Override
                                    public boolean apply(@Nullable DefaultConferenceUser input) {
                                        return !currentUserOrEmpty().equals(input.getUserId());
                                    }
                                });

                                for (DefaultConferenceUser conferenceUser : list) {
                                    String userId = conferenceUser.getUserId();
                                    if (null == foundExternalId) {
                                        String externalId = conferenceUser.getUserInfo().getExternalId();
                                        String cachedExternalId = information.getUserIdsCached().get(userId);
                                        Log.d(TAG, "onEvent: " + userId + " " + externalId + " " + cachedExternalId);

                                        foundUser = conferenceUser;
                                        if (null != userId && (null == externalId || userId.equals(externalId))) {
                                            //should be different than null
                                            externalId = cachedExternalId;
                                            if (null != externalId) {
                                                foundExternalId = cachedExternalId;
                                            }
                                        }
                                    }
                                }

                                if (foundUser != null && null == foundExternalId) {
                                    foundExternalId = foundUser.getUserId();
                                    Log.d(TAG, "externalId is null, setting it to userId");
                                }

                                Map<String, String> infos = new HashMap<>();

                                infos.put(VoxeetIntentFactory.INVITER_ID, foundUser.getUserId());
                                infos.put(VoxeetIntentFactory.INVITER_NAME, foundUser.getUserInfo().getName());
                                infos.put(VoxeetIntentFactory.NOTIF_TYPE, result.getType());
                                infos.put(VoxeetIntentFactory.INVITER_EXTERNAL_ID, foundExternalId);
                                infos.put(VoxeetIntentFactory.INVITER_URL, foundUser.getUserInfo().getAvatarUrl());
                                infos.put(VoxeetIntentFactory.CONF_ID, result.getConferenceId());


                                getTwig().i("Starting default incoming view " + finalConferenceId + " " + result.getConferenceId());

                                Intent intent = VoxeetIntentFactory.buildFrom(mContext, VoxeetPreferences.getDefaultActivity(), infos);
                                if (intent != null)
                                    mContext.startActivity(intent);

                                mEventBus.post(new IncomingCallEvent());
                            }
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(Throwable error) {
                            error.printStackTrace();
                        }
                    });
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(OwnConferenceStartedEvent ownConferenceStartedEvent) {
        if (null != ownConferenceStartedEvent.event().getConferenceInfos()) {
            mConferenceId = ownConferenceStartedEvent.event().getConferenceInfos().getConferenceId();
        }

        DefaultConference conference = getConference();
        getTwig().i("Own mConference started :: " + mConferenceId + " " + conference);

        conference.setConferenceInfos(ownConferenceStartedEvent.event().getConferenceInfos());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final StartVideoAnswerEvent event) {
        Log.d(TAG, "onEvent: start video event " + event.isSuccess() + " " + event.message());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(RenegociationUpdate event) {
        RenegociationEndedEvent renego = event.getEvent();

        Log.d(TAG, "onEvent: Renegociation " + renego.getConferenceId()
                + " " + renego.getType() + " " + renego.getAnswerReceived()
                + " " + renego.getOfferSent());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final OfferCreatedEvent event) {
        Log.d(TAG, "onEvent: OfferCreatedEvent " + event.message());
        try {
            handleAnswer(event.offer())
                    .then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onCall: answer called, result is " + aBoolean + " " + event.offer().getUserId() + " " + VoxeetPreferences.id());
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(Throwable error) {
                            error.printStackTrace();
                        }
                    });
        } catch (MediaEngineException e) {
            e.printStackTrace();
            getTwig().e(e);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserAddedEvent event) {
        if (!mConferenceId.equals(event.getEvent().getConferenceId())) {
            Log.d(TAG, "onEvent: USER ADDED FOR ANOTHER CONFERENCE");
        }

        DefaultConference conference = getOrSetConference(event.getEvent().getConferenceId());
        ConferenceUserAdded content = event.getEvent();

        if (null == conference) {
            Log.d(TAG, "onEvent: ERROR THE CONFERENCE IS NULL, PLEASE CHECK WORKFLOW");
            createOrSetConferenceWithParams(event.getEvent().getConferenceId(),
                    event.getEvent().getConferenceAlias());
            conference = getConference();
        }

        if (null != content) {
            Log.d(TAG, "onEvent: ParticipantAdded " + content + " " + content.getUserId() + " " + conference.getConferenceUsers().size());

            DefaultConferenceUser user = findUserById(event.getEvent().getUserId());

            if (null == user) {
                user = new DefaultConferenceUser(content.getUserId(), null);
                user.setIsOwner(false);
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));

                conference.getConferenceUsers().add(user);
            }

            if (null == user.getUserInfo()) {
                Log.d(TAG, "onEvent: existing user info are invalid, refreshing");
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));
            }

            ConferenceUserStatus status = ConferenceUserStatus.fromString(content.getStatus());
            if (null != status) {
                user.setConferenceStatus(status);
                user.setStatus(content.getStatus());
            }

            mTwig.i("from Event ConferenceUserAddedEvent, user joined with id := " + user.getUserId() + " " + (user.getUserInfo() != null ? user.getUserInfo().getExternalId() : "no external id"));

            MediaStream stream = null;
            if (null != media) {
                HashMap<String, MediaStream> streams = getMapOfStreams();
                if (streams.containsKey(user.getUserId())) {
                    stream = streams.get(user.getUserId());
                }
            }

            mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
        }

        //in a previous version in the v1.0, the createOffer was forced right here
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantUpdatedEvent event) {
        DefaultConferenceUser user = findUserById(event.getUserId());
        final ConferenceUserStatus status = ConferenceUserStatus.valueOf(event.getStatus());

        if (user != null) {
            startTransactionConferenceUser();
            user.setStatus(event.getStatus());
            commitTransactionConferenceUser();
        } else {
            getTwig().i("Not in mConference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
        }

        switch (status) {
            case CONNECTING:
                if (null != user && !user.getUserId().equals(VoxeetPreferences.id())) {
                    mTwig.i("Cancelling timeout timer from user connecting");
                    removeTimeoutCallbacks();
                }
                break;
            case LEFT:
            case DECLINE:
                if (media != null && user != null) {
                    getTwig().i("In mConference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
                    media.removePeer(event.getUserId());
                }
                break;
            default:
                getTwig().i("status not managed updated to " + event.getStatus());

        }

        //check if the timeout runnable was not triggered
        boolean timeout_managed = timeoutRunnable != null && timeoutRunnable.isTriggered();

        if (!timeout_managed) {
            switch (status) {
                case DECLINE:
                    if (!event.getUserId().equals(VoxeetPreferences.id())) {
                        getTwig().i("Conference user with id: " + event.getUserId() + " declined the call");

                        mEventBus.post(new ConferenceUserCallDeclinedEvent(event.getConfId(), event.getUserId(), event.getStatus()));
                    }
                    break;
                case LEFT:
                    onUserLeft(event.getUserId());
                    break;
                default:
                    MediaStream stream = null;
                    if (null != media && null != user) {
                        HashMap<String, MediaStream> streams = getMapOfStreams();
                        if (streams.containsKey(user.getUserId())) {
                            stream = streams.get(user.getUserId());
                        }
                    }

                    if (null != user) {
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    }
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantAddedEvent event) {
        DefaultConferenceUser user = event.getUser();

        if (user != null) {
            mTwig.i("Conference user joined with id: " + user.getUserId());

            MediaStream stream = null;
            if (null != media) {
                HashMap<String, MediaStream> streams = getMapOfStreams();
                if (streams.containsKey(user.getUserId())) {
                    stream = streams.get(user.getUserId());
                }
            }

            mEventBus.post(new ConferenceUserJoinedEvent(user, stream));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceDestroyedPushEvent event) {
        if (null == mConferenceId || mConferenceId.equals(event.getPush().getConferenceId())) {
            closeMedia();

            getTwig().i("Conference has ended");
            mEventBus.post(event.getPush());
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceEndedEvent event) {
        if (null == mConferenceId || mConferenceId.equals(event.getEvent().getConferenceId())) {
            closeMedia();

            getTwig().i("Conference has ended");

            mEventBus.post(event.getEvent());
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final QualityUpdatedEvent event) {
        for (DefaultConferenceUser conferenceUser : event.getEvent().getUser()) {
            DefaultConferenceUser user = findUserById(conferenceUser.getUserId());
            if (user != null) {
                mTwig.i("Quality updated for " + user.getUserId());

                startTransactionConferenceUser();
                user.setQuality(conferenceUser.getQuality());
                commitTransactionConferenceUser();

                mEventBus.post(new ConferenceUserQualityUpdatedEvent(user));
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final RecordingStatusUpdate event) {
        String conferenceId = event.getEvent().getConferenceId();
        Log.d(TAG, "onEvent: eevnt for " + conferenceId);

        DefaultConference conference = getConferenceInformation(conferenceId).getConference();
        RecordingStatusUpdateEvent recordingStatusUpdateEvent = event.getEvent();
        DefaultConferenceUser user = findUserById(recordingStatusUpdateEvent.getUserId());
        if (user != null) {
            RecordingStatus status = RecordingStatus.valueOf(recordingStatusUpdateEvent.getRecordingStatus());
            if (status == RecordingStatus.RECORDING) {
                isRecording = true;
                conference.setStartRecordTimestamp(new Date(recordingStatusUpdateEvent.getTimeStamp()));
                conference.setRecordingStatus(RecordingStatus.RECORDING);
                conference.setRecordingUser(recordingStatusUpdateEvent.getUserId());
                user.setIsRecordingOwner(true);
            } else {
                isRecording = false;
                conference.setStartRecordTimestamp(null);
                conference.setRecordingStatus(RecordingStatus.NOT_RECORDING);
                conference.setRecordingUser(null);
                user.setIsRecordingOwner(false);
            }
        }

        getTwig().i("Conference's recording status has changed to: " + event.getEvent().getRecordingStatus());

        mEventBus.post(recordingStatusUpdateEvent);
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Protected management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    //new DefaultConference()
    //new DefaultConferenceRealm()
    protected abstract DefaultConference createConference();

    //mConference.getConferenceUsers().add(new DefaultConferenceUser(offer.getUserId(), offer.getDevice(), new UserInfo("Sophie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png")));
    protected abstract DefaultConferenceUser createConferenceUser(String userId, String device, UserInfo userInfo);

    protected abstract UserInfo createUserInfo(String userName, String externalId, String avatarUrl);

    protected Twig getTwig() {
        return mTwig;
    }

    protected void setConference(@NonNull DefaultConference conference) {
        mConferenceId = conference.getConferenceId();
        ConferenceInformation information = getCurrentConferenceInformation();
        information.setConference(conference);
    }

    protected void setConferenceAlias(String alias) {
        DefaultConference conference = getConference();
        conference.setConferenceAlias(alias);
        //null ? throw error B)
    }

    protected EventBus getEventBus() {
        return mEventBus;
    }

    protected ConferenceListener getConferenceListener() {
        return mListener;
    }

    protected MediaSDK getMedia() {
        return media;
    }

    protected abstract void startTransactionConferenceUser();

    protected abstract void commitTransactionConferenceUser();

    /**
     * Return true if the current implementation is SDK
     * false otherwise
     *
     * @return the state of this implementation
     */
    protected abstract boolean isSDK();

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected Promise<Integer> answer(final String peer, final SdpMessage message) {
        return new Promise<>(new PromiseSolver<Integer>() {
            @Override
            public void onCall(@NonNull final Solver<Integer> solver) {
                try {
                    if (null == mConferenceId) {
                        throw new NotInConferenceException();
                    }
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                    return;
                }

                Log.d("SDKMEDIA", "answer: peer := " + peer + " message := " + message);

                final Observable<Response<ResponseBody>> user = mConferenceObservableProvider.answerConference(mConferenceId, peer, message.getDescription());
                user.subscribeOn(Schedulers.newThread())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {
                            }

                            @Override
                            public void onError(Throwable e) {
                                getTwig().e("Offer created for " + peer + " : KO");

                                try {
                                    ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(handleError(e));
                                    throw new PromiseParticipantAddedErrorEventException(event);
                                } catch (PromiseParticipantAddedErrorEventException exception) {
                                    solver.reject(exception);
                                }
                            }

                            @Override
                            public void onNext(Response<ResponseBody> response) {
                                if (response.code() != 200) {
                                    getTwig().i("Offer created for " + peer + " : KO");

                                    try {
                                        ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(response.code() + "");
                                        throw new PromiseParticipantAddedErrorEventException(event);
                                    } catch (PromiseParticipantAddedErrorEventException exception) {
                                        solver.reject(exception);
                                    }
                                } else {
                                    getTwig().i("Offer created for " + peer + " : OK");
                                    solver.resolve(response.code());
                                }
                            }
                        });
            }
        });
    }


    /**
     * resolve -> true everything went fine
     * resolve -> false an error occured
     * reject -> media exception
     *
     * @param offer
     * @return a promise with the result of the call
     * @throws MediaEngineException
     */
    protected Promise<Boolean> handleAnswer(final OfferCreated offer) throws MediaEngineException {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                if (null == media) {
                    try {
                        throw new MediaEngineException("handleAnswer media is null");
                    } catch (MediaEngineException exception) {
                        solver.reject(exception);
                    }
                }


                final String conferenceId = offer.getConferenceId();

                if (!conferenceId.equals(mConferenceId)) {
                    Log.d(TAG, "onCall: CONFERENCE IS NOT THE SAME ! ANSWER SHALL BE DISCARDED");
                }

                SdpDescription description = new SdpDescription(offer.getDescription().getType(), offer.getDescription().getSdp());

                List<SdpCandidate> candidates = new ArrayList<>();
                for (OfferCandidate candidate : offer.getCandidates()) {
                    candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                }

                String userId = offer.getUserId();//isSDK() ? offer.getExternalId() : offer.getUserId();
                Log.d(TAG, "handleAnswer: offer := " + offer.getUserId() + " " + offer.getExternalId());
                Log.d("SDKMEDIA", "handleAnswer: " + offer.getUserId() + " " + offer.getExternalId() + " " + description.getSsrc());

                try {
                    media.createAnswerForPeer(userId,
                            description.getSsrc(),
                            description,
                            candidates,
                            offer.isMaster(),
                            new PendingPeerCallback() {
                                @Override
                                public void onMessage(@Nullable SdpMessage message) {
                                    try {
                                        if (null == mConferenceId) {
                                            Log.d(TAG, "onMessage: INVALID CONFERENCE ID WHEN OFFER IS RECEIVED");
                                            mConferenceId = offer.getConferenceId();
                                        }
                                        DefaultConference conference = getConference();

                                        DefaultConferenceUser user = null;

                                        for (DefaultConferenceUser in_conf : conference.getConferenceUsers()) {
                                            if (in_conf.getUserId() != null && in_conf.getUserId().equals(offer.getUserId())) {
                                                user = in_conf;
                                            }
                                        }

                                        UserInfo infos;
                                        if (offer.getUserId().contains("11111"))
                                            infos = createUserInfo("Julie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_1.png");
                                        else if (offer.getUserId().contains("22222"))
                                            infos = createUserInfo("Sophie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png");
                                        else if ((offer.getUserId().contains("33333")))
                                            infos = createUserInfo("Mike", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_3.png");
                                        else
                                            infos = createUserInfo(offer.getName(), offer.getExternalId(), offer.getAvatarUrl());

                                        if (user == null) {
                                            user = createConferenceUser(offer.getUserId(), offer.getDevice(), infos);
                                            conference.getConferenceUsers().add(user);
                                        } else {
                                            user.setUserInfo(infos);
                                        }


                                        setUserPosition(offer.getUserId(), 0, 0.5);

                                        answer(offer.getUserId(), message)
                                                .then(new PromiseExec<Integer, Object>() {
                                                    @Override
                                                    public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                        Log.d(TAG, "onSuccess: " + result);

                                                        solver.resolve(true);
                                                    }
                                                })
                                                .error(new ErrorPromise() {
                                                    @Override
                                                    public void onError(Throwable error) {
                                                        if (error instanceof PromiseParticipantAddedErrorEventException)
                                                            mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                        else
                                                            error.printStackTrace();

                                                        solver.resolve(false);
                                                    }
                                                });
                                    } catch (Exception e) {
                                        Log.d(TAG, "onMessage: unlockPeerOperation" + e.getMessage());
                                        solver.resolve(false);
                                    }
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }
    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected void safeUserManagementStarted() {

    }

    protected void safeUserManagementEnded() {

    }

    protected void safeConferenceManagementStarted() {

    }

    protected void safeConferenceManagementEnded() {

    }

    protected DefaultConferenceUser updateConferenceParticipants(String userId, ConferenceUserStatus status) {
        DefaultConferenceUser user = findUserById(userId);
        DefaultConference conference = getConference();

        if (user != null) {
            startTransactionConferenceUser();
            user.setIsOwner(conference.getOwnerProfile() == null || conference.getOwnerProfile().getUserId().equals(user.getProfile().getUserId()))
                    .setConferenceStatus(status)
                    .setQuality(ConferenceQuality.ULTRA_HIGH.StringValue());
            commitTransactionConferenceUser();

            return user;
        }
        return null;
    }

    protected void initMedia() {
        Validate.notNull(mContext, "mContext");
        Validate.notNull(VoxeetPreferences.id(), "user id");

        if (media == null) {
            try {
                media = new MediaSDK(context, VoxeetPreferences.id(), mediaStreamListener, isVideoOn(), Validate.hasMicrophonePermissions(context));
                getAudioService().setSpeakerMode(isDefaultOnSpeaker);
                if (!Validate.hasMicrophonePermissions(context)) {
                    getTwig().i("the app does not seem to have mic permission, disabling mic");
                    mute(true);
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                }
            } catch (MediaEngineException e) {
                getTwig().e(e);
            }
        }
    }

    protected void closeMedia() {
        if (media != null) {
            Conference conference = getConference();
            try {
                if (null != conference) {
                    for (DefaultConferenceUser user : conference.getConferenceUsers())
                        if (user != null && user.getUserId() != null)
                            media.removePeer(user.getUserId());
                }
            } catch (Exception ex) {
                mTwig.e(ex);
            }
            media.stop();
            media = null;

            getTwig().i("Microphone value restored to true");

            isVideoOn = false;
            mTwig.i("Video value restored to false");

            isListenerMode = false;
            getTwig().i("Listener mode set back to false");
        }

        getTwig().i("Media closed");
        setIsInConference(false);
        mConferenceId = null;
    }

    /**
     * Remove any previous timeout callback
     * <p>
     * To be used when it is mandatory to remove such behaviour
     * for instance, when joining a new mConference
     * <p>
     * It happens since we do not want to manage every edge case :
     * - decline
     * - network error
     * - etc...
     */
    private void removeTimeoutCallbacks() {
        if (timeoutRunnable != null) {
            timeoutRunnable.setCanceled(true);
            handler.removeCallbacks(timeoutRunnable);
        }
    }

    private void sendTimeoutCallbacks() {
        if (mTimeOutTimer != -1) {

            timeoutRunnable = new TimeoutRunnable(this,
                    getTwig(),
                    mEventBus,
                    mTimeOutTimer);

            getTwig().i("scheduling timer to leave the mConference in " + mTimeOutTimer);

            handler.postDelayed(timeoutRunnable, mTimeOutTimer);
        }
    }

    private void onUserLeft(@NonNull String peer) {
        DefaultConferenceUser user = findUserById(peer);
        if (user != null) {
            startTransactionConferenceUser();
            user.setConferenceStatus(ConferenceUserStatus.LEFT);

            if (getConferenceUsers().contains(user)) getConferenceUsers().remove(user);

            getTwig().i("Conference user left with id: " + user.getUserId());
            commitTransactionConferenceUser();

            if (mapOfStreams.containsKey(user.getUserId())) {
                mapOfStreams.remove(user.getUserId());
            }

            if (mapOfScreenShareStreams.containsKey(user.getUserId())) {
                mapOfScreenShareStreams.remove(user.getUserId());
            }

            mEventBus.post(new ConferenceUserLeftEvent(user));
        }
    }

    protected VoxeetSdkTemplate getVoxeetSDK() {
        return mSDK;
    }

    public void setStatEnabled(boolean stats) {
        mEnableStats = stats;
    }

    private AudioService getAudioService() {
        return getVoxeetSDK().getAudioService();
    }

    private void createOrSetConferenceWithParams(@NonNull String conferenceId, @Nullable String conferenceAlias) {
        mConferenceId = conferenceId;
        Log.d(TAG, "createOrSetConferenceWithParams: set conference id := " + mConferenceId);
        DefaultConference conference = getCurrentConferenceInformation().getConference();

        conference.setConferenceId(mConferenceId);
        conference.setConferenceAlias(conferenceAlias);
    }

    @Nullable
    private ConferenceInformation getCurrentConferenceInformation() {
        return getConferenceInformation(mConferenceId);
    }

    private ConferenceInformation getConferenceInformation(@Nullable String conferenceId) {
        if (null == conferenceId) {
            Log.d(TAG, "getConferenceInformation: CAN NOT GET CONFERENCE INFORMATION");
            return null;
        }
        return mConferenceInformationHolder.getInformation(conferenceId);
    }

}
