package voxeet.com.sdk.core.preferences;

import android.accounts.Account;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.provider.Settings.Secure;
import android.support.annotation.NonNull;
import android.util.Log;

import com.voxeet.kernel.R;

import java.util.Date;

/**
 * The type Voxeet preferences.
 */
public class VoxeetPreferences {

    private final static String TAG = VoxeetPreferences.class.getSimpleName();

    public final static String ACCOUNT_TYPE = "acct_type";
    /**
     * The constant USER_ID.
     */

    public final static String ID = "id";
    /**
     * The constant ACCESS_TOKEN.
     */
    public final static String ACCESS_TOKEN = "access_token";

    /**
     * The constant UPLOAD_TOKEN.
     */
    public final static String UPLOAD_TOKEN = "upload_token";

    /**
     * The constant LOGIN.
     */

    public final static String LOGIN = "login";

    /**
     * The constant COOKIE.
     */
    public final static String COOKIE = "cookie";

    /**
     * The constant PASSWORD.
     */
    public final static String PASSWORD = "password";

    /**
     * The constant MINIMAL_VERSION.
     */
    public final static String MINIMAL_VERSION = "minimal_version";

    /**
     * The constant APP_VERSION.
     */
    public final static String APP_VERSION = "app_version";

    /**
     * The constant REFERRAL_ENABLED.
     */
    public final static String REFERRAL_ENABLED = "referral_enabled";

    /**
     * The constant REFERRAL_URL.
     */
    public final static String REFERRAL_URL = "referral_url";

    /**
     * The constant REFERRAL_IMAGE_URL.
     */
    public final static String REFERRAL_IMAGE_URL = "referral_image_url";

    /**
     * The constant FACEBOOK_ENABLED.
     */
    public final static String FACEBOOK_ENABLED = "facebook_enabled";

    /**
     * The constant FACEBOOK_CONNECTED.
     */
    public final static String FACEBOOK_CONNECTED = "facebook_connected";

    /**
     * The constant AUTOLOGIN.
     */
    public final static String AUTOLOGIN = "autologin";

    /**
     * The constant PREMIUM.
     */
    public final static String PREMIUM = "premium";

    /**
     * The constant GOOGLE_CONNECTED.
     */
    public final static String GOOGLE_CONNECTED = "google_connected";

    /**
     * The constant ADDRESS_BOOK_SYNCED.
     */
    public final static String ADDRESS_BOOK_SYNCED = "adress_book_synced";

    /**
     * The constant DEFAULT_ACTIVITY to be used when clicking on a notification.
     */
    public final static String DEFAULT_ACTIVITY = "default_activity";

    /**
     * The constant MANIFEST_URL.
     */
    public final static String MANIFEST_URL = "manifest_url";

    /**
     * The constant CONFERENCE TIMEOUT.
     */
    public final static String CONFERENCE_TIMEOUT = "conference_timeout";

    /**
     * The constant DELAY_ESTIMATION.
     */
    public final static String DELAY_ESTIMATION = "delay_estimation_new";

    private static IPreferencesProvider preferences;

    private static String deviceIdentifier;


    private VoxeetPreferences() {

    }

    /**
     * Init.
     *
     * @param context the context
     */
    public static void init(@NonNull Context context) {
        SharedPreferences prefs = context.getSharedPreferences("main", Context.MODE_PRIVATE);

        deviceIdentifier = Secure.getString(context.getContentResolver(), Secure.ANDROID_ID);

        setCustomPreferencesProvider(context, new SharedPreferencesCacheWrapper(prefs));
    }

    public static void setCustomPreferencesProvider(@NonNull Context context,
                                                    @NonNull IPreferencesProvider provider) {

        int version = 0;
        try {
            version = context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionCode;
        } catch (PackageManager.NameNotFoundException e) {
            Log.e(TAG, "Error accessing application version number", e);
        }

        preferences = provider;
        preferences.edit()
                .putString(MANIFEST_URL, context.getString(R.string.manifestUrl))
                .putInt(APP_VERSION, version)
                .apply();

    }


    /**
     * user id string.
     *
     * @return the string
     */
    public static String id() {
        if (preferences != null && preferences.contains(ID)) {
            return preferences.getString(ID, null);
        } else {
            return null;
        }
    }

    /**
     * Current Account Type.
     *
     * Note that in the SDK, the account type is not used
     * Fallback is SDK
     *
     * @return the current account type
     */
    @NonNull
    public static AccountType accountType() {
        if (preferences != null && preferences.contains(ACCOUNT_TYPE)) {
            Log.d(TAG, "accountType: " + preferences.getString(ACCOUNT_TYPE, null));

            return AccountType.fromName(preferences.getString(ACCOUNT_TYPE, null));
        } else {
            return AccountType.SDK;
        }
    }

    /**
     * Is logged in boolean.
     *
     * @return the boolean
     */
    public static boolean isLoggedIn() {
        Log.d(TAG, "isLoggedIn "+preferences.getString(COOKIE, null) + " "+ accountType());
        if (preferences != null) {
            String cookie = preferences.getString(COOKIE, null);

            if(cookie == null || cookie.length() == 0) return false;
            return hasLogin() && AccountType.CONNECTED.equals(accountType());
        } else
            return false;
    }

    /**
     * Sets id.
     *
     * @param id the id
     */
    public static void setId(String id) {
        setId(id, sCurrentAccountType);
    }

    /**
     * Sets id.
     *
     * @param id the id
     */
    public static void setId(String id, AccountType type) {
        if (preferences != null) {
            preferences.edit()
                    .putString(ID, id)
                    .putString(ACCOUNT_TYPE, type.name())
                    .apply();
        }
    }

    /**
     * Login string.
     *
     * @return the string
     */
    public static String login() {
        if (preferences == null) {
            return null;
        }
        return preferences.getString(LOGIN, null);
    }

    private static boolean hasLogin() {
        String login = login();
        return null != login && login.length() > 0;
    }

    /**
     * Password string.
     *
     * @return the string
     */
    public static String password() {
        if (preferences == null) {
            return null;
        }
        return preferences.getString(PASSWORD, null);
    }

    /**
     * Token string.
     *
     * @return the string
     */
    public static String token() {
        if (preferences == null) {
            return null;
        }
        return preferences.getString(ACCESS_TOKEN, null);
    }

    /**
     * @return the conference timeout duration.
     */
    public static long conferenceTimeout() {
        if (preferences == null) {
            return -1;
        }
        return preferences.getLong(CONFERENCE_TIMEOUT, -1);
    }

    /**
     * Upload token string.
     *
     * @return the string
     */
    public static String uploadToken() {
        if (preferences == null) {
            return null;
        }
        return preferences.getString(UPLOAD_TOKEN, null);
    }

    /**
     * Device identifier string.
     *
     * @return the string
     */
    public static String deviceIdentifier() {
        if (deviceIdentifier == null) {
            return null;
        }
        return deviceIdentifier;
    }

    /**
     * Is facebook connected boolean.
     *
     * @return the boolean
     */
    public static boolean isFacebookConnected() {
        if (preferences == null) {
            return false;
        }
        return preferences.getBoolean(FACEBOOK_CONNECTED, false);
    }

    /**
     * Is google connected boolean.
     *
     * @return the boolean
     */
    public static boolean isGoogleConnected() {
        if (preferences == null) {
            return false;
        }
        return preferences.getBoolean(GOOGLE_CONNECTED, false);
    }

    /**
     * Delay ms int.
     *
     * @return the int
     */
    public static int delayMs() {
        if (preferences == null) {
            return 0;
        }
        return preferences.getInt(DELAY_ESTIMATION, -1);
    }

    /**
     * Gets auto login.
     *
     * @return the auto login
     */
    public static boolean getAutoLogin() {
        return preferences.getBoolean(AUTOLOGIN, false);
    }

    /**
     * Sets auto login.
     *
     * @param value the value
     */
    public static void setAutoLogin(boolean value) {
        if (preferences != null) {
            preferences.edit()
                    .putBoolean(AUTOLOGIN, value)
                    .apply();
        }
    }

    /**
     * Sets login.
     *
     * @param login the login
     */
    public static void setLogin(String login) {
        if (preferences != null) {
            preferences.edit()
                    .putString(LOGIN, login)
                    .apply();
        }
    }

    /**
     * Sets the conference timeout.
     *
     * @param timeout the timeout duration
     */
    public static void setConferenceTimeout(long timeout) {
        if (preferences != null) {
            preferences.edit()
                    .putLong(CONFERENCE_TIMEOUT, timeout)
                    .apply();
        }
    }

    /**
     * Sets password.
     *
     * @param password the password
     */
    public static void setPassword(String password) {
        if (preferences != null) {
            preferences.edit()
                    .putString(PASSWORD, password)
                    .apply();
        }
    }

    /**
     * Clear preferences on logout.
     */
    public static void onLogout() {
        Log.d(TAG, "onLogout");
        if (preferences != null) {
            SharedPreferencesCacheWrapper.Editor editor = preferences.edit();

            editor.putString(LOGIN, "")
                    .putString(PASSWORD, "")
                    .putString(ACCESS_TOKEN, "")
                    .putString(UPLOAD_TOKEN, "")
                    .putString(COOKIE, "")
                    .putString(ID, "")

                    .putBoolean(FACEBOOK_CONNECTED, false)
                    .putBoolean(GOOGLE_CONNECTED, false)

                    .apply();
        }
    }

    /**
     * Gets login cookie.
     *
     * @return the login cookie
     */
    public static String getLoginCookie() {
        return preferences.getString(COOKIE, null);
    }

    /**
     * Save login cookie.
     *
     * @param cookie the cookie
     */
    public static void saveLoginCookie(String cookie) {
        Log.d(TAG, "saveLoginCookie: "+cookie);
        preferences.edit()
                .putString(COOKIE, cookie)
                .apply();
    }

    /**
     * Gets upload token.
     *
     * @return the upload token
     */
    public static String getUploadToken() {
        if (preferences != null) {
            return preferences.getString(UPLOAD_TOKEN, null);
        }

        return null;
    }

    /**
     * Determines if the user is premium.
     *
     * @return the boolean
     */
    public static boolean isPremium() {
        if (preferences != null) {
            return preferences.getBoolean(PREMIUM, false);
        }

        return false;
    }

    /**
     * Sets is premium.
     *
     * @param isPremium the is premium paramet
     */
    public static void setIsPremium(boolean isPremium) {
        if (preferences != null) {
            preferences.edit()
                    .putBoolean(PREMIUM, isPremium)
                    .apply();
        }
    }

    /**
     * Sets upload token.
     *
     * @param uploadToken the upload token
     */
    public static void setUploadToken(String uploadToken) {
        if (preferences != null) {
            preferences.edit()
                    .putString(UPLOAD_TOKEN, uploadToken)
                    .apply();
        }
    }

    /**
     * Sets last address book import date.
     *
     * @param date the date
     */
    public static void setLastAddressBookImport(Date date) {
        if (preferences != null) {
            preferences.edit()
                    .putLong(ADDRESS_BOOK_SYNCED, date.getTime())
                    .apply();
        }
    }

    /**
     * Gets the date of the last addresss book scan.
     *
     * @return the last addresss book scan
     */
    public static Date getLastAddresssBookScan() {
        if (preferences != null && preferences.contains(ADDRESS_BOOK_SYNCED))
            return new Date(preferences.getLong(ADDRESS_BOOK_SYNCED, new Date().getTime()));
        else
            return null;
    }

    /**
     * Sets the default activity
     *
     * @param pathActivity the path to the activity to be called
     */
    public static void setDefaultActivity(String pathActivity) {
        if (preferences != null) {
            preferences.edit()
                    .putString(DEFAULT_ACTIVITY, pathActivity)
                    .apply();
        }
    }

    /**
     * Get the path for the default activity
     *
     * @return the path to the activity to be called
     */
    public static String getDefaultActivity() {
        return preferences.getString(DEFAULT_ACTIVITY, null);
    }


    private static AccountType sCurrentAccountType = AccountType.SDK;

    public static void setCurrentAccountType(@NonNull AccountType accountType) {
        Log.d(TAG, "setCurrentAccountType: " + accountType+" "+accountType.name());
        sCurrentAccountType = accountType;
        if(preferences != null) {
            preferences.edit()
                    .putString(ACCOUNT_TYPE, accountType.name())
                    .apply();
        }
    }

    public enum AccountType {
        SDK,
        GUEST,
        CONNECTED;

        static AccountType fromName(String name) {
            if(null == name) return SDK;
            switch(name) {
                case "CONNECTED": return CONNECTED;
                case "GUEST": return GUEST;
                default: return SDK;
            }
        }
    }
}
