package voxeet.com.sdk.core.services.abstracts;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.HashMap;
import java.util.List;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Response;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.schedulers.Schedulers;
import voxeet.com.sdk.core.AbstractVoxeetService;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.services.holder.ServiceProviderHolder;

/**
 * Created by kevinleperf on 06/08/2018.
 */

public abstract class AbstractPresentationService<KLASS> extends AbstractVoxeetService<KLASS> {
    private final VoxeetSdkTemplate mInstance;

    protected AbstractPresentationService(@NonNull VoxeetSdkTemplate instance, @Nullable Class<KLASS> service_klass) {
        super(instance, service_klass);


        mInstance = instance;
    }

    protected AbstractPresentationService(@NonNull VoxeetSdkTemplate instance, @NonNull ServiceProviderHolder<KLASS> holder) {
        super(instance, holder);

        mInstance = instance;
    }

    protected String getConferenceId() {
        return mInstance.getConferenceService().getConferenceId();
    }

    protected VoxeetSdkTemplate getVoxeetSdkInstance() {
        return mInstance;
    }

    protected Promise<Response<ResponseBody>> internalCall(final Observable<Response<ResponseBody>> observable_caller) {
        return new Promise<>(new PromiseSolver<Response<ResponseBody>>() {
            @Override
            public void onCall(@NonNull final Solver<Response<ResponseBody>> solver) {

                String conferenceId = mInstance.getConferenceService()
                        .getConferenceId();

                if (null == conferenceId) {
                    solver.reject(new IllegalStateException("Not in conference"));
                    return;
                }

                observable_caller
                        .subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {
                            }

                            @Override
                            public void onError(Throwable e) {
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> responseBodyResponse) {
                                if (responseBodyResponse.code() >= 200 && responseBodyResponse.code() < 300) {
                                    solver.resolve(responseBodyResponse);
                                } else {
                                    try {
                                        throw new IllegalStateException("Exception while managing presentation, invalid state ?");
                                    } catch (IllegalStateException exception) {
                                        solver.reject(exception);
                                    }
                                }
                            }
                        });
            }
        });
    }

    protected <PROMISE_RETURN_TYPE, EVENT_TYPE> void consumeInternalCall(final Solver<EVENT_TYPE> solver,
                                                                         final String key,
                                                                         final HashMap<String, Solver<EVENT_TYPE>> list,
                                                                         Promise<PROMISE_RETURN_TYPE> promise) {
        promise.then(new PromiseExec<PROMISE_RETURN_TYPE, EVENT_TYPE>() {
            @Override
            public void onCall(@Nullable PROMISE_RETURN_TYPE responseBodyResponse, @NonNull Solver<EVENT_TYPE> internal_solver) {
                list.put(key, solver);
            }
        }).error(new ErrorPromise() {
            @Override
            public void onError(@NonNull Throwable throwable) {
                solver.reject(throwable);
            }
        });
    }

    protected <PROMISE_RETURN_TYPE, EVENT_TYPE> void consumeInternalCall(final Solver<EVENT_TYPE> solver,
                                                                         final List<Solver<EVENT_TYPE>> list,
                                                                         Promise<PROMISE_RETURN_TYPE> promise) {
        promise.then(new PromiseExec<PROMISE_RETURN_TYPE, EVENT_TYPE>() {
            @Override
            public void onCall(@Nullable PROMISE_RETURN_TYPE responseBodyResponse, @NonNull Solver<EVENT_TYPE> internal_solver) {
                list.add(solver);
            }
        }).error(new ErrorPromise() {
            @Override
            public void onError(@NonNull Throwable throwable) {
                solver.reject(throwable);
            }
        });
    }

    protected <PRESENTATION> void tryUnlock(String fileId,
                                            PRESENTATION object,
                                            HashMap<String, Solver<PRESENTATION>> maps) {
        Solver<PRESENTATION> solver = maps.get(fileId);
        if (null != solver) {
            solver.resolve(object);
            if (maps.containsKey(fileId))
                maps.remove(solver);
        }
    }

    protected <PRESENTATION> void tryUnlock(PRESENTATION object,
                                            List<Solver<PRESENTATION>> list) {
        for (Solver<PRESENTATION> solver : list) {
            if (null != solver) {
                solver.resolve(object);
            }
        }
        list.clear();
    }
}
