package voxeet.com.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import retrofit2.Response;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.schedulers.Schedulers;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.network.ISdkFilePresentationRService;
import voxeet.com.sdk.events.success.FileConvertedEvent;
import voxeet.com.sdk.events.success.FilePresentationStartedEvent;
import voxeet.com.sdk.events.success.FilePresentationStoppedEvent;
import voxeet.com.sdk.events.success.FilePresentationUpdatedEvent;
import voxeet.com.sdk.json.FileConverted;
import voxeet.com.sdk.json.FilePresentationStarted;
import voxeet.com.sdk.json.FilePresentationStopped;
import voxeet.com.sdk.json.FilePresentationUpdated;
import voxeet.com.sdk.models.FilePresentationConverted;
import voxeet.com.sdk.models.impl.DefaultFile;

/**
 *
 */
public class SDKFilePresentationService extends voxeet.com.sdk.core.AbstractVoxeetService<ISdkFilePresentationRService> {
    private static final String TAG = SDKFilePresentationService.class.getSimpleName();
    private final VoxeetSdkTemplate mInstance;


    private HashMap<String, Solver<FilePresentationConverted>> mCacheSolvers;
    private HashMap<String, Solver<FilePresentationStarted>> mCacheStartedSolvers;
    private HashMap<String, Solver<FilePresentationStopped>> mCacheStoppedSolvers;
    private HashMap<String, Solver<FilePresentationUpdated>> mCacheUpdatedSolvers;

    /**
     * Instantiates a new User service.
     *
     * @param instance the parent instance
     */
    public SDKFilePresentationService(VoxeetSdkTemplate instance) {
        super(instance, ISdkFilePresentationRService.class);

        mInstance = instance;
        mCacheSolvers = new HashMap<>();
        mCacheStartedSolvers = new HashMap<>();
        mCacheStoppedSolvers = new HashMap<>();
        mCacheUpdatedSolvers = new HashMap<>();
        registerEventBus();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationStartedEvent event) {
        tryUnlock(event.getEvent().getFileId(), event.getEvent(), mCacheStartedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationStoppedEvent event) {
        tryUnlock(event.getEvent().getFileId(), event.getEvent(), mCacheStoppedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationUpdatedEvent event) {
        tryUnlock(event.getEvent().getFileId(), event.getEvent(), mCacheUpdatedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FileConvertedEvent event) {
        if (null != event) {
            FileConverted converted = event.getEvent();
            List<DefaultFile> list = converted.getFiles();

            for (DefaultFile file : list) {
                Log.d(TAG, "onEvent: " + file.getName());
                String key = findSolverFor(file.getName());
                Log.d(TAG, "onEvent: " + key);
                if (null != key) {
                    FilePresentationConverted temp = new FilePresentationConverted(
                            file.getName(),
                            file.getFileId(),
                            file.getSize(),
                            file.getNbImageConverted());

                    tryUnlock(key, temp, mCacheSolvers);
                }
            }
        }
    }

    /**
     * Get the url for an image
     *
     * @param fileId     the remote file id
     * @param pageNumber the requested page number
     * @return a formatted string
     */
    public String getImage(String fileId, int pageNumber) {
        return String.format("%s/v1/files/%s/converted/%d?token=%s",
                getURLRoot(mInstance), fileId, pageNumber, getInternalJwtToken(mInstance));
    }

    /**
     * Get the url for a thumbnail
     *
     * @param fileId     the remote file id
     * @param pageNumber the requested page number
     * @return a formatted string
     */
    public String getThumbnail(String fileId, int pageNumber) {
        return String.format("%s/v1/files/%s/converted/%d/thumbnail?token=%s",
                getURLRoot(mInstance), fileId, pageNumber, getInternalJwtToken(mInstance));
    }

    @Nullable
    private String findSolverFor(@NonNull String name) {
        for (String value : mCacheSolvers.keySet())
            if (name.indexOf(value) == 0) return value;
        return null;
    }

    public Promise<FilePresentationConverted> convertFile(final File file) {
        return new Promise<>(new PromiseSolver<FilePresentationConverted>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationConverted> solver) {
                final String uuid = UUID.randomUUID().toString();

                String appended_name = uuid + file.getName();
                RequestBody requestFile = RequestBody.create(MediaType.parse("multipart/form-data"), file);

                // MultipartBody.Part is used to send also the actual file name
                MultipartBody.Part body = MultipartBody.Part.createFormData("file", appended_name, requestFile);


                // finally, execute the request
                getService().convertFile(requestFile, body)
                        .subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {
                                Log.d(TAG, "onCompleted: ");
                            }

                            @Override
                            public void onError(Throwable e) {
                                Log.d(TAG, "onError: " + e);
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> responseBodyResponse) {
                                //will wait for conversion result, so we put into cache...
                                mCacheSolvers.put(uuid, solver);
                            }
                        });
            }
        });
    }

    public Promise<FilePresentationStarted> startPresentation(@NonNull final FilePresentationConverted body) {
        return startPresentation(body, 0);
    }

    public Promise<FilePresentationStarted> startPresentation(@NonNull final FilePresentationConverted body,
                                                              final int position) {
        return new Promise<>(new PromiseSolver<FilePresentationStarted>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationStarted> solver) {
                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(body.getFileId(), body.getName(), position, body.getNbImageConverted());

                internalCall(getService()
                        .startFilePresentation(mInstance.getConferenceService().getConferenceId(), body_sent))
                        .then(new PromiseExec<Response<ResponseBody>, Object>() {
                            @Override
                            public void onCall(@Nullable Response<ResponseBody> responseBodyResponse, @NonNull Solver<Object> internal_solver) {
                                mCacheStartedSolvers.put(body.getFileId(), solver);
                            }
                        })
                        .error(new ErrorPromise() {
                            @Override
                            public void onError(@NonNull Throwable throwable) {
                                solver.reject(throwable);
                            }
                        });
            }
        });
    }

    public Promise<FilePresentationStopped> stopPresentation(@NonNull final String fileId) {
        return new Promise<>(new PromiseSolver<FilePresentationStopped>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationStopped> solver) {
                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(fileId);

                internalCall(getService()
                        .stopFilePresentation(mInstance.getConferenceService().getConferenceId(), body_sent))
                        .then(new PromiseExec<Response<ResponseBody>, Object>() {
                            @Override
                            public void onCall(@Nullable Response<ResponseBody> responseBodyResponse, @NonNull Solver<Object> internal_solver) {
                                mCacheStoppedSolvers.put(fileId, solver);
                            }
                        })
                        .error(new ErrorPromise() {
                            @Override
                            public void onError(@NonNull Throwable throwable) {
                                solver.reject(throwable);
                            }
                        });
            }
        });
    }

    public Promise<FilePresentationUpdated> updatePresentation(@NonNull final String fileId,
                                                               final int position) {
        return new Promise<>(new PromiseSolver<FilePresentationUpdated>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationUpdated> solver) {

                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(fileId, position);

                internalCall(getService().updateFilePresentation(mInstance.getConferenceService().getConferenceId(), body_sent))
                        .then(new PromiseExec<Response<ResponseBody>, Object>() {
                            @Override
                            public void onCall(@Nullable Response<ResponseBody> responseBodyResponse, @NonNull Solver<Object> internal_solver) {
                                mCacheUpdatedSolvers.put(fileId, solver);
                            }
                        })
                        .error(new ErrorPromise() {
                            @Override
                            public void onError(@NonNull Throwable throwable) {
                                solver.reject(throwable);
                            }
                        });
            }
        });
    }

    private Promise<Response<ResponseBody>> internalCall(final Observable<Response<ResponseBody>> observable_caller) {
        return new Promise<>(new PromiseSolver<Response<ResponseBody>>() {
            @Override
            public void onCall(@NonNull final Solver<Response<ResponseBody>> solver) {

                String conferenceId = mInstance.getConferenceService()
                        .getConferenceId();

                if (null == conferenceId) {
                    solver.reject(new IllegalStateException("Not in conference"));
                    return;
                }

                observable_caller
                        .subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(new Subscriber<Response<ResponseBody>>() {
                            @Override
                            public void onCompleted() {
                                Log.d(TAG, "onCompleted: ");
                            }

                            @Override
                            public void onError(Throwable e) {
                                solver.reject(e);
                            }

                            @Override
                            public void onNext(Response<ResponseBody> responseBodyResponse) {
                                if (responseBodyResponse.code() >= 200 && responseBodyResponse.code() < 300) {
                                    solver.resolve(responseBodyResponse);
                                } else {
                                    try {
                                        throw new IllegalStateException("Exception while managing presentation, invalid state ?");
                                    } catch (IllegalStateException exception) {
                                        solver.reject(exception);
                                    }
                                }
                            }
                        });
            }
        });
    }

    private <FILE_PRESENTATION> void tryUnlock(String fileId,
                                               FILE_PRESENTATION object,
                                               HashMap<String, Solver<FILE_PRESENTATION>> maps) {
        Log.d(TAG, "tryUnlock: fileId := " + fileId + " " + object + " " + maps);
        Solver<FILE_PRESENTATION> solver = maps.get(fileId);
        Log.d(TAG, "tryUnlock: solver ? := " + solver);
        if (null != solver) {
            solver.resolve(object);
            if (maps.containsKey(fileId))
                maps.remove(solver);
        }
    }

}