package voxeet.com.sdk.core.services.authenticate.token;


import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import java.io.IOException;
import java.util.HashMap;
import java.util.concurrent.TimeUnit;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.Authenticator;
import okhttp3.ConnectionPool;
import okhttp3.Headers;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.Route;
import retrofit2.Call;
import retrofit2.Callback;
import voxeet.com.sdk.core.network.IVoxeetRService;
import voxeet.com.sdk.json.UserInfo;
import voxeet.com.sdk.models.TokenResponse;
import voxeet.com.sdk.models.UserTokenResponse;

/**
 * Provide a valid TokenAccess holder object
 */
public abstract class TokenResponseProvider {
    protected static final String HEADER_NAME_AUTHORIZATION = "Authorization";
    private static final String HEADER_VOXEET_TOKEN = "Voxeet-Token";
    private static final String HEADER_NAME_XTOKEN = "X-Token";
    public final String TAG = TokenResponseProvider.class.getSimpleName();

    @Nullable
    private UserInfo mUserInfo;

    private String resetTokenValue = "";

    /**
     * The Token response.
     */
    @Nullable
    protected TokenResponse tokenResponse;
    private UserTokenResponse mUserTokenResponse;
    protected IVoxeetRService service;
    private IVoxeetRService serviceSubAuthenticator;

    @NonNull
    private final HashMap<String, UserTokenResponse> responses = new HashMap<>();

    public TokenResponseProvider() {

    }

    public void resetVoxeetHttp() {
        resetTokenValue = "resetting";
    }

    @Nullable
    private TokenResponse multipleRefreshTokenResponseTry() {
        for (int tries = 0; tries < 5; tries++) {
            TokenResponse answer = refreshTokenResponse();
            if (null != answer) return answer;

            wait(tries);
        }
        return null;
    }

    @Nullable
    public Response executeIdentify(Interceptor.Chain chain, String tag) throws IOException {
        Request.Builder builder = chain.request()
                .newBuilder();

        if (null != tokenResponse) {
            addHeader(builder.removeHeader(HEADER_NAME_AUTHORIZATION))
                    .addHeader(HEADER_NAME_AUTHORIZATION, "bearer " + resetTokenValue + tokenResponse.getAccessToken());
        }
        return chain.proceed(builder.build());
    }

    @Nullable
    public Response execute(Interceptor.Chain chain) throws IOException {
        try {
            Request.Builder builder = chain.request()
                    .newBuilder();

            if (null != tokenResponse) {
                //addHeader(builder.removeHeader(HEADER_NAME_AUTHORIZATION))
                builder//.removeHeader(HEADER_NAME_AUTHORIZATION)
                        .addHeader(HEADER_NAME_AUTHORIZATION, "bearer " + resetTokenValue + tokenResponse.getAccessToken());
            }

            if (null != mUserTokenResponse) {
                builder.removeHeader(HEADER_NAME_XTOKEN);
                if (!TextUtils.isEmpty(mUserTokenResponse.getUserToken()))
                    builder.addHeader(HEADER_VOXEET_TOKEN, mUserTokenResponse.getUserToken());
                if (null != mUserTokenResponse.getJwtUserToken())
                    builder.addHeader(HEADER_NAME_XTOKEN, mUserTokenResponse.getJwtUserToken());
            }
            return chain.proceed(builder.build());
        } catch (Exception e) {
            Log.d(TAG, "execute: ISSUE WHILE EXECUTING " + chain.call().request().toString());
            if (e instanceof IOException) throw e;
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Retrieve the current user token information
     * Note that :
     * - The current SDK must be logged in when called (call retrieveTokenResponse or response from 403)
     * - it will resolve a valid object
     * - it will reject in case of network error
     * <p>
     * Note : don't forget to call .then().error()
     *
     * @param userInfo valid user info to identify the user
     * @return a promise to resolve and manage the error
     */
    @NonNull
    private Promise<UserTokenResponse> retrieveUserTokenResponse(@NonNull final UserInfo userInfo) {
        return new Promise<>(new PromiseSolver<UserTokenResponse>() {
            @Override
            public void onCall(@NonNull final Solver<UserTokenResponse> solver) {
                UserTokenResponse response = getUserTokenResponse(userInfo.getExternalId());
                if (null != response) {
                    Log.d(TAG, "onCall: the user is known, it is better to resolve it...");
                    solver.resolve(response);
                } else {
                    final Call<UserTokenResponse> user = service.identify(userInfo);
                    user.enqueue(new Callback<UserTokenResponse>() {
                        @Override
                        public void onResponse(@NonNull Call<UserTokenResponse> call,
                                               @NonNull retrofit2.Response<UserTokenResponse> response) {
                            UserTokenResponse token = response.body();
                            if(null != token) {
                                responses.put(userInfo.getExternalId(), token);
                            }
                            solver.resolve(token);
                        }

                        @Override
                        public void onFailure(@NonNull Call<UserTokenResponse> call, @NonNull Throwable t) {
                            solver.reject(t);
                        }
                    });
                }
            }
        });
    }

    @Nullable
    private Request reauthenticate(Route route, Response response) {
        resetTokenValue = "";
        boolean wasIdentifying = false;
        Headers headers = response.request().headers();
        if (null != headers) {
            for (int i = 0; i < headers.size(); i++) {
                String name = headers.name(i);
                String header = headers.get(name);

                if (null != header && header.contains("identifying")) {
                    wasIdentifying = true;
                }
            }
        }

        Request.Builder builder = response.request().newBuilder();
        TokenResponse tokenResponse = multipleRefreshTokenResponseTry();//old implementation = tokenResponseProvider.refreshTokenResponse();

        if (null != tokenResponse) {
            this.tokenResponse = tokenResponse;
        } else {
            //throw new IllegalStateException("Invalid credentials");
            return null;
        }

        if (!wasIdentifying && null != mUserInfo && null == mUserTokenResponse) {
            try {
                UserTokenResponse userTokenResponse = getUserTokenResponse(mUserInfo.getExternalId());
                if (null == userTokenResponse) {
                    userTokenResponse = serviceSubAuthenticator.identify(mUserInfo).execute().body();
                }
                if (null != userTokenResponse) {
                    mUserTokenResponse = userTokenResponse;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else if(wasIdentifying)
            Log.d(TAG, "reauthenticate: no identify in it since it was identifying the user B)");

        addHeader(builder.removeHeader(HEADER_NAME_AUTHORIZATION))
                .addHeader(HEADER_NAME_AUTHORIZATION, "bearer " + tokenResponse.getAccessToken());

        if (!wasIdentifying && null != mUserTokenResponse) {
            builder.removeHeader(HEADER_NAME_XTOKEN)
                    .addHeader(HEADER_VOXEET_TOKEN, mUserTokenResponse.getUserToken())
                    .addHeader(HEADER_NAME_XTOKEN, mUserTokenResponse.getJwtUserToken());
        } else {
            Log.d(TAG, "authenticate: no jwt token or wasIdentifying:=" + wasIdentifying);
        }

        return builder.build();
    }

    @Nullable
    public abstract TokenResponse refreshTokenResponse();


    /**
     * Retrieve a valid TokenResponse from the servers
     * <p>
     * It will use on of the two available method available :
     * - resolve the injected tokenAccess
     * - resolve a token response obtained from Voxeet servers
     * <p>
     * Note : don't forget to call .then().error()
     *
     * @return a promise to resolve and manage the error
     */
    @NonNull
    public Promise<UserTokenResponse> retrieve() {
        return new Promise<>(new PromiseSolver<UserTokenResponse>() {
            @Override
            public void onCall(@NonNull Solver<UserTokenResponse> solver) {
                Promise<TokenResponse> response = retrieveTokenResponse();
                response.then(new PromiseExec<TokenResponse, UserTokenResponse>() {
                    @Override
                    public void onCall(@Nullable TokenResponse result, @NonNull Solver<UserTokenResponse> s) {
                        tokenResponse = result;
                        if(null != mUserInfo) {
                            s.resolve(retrieveUserTokenResponse(mUserInfo));
                        } else {
                            try {
                                throw new IllegalStateException("user info null, disconnected ?");
                            }catch (Exception e){
                                s.reject(e);
                            }
                        }
                    }
                }).then(new PromiseExec<UserTokenResponse, Object>() {
                    @Override
                    public void onCall(@Nullable UserTokenResponse result, @NonNull Solver<Object> s) {
                        mUserTokenResponse = result;
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        solver.reject(error);
                    }
                });
            }
        });
    }

    @NonNull
    protected abstract Promise<TokenResponse> retrieveTokenResponse();

    @NonNull
    protected abstract Request.Builder addHeader(@NonNull Request.Builder builder);

    private void wait(int tries) {
        try {
            Thread.sleep(200 * tries);
        } catch (Exception ignored) {

        }
    }

    public void setUserInfo(@Nullable UserInfo userInfo) {
        mUserInfo = userInfo;
    }

    @Nullable
    public TokenResponse getTokenResponse() {
        return tokenResponse;
    }

    @Nullable
    public String getToken() {
        return mUserTokenResponse != null ? mUserTokenResponse.getUserToken() : null;
    }

    @Nullable
    public String getJwtToken() {
        return mUserTokenResponse != null ? mUserTokenResponse.getJwtUserToken() : null;
    }

    public void initService(IVoxeetRService service, IVoxeetRService serviceSubAuthenticator) {
        this.service = service;
        this.serviceSubAuthenticator = serviceSubAuthenticator;
    }

    private Authenticator createAuthenticator() {
        return new Authenticator() {
            @Override
            public Request authenticate(@NonNull Route route, @NonNull Response response) throws IOException {
                return reauthenticate(route, response);
            }
        };
    }

    public void configureOkHttpClientBuilder(OkHttpClient.Builder builder, boolean useAuthenticator) {
        if (useAuthenticator) {
            builder = builder.authenticator(createAuthenticator());
        }

        builder.connectionPool(new ConnectionPool(5, 45, TimeUnit.SECONDS))
                .retryOnConnectionFailure(true)
                .pingInterval(30, TimeUnit.SECONDS)
                .readTimeout(15, TimeUnit.SECONDS)
                .writeTimeout(15, TimeUnit.SECONDS);
    }

    @Nullable
    private UserTokenResponse getUserTokenResponse(@NonNull String userId) {
        synchronized (responses) {
            if (responses.containsKey(userId)) {
                return responses.get(userId);
            }
        }
        return null;
    }

    public void cleanUserSession(@NonNull String userId) {
        Log.d(TAG, "cleanUserSession: clean the information for user " + userId);
        try {
            synchronized (responses) {
                if (responses.containsKey(userId)) {
                    responses.remove(userId);
                }
            }
        } catch (Exception e) {
            Log.e(TAG, "cleanUserSession: error while removing user session := " + userId);
            e.printStackTrace();
        }
    }
}
