package voxeet.com.sdk.core.network.websocket;

import android.content.Context;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import okhttp3.HttpUrl;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import voxeet.com.sdk.core.VoxeetHttp;
import voxeet.com.sdk.core.services.authenticate.WebSocketState;
import voxeet.com.sdk.events.error.HttpException;
import voxeet.com.sdk.utils.DeviceStateUtils;

public class WebSocketProxy implements Runnable {

    private final static Handler HANDLER = new Handler();
    private static final int MAXIMUM_RETRY = 25;
    private static final long SOCKET_RETRY_MAX = 10000; //10s
    private static final long SOCKET_RETRY_AFTER = 500; //0.5s
    private static final String TAG = WebSocketProxy.class.getSimpleName();

    private WebSocketState mState;
    private Context mContext;
    private ExecutorService mExecutorService;
    private String mSocketUrl;
    private int mCount = 0;
    private boolean shouldRetry = true;
    private WebSocketListener mAdapter;
    private List<SocketListener> mListeners;
    private long mSocketCurrentRetryDelay;
    private WebSocket mWebSocket;
    private ConnectListener mConnectListener;
    private boolean isCanceled = false;
    private VoxeetHttp mProvider;

    public WebSocketProxy(Context context, @NonNull String socketUrl) {
        mState = WebSocketState.CLOSED;
        mSocketUrl = socketUrl;

        mContext = context;
        mExecutorService = Executors.newSingleThreadExecutor();


        mListeners = new ArrayList<>();
        mSocketCurrentRetryDelay = SOCKET_RETRY_AFTER;
        mCount = 0;

        mAdapter = new WebSocketListener() {

            @Override
            public void onOpen(@NonNull WebSocket websocket, @NonNull Response response) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                mWebSocket = websocket;

                mState = WebSocketState.CONNECTED;

                //reset values to let the app reconnect on error
                mSocketCurrentRetryDelay = SOCKET_RETRY_AFTER;
                mCount = 0;
                shouldRetry = true;

                attemptConnectListenerConnected(websocket);

                onStateChanged(websocket, mState);
                for (SocketListener listener : mListeners) {
                    try {
                        listener.onConnect(websocket);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            @Override
            public void onMessage(@NonNull WebSocket websocket, @NonNull String message) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                for (SocketListener listener : mListeners) {
                    try {
                        listener.onTextMessage(message);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            @Override
            public void onFailure(@NonNull WebSocket websocket, @NonNull Throwable cause, Response response) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }
                cause.printStackTrace();

                HttpException error = HttpException.throwResponse(response);
                for (SocketListener listener : mListeners) {
                    try {
                        listener.onError(error);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            @Override
            public void onClosing(@NonNull WebSocket webSocket, int code, @NonNull String reason) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }
                mState = WebSocketState.CLOSING;
                //WebSocketProxy.this.onClosing();
            }

            @Override
            public void onClosed(@NonNull WebSocket webSocket, int code, @NonNull String reason) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                if (shouldRetry) {
                    if (mCount < MAXIMUM_RETRY) {

                        mState = WebSocketState.CONNECTING;
                        mCount++;
                        HANDLER.postDelayed(WebSocketProxy.this, mSocketCurrentRetryDelay);
                    } else {
                        mState = WebSocketState.CLOSED;
                        attemptConnectListenerDisconnected();
                        WebSocketProxy.this.onDisconnected();
                    }
                } else {
                    mState = WebSocketState.CLOSED;
                    onStateChanged(webSocket, mState);
                    WebSocketProxy.this.onDisconnected();
                }
            }


            void onStateChanged(WebSocket websocket, WebSocketState newState) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                for (SocketListener listener : mListeners) {
                    try {
                        listener.onStateChanged(newState);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        };
    }

    private void attemptConnectListenerDisconnected() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (mConnectListener != null) {
            mConnectListener.onConnectError(new HttpException((Response) null));
            mConnectListener = null;
        }
    }

    private void attemptConnectListenerConnected(WebSocket webSocket) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (mConnectListener != null) {
            mConnectListener.onConnect(webSocket);
            mConnectListener = null;
        }
    }

    private Request setWebSocketRequest() {
        if (isCanceled) {
            checkCancel();
            return null;
        }

        String socketUrl = mSocketUrl;

        String jwtToken = mProvider.getJwtToken();
        if (null != jwtToken) {
            if (!socketUrl.contains("?")) socketUrl += "?Token=" + jwtToken;
            else socketUrl += "Token=" + jwtToken;

            /*if (null == httpUrl) {
                Log.d(TAG, "setWebSocketRequest: httpUrl is null := " + socketUrl);
                return null;
            }*/

            return new Request.Builder()
                    .url(socketUrl)
                    .build();
        }
        return null;
    }

    public void connect(@NonNull VoxeetHttp provider, @Nullable ConnectListener connectListener) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        mConnectListener = connectListener;
        mProvider = provider;

        Request request = setWebSocketRequest();

        if (null != request && DeviceStateUtils.isNetworkAvailable(mContext)) {
            mState = WebSocketState.CONNECTING;

            mWebSocket = provider.getClient().newWebSocket(request, mAdapter);
            //ping set in TokenResponseProvider
            //mWebSocket.addExtension(WebSocketExtension.PERMESSAGE_DEFLATE);
            //mWebSocket.clearHeaders();
            //if (mUserToken != null) mWebSocket.addHeader("Voxeet-Token", mUserToken.replaceAll("\"", ""));
            //mWebSocket.addListener(mAdapter);
            //if (mAdapterListener != null) mWebSocket.addListener(mAdapterListener);
        } else {
            mState = WebSocketState.CLOSED;
            if (mConnectListener != null) {
                mConnectListener.onNoNetwork();
            }
        }
    }

    private void onDisconnected() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        for (SocketListener listener : mListeners) {
            listener.onDisconnected();
        }
    }

    public void addListener(@NonNull SocketListener listener) {
        if (!mListeners.contains(listener)) {
            mListeners.add(listener);
        }
    }

    public void removeListener(@NonNull SocketListener listener) {
        if (mListeners.contains(listener)) {
            mListeners.remove(listener);
        }
    }

    @Override
    public void run() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (DeviceStateUtils.isNetworkAvailable(mContext)) { // No point trying to reconnect if no network is available
            connect(mProvider, mConnectListener);
        } else {

            mSocketCurrentRetryDelay = Math.min(mSocketCurrentRetryDelay * 2, SOCKET_RETRY_MAX);

            HANDLER.postDelayed(this, mSocketCurrentRetryDelay);
        }
    }

    public boolean isOpen() {
        return WebSocketState.CONNECTED.equals(mState);//mWebSocket != null && mWebSocket.isOpen();
    }

    public boolean sendPing() {
        Log.e(TAG, "sendPing:");
        return false;//mWebSocket != null && mWebSocket.request().sendPing().getState() == WebSocketState.OPEN;
    }

    public void disconnect() {
        mState = WebSocketState.CLOSED;
        //prevent reconnection
        mCount = MAXIMUM_RETRY;
        shouldRetry = false;

        //disconnect
        if (null != mWebSocket) {
            mWebSocket.close(1000, "Closing socket from client");
            //mWebSocket.disconnect(WebSocketCloseCode.NORMAL, null, CLOSE_VOXEET_AFTER_MILLISECS);
        }

        for (SocketListener listener : mListeners) {
            listener.onClose();
        }

        mExecutorService.shutdown();
    }

    public void removeListeners() {
        mListeners.clear();
        mAdapter = null;
    }

    public WebSocketState getState() {
        return mState;
    }

    public WebSocket getWebSocket() {
        return mWebSocket;
    }

    private void checkCancel() {
        if (isCanceled && isOpen()) {
            disconnect();
        }
    }

    public void cancel() {
        removeListeners();
        isCanceled = true;
    }
}
