package voxeet.com.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.audio.AudioRoute;

import org.webrtc.EglBase;
import org.webrtc.VideoRenderer;

import java.util.List;

import eu.codlab.simplepromise.Promise;
import voxeet.com.sdk.events.success.ConferenceRefreshedEvent;
import voxeet.com.sdk.events.success.ConferenceUsersInvitedEvent;
import voxeet.com.sdk.events.success.DeclineConferenceResultEvent;
import voxeet.com.sdk.events.success.GetConferenceHistoryEvent;
import voxeet.com.sdk.events.success.GetConferenceStatusEvent;
import voxeet.com.sdk.models.abs.ConferenceUser;
import voxeet.com.sdk.models.impl.DefaultConferenceUser;


/**
 * Created by RomainBenmansour on 4/18/16.
 */
public interface SdkConferenceService {

    /**
     * The constant MUTE_FACTOR.
     */
    public static final int MUTE_FACTOR = 10000;

    /**
     * The constant UNMUTE_FACTOR.
     */
    public static final int UNMUTE_FACTOR = 1;


    /**
     * Sets listener mode. To be called before starting a conference
     *
     * @param isListener true to be a listener and not use the user microphone, false otherwise
     */
    void setListenerMode(boolean isListener);

    /**
     * Joins a conference as a listener meaning no camera and microphone for this user.
     *
     * @param conferenceId id of the conference to join as a listener
     */
    public abstract Promise<Boolean> listenConference(String conferenceId);

    /**
     * Gets sdk peer vu meter.
     *
     * @param peerId the peer id
     * @return the sdk peer vu meter
     */
    int getSdkPeerVuMeter(String peerId);

    /**
     * Gets current conference id.
     *
     * @return the current conference id. Can be null
     */
    String getCurrentConferenceId();

    /**
     * Gets conference room size.
     *
     * @return the conference room size
     */
    int getConferenceRoomSize();

    /**
     * Conference history given for a conference.
     *
     * @param conferenceId the conference id
     */
    Promise<GetConferenceHistoryEvent> conferenceHistory(String conferenceId);

    /**
     * Get Current speaker id.
     *
     * @return the string
     */
    String currentSpeaker();

    /**
     * Find user by id conference user.
     *
     * @param userId the user id
     * @return the conference user associated with the user id if it exists
     */
    ConferenceUser findUserById(String userId);

    /**
     * Send broadcast message.
     *
     * @param string the string
     */
    Promise<Boolean> sendBroadcastMessage(String string);

    /**
     * Leave the conference.
     */
    Promise<Boolean> leave();

    /**
     * Join the sdk conference.
     *
     * @param alias the conference alias to join
     */
    Promise<Boolean> join(String alias);

    /**
     * Join the sdk conference.
     *
     * @param conferenceId the conference id given by the notifications or internal calls
     */
    Promise<Boolean> joinVoxeetConference(String conferenceId);

    /**
     * Register the media and subscribe to the eventbus.
     */
    void register();

    /**
     * Unregister the media and unsubscribe from the eventbus.
     */
    void unregister();

    /**
     * Create a new conference.
     */
    Promise<Boolean> create();

    /**
     * Create a demo conference.
     */
    Promise<Boolean> demo();

    /**
     * Logout.
     */
    public abstract Promise<Boolean> logout();

    /**
     * Replay a conference.
     *
     * @param conferenceId the conference id
     */
    Promise<Boolean> replay(String conferenceId, long offset);

    /**
     * Mute or unmute the current user in the conference.
     *
     * @param mute the new mute state
     */
    boolean mute(boolean mute);

    /**
     * Check if the user is muted.
     *
     * @return the boolean
     */
    boolean isMuted();

    /**
     * CHeck if the user associated with the userId is muted or not.
     *
     * @param userId the user id
     * @return the boolean
     */
    boolean isUserMuted(String userId);

    /**
     * Gets conference status.
     *
     * @param conferenceId the conference id
     */
    Promise<GetConferenceStatusEvent> getConferenceStatus(String conferenceId);

    /**
     * Mute an user.
     *  @param userId     the user id
     * @param shouldMute the should mute
     */
    boolean muteUser(String userId, boolean shouldMute);

    /**
     * Gets alias id for the conference.
     *
     * @return the alias id
     */
    String getAliasId();

    /**
     * Switch camera.
     */
    Promise<Boolean> switchCamera();

    /**
     * Gets conference id.
     *
     * @return the conference id if it exists. null otherwise
     */
    String getConferenceId();

    /**
     * Toggle recording.
     */
    void toggleRecording();

    /**
     * Change peer position.
     *
     * @param userId   the user id
     * @param angle    the angle
     * @param distance the distance
     * @return true if media was initialized
     */
    boolean changePeerPosition(String userId, double angle, double distance);

    /**
     * Gets conference users.
     *
     * @return the conference users
     */
    @Nullable
    List<DefaultConferenceUser> getConferenceUsers();

    /**
     * Toggle video.
     */
    void toggleVideo();

    /**
     * Check if conference is live.
     *
     * @return the boolean
     */
    boolean isLive();

    /**
     * Check if conference is listening
     */
    boolean isListenerMode();

    /**
     * Subscribe to a conference's events.
     *
     * @param conferenceId the conference id
     */
    Promise<Boolean> subscribe(String conferenceId);

    /**
     * Unsubscribe from a conference's events.
     */
    Promise<Boolean> unSubscribe();

    /**
     * Subscribe to a specific conference start event
     */
    Promise<Boolean> subscribeForCall(String conferenceId);

    /**
     * Unsubscribe from a specific conference start event
     */
    Promise<Boolean> unSubscribeFromCall(String conferenceId);

    /**
     * Attach media stream.
     *  @param stream the stream
     * @param render the render
     */
    boolean attachMediaStream(MediaStream stream, VideoRenderer.Callbacks render);

    /**
     * Unattach media stream.
     *  @param stream the stream
     * @param render the render
     */
    boolean unAttachMediaStream(MediaStream stream, VideoRenderer.Callbacks render);

    Promise<DeclineConferenceResultEvent> decline(String conferenceId);

    /**
     * Invite a specific set of users
     *
     * @param ids a non null list of non null strings
     */
    Promise<List<ConferenceRefreshedEvent>> invite(@NonNull List<String> ids);

    /**
     * Gets egl context.
     *
     * @return the egl context
     */
    EglBase.Context getEglContext();

    /**
     * Define a timeout to leave a call if none joins the conference
     *
     * @param timeout the time to wait before leaving the call
     */
    boolean setTimeOut(long timeout);

    /**
     * @return the time to wait before leaving the call
     */
    long getTimeout();

    /**
     * @return asynchronously list of invited users to the conference
     */
    Promise<ConferenceUsersInvitedEvent> getInvitedUsers();
}
