package voxeet.com.sdk.core.preferences;

import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.HashMap;

/**
 * Class to wrap preferences
 * <p>
 * prevent issues with flush/ed keys by surrounding cache
 */

public class SharedPreferencesCacheWrapper {
    @NonNull
    private HashMap<String, String> mStrings;

    @NonNull
    private HashMap<String, Boolean> mBooleans;

    @NonNull
    private HashMap<String, Long> mLongs;

    @NonNull
    private HashMap<String, Integer> mInts;

    private SharedPreferences mPreferences;

    private SharedPreferencesCacheWrapper() {
        mStrings = new HashMap<>();
        mBooleans = new HashMap<>();
        mLongs = new HashMap<>();
        mInts = new HashMap<>();
    }

    public SharedPreferencesCacheWrapper(SharedPreferences preferences) {
        this();

        mPreferences = preferences;
    }

    @NonNull
    public Editor edit() {
        return new Editor(this);
    }

    @Nullable
    public String getString(@NonNull String key, @Nullable String def) {
        if (mStrings.containsKey(key))
            return mStrings.get(key);
        String result = mPreferences.getString(key, def);

        return result;
    }

    public boolean getBoolean(@NonNull String key, boolean def) {
        if (mBooleans.containsKey(key))
            return mBooleans.get(key);
        return mPreferences.getBoolean(key, def);
    }

    public int getInt(@NonNull String key, int def) {
        if (mInts.containsKey(key))
            return mInts.get(key);
        return mPreferences.getInt(key, def);
    }

    public long getLong(@NonNull String key, long def) {
        if (mLongs.containsKey(key))
            return mLongs.get(key);
        return mPreferences.getLong(key, def);
    }

    public boolean contains(@NonNull String key) {
        if (mStrings.containsKey(key)) return true;
        if (mBooleans.containsKey(key)) return true;
        if (mLongs.containsKey(key)) return true;
        return mPreferences.contains(key);
    }

    public class Editor {
        private SharedPreferencesCacheWrapper _wrapper;
        private SharedPreferences.Editor _editor;

        private Editor() {

        }

        protected Editor(SharedPreferencesCacheWrapper wrapper) {
            this();

            _wrapper = wrapper;
            _editor = wrapper.mPreferences.edit();
        }

        public Editor putString(@NonNull String key, @Nullable String value) {
            _wrapper.mStrings.put(key, value);
            _editor.putString(key, value);
            return this;
        }

        public Editor putLong(@NonNull String key, long value) {
            _wrapper.mLongs.put(key, value);
            _editor.putLong(key, value);
            return this;
        }

        public Editor putBoolean(@NonNull String key, boolean value) {
            _wrapper.mBooleans.put(key, value);
            _editor.putBoolean(key, value);
            return this;
        }

        public Editor putInt(@NonNull String key, int value) {
            _wrapper.mInts.put(key, value);
            _editor.putInt(key, value);
            return this;
        }

        /**
         * Apply the values to the preferences
         */
        public void apply() {
            //when calling apply everything is fine
            //TODO if revert is called, restore values
            _editor.apply();
        }
    }


}