package voxeet.com.sdk.core;

import android.app.Application;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.util.Log;
import android.widget.Toast;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.neovisionaries.ws.client.WebSocketState;
import com.voxeet.android.media.Media;
import com.voxeet.kernel.R;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.EventBusException;

import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.OkHttpClient;
import retrofit2.Retrofit;
import voxeet.com.sdk.core.abs.AbstractConferenceSdkService;
import voxeet.com.sdk.core.preferences.VoxeetPreferences;
import voxeet.com.sdk.core.services.ConferenceService;
import voxeet.com.sdk.core.services.UserService;
import voxeet.com.sdk.events.success.SocketStateChangeEvent;
import voxeet.com.sdk.factories.EventsFactory;
import voxeet.com.sdk.json.Event;
import voxeet.com.sdk.json.UserInfo;
import voxeet.com.sdk.models.TokenResponse;
import voxeet.com.sdk.models.UserTokenResponse;
import voxeet.com.sdk.utils.ConferenceListener;
import voxeet.com.sdk.utils.Twig;
import voxeet.com.sdk.utils.VoxeetServerUrl;

/**
 * Created by kevinleperf on 09/11/2017.
 */

public abstract class VoxeetSdkTemplate<T extends AbstractConferenceSdkService> implements ConferenceListener {
    static final long DEFAULT_TIMEOUT_MS = 10 * 1000;
    private UserInfo _user_info;
    private VoxeetWebSocket _voxeet_websocket;
    private VoxeetHttp _voxeet_http;

    protected Context _application_context;
    protected EventBus _event_bus;

    private Twig _twig;

    private VoxeetServerUrl _voxeet_server_url;

    private HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> _services;

    private VoxeetSdkTemplate() {

    }

    protected VoxeetSdkTemplate(@NonNull Application application_context,
                                @NonNull String appId,
                                @NonNull String password,
                                @NonNull UserInfo userInfo,
                                boolean debug) {
        this();

        _application_context = application_context;
        _event_bus = EventBus.getDefault();

        //make sure the looper is the main looper for promise execution
        Promise.setHandler(new Handler(Looper.getMainLooper()));

        //call pre init - can instantiate specific variables
        onSdkPreinit();

        _user_info = userInfo;

        _twig = new Twig(2, "VoxeetSDK :: " + getTag(), false);

        _voxeet_server_url = new VoxeetServerUrl(this);

        _voxeet_websocket = new VoxeetWebSocket(application_context, this, _voxeet_server_url.getSocketUrl());


        _voxeet_http = new VoxeetHttp.Builder()
                .setApplication(application_context)
                .setVoxeetSDK(this)
                .setServerUrl(_voxeet_server_url.getServerUrl())
                .setServerPort(_voxeet_server_url.getServerPort())
                .setAppId(appId)
                .setPassword(password)
                .setVoxeetServiceListener(new VoxeetHttp.VoxeetServiceListener() {

                    @Override
                    public void onIdentifySuccess(UserTokenResponse userToken, TokenResponse tokenResponse) {

                        _twig.d("onIdentifySuccess " + userToken + " " + tokenResponse +" "+userToken.getJwtUserToken());
                        //if (sdkConferenceService == null)
                        //    sdkConferenceService = new AbstractConferenceSdkService(applicationContext, conferenceListener);

                        _voxeet_websocket.connect(userToken.getUserToken(), userToken.getJwtUserToken());
                    }

                    @Override
                    public void onIdentifyError(String error) {
                        Log.e(getTag(), error);
                    }

                    @Override
                    public void onNetworkError(UnknownHostException error) {
                        _event_bus.post(new SocketStateChangeEvent(WebSocketState.CLOSED.name()));
                    }
                })
                .setDebug(debug)
                .build();


        _services = new HashMap<>();
        VoxeetPreferences.init(getApplicationContext());
        initServices();
    }


    //TODO check for logout ?
    public void logUser(@NonNull UserInfo userInfo) {
        _user_info = userInfo;

        openSdkSession();
    }

    /**
     * The method logout() must be defined for every SDKs
     */
    public abstract Promise<Boolean> logout();


    public abstract void onSdkPreinit();


    protected abstract void initServices();

    protected HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> getServices() {
        return _services;
    }

    public T getConferenceService() {
        //explicitly cast, ConferenceSdkService is stored as AbstractConferenceSdkService
        return (T) getServiceForKlass(AbstractConferenceSdkService.class);
    }

    /**
     * Sends conference reminder.
     *
     * @param ids the ids
     */
    public void sendConferenceReminder(List<String> ids) {
        ConferenceService service = getServiceForKlass(ConferenceService.class);
        if (service != null) {
            service.sendReminder(ids);
        }
    }

    protected <T extends AbstractVoxeetService> T getServiceForKlass(Class<T> klass) {
        T service = null;
        Iterator<Class<? extends AbstractVoxeetService>> iterator = _services.keySet().iterator();
        while (iterator.hasNext()) {
            Class<? extends AbstractVoxeetService> next = iterator.next();
            AbstractVoxeetService item = _services.get(next);


            if (klass.isInstance(item)) {
                service = (T) item;
            }
        }

        if (service == null) {
            getTwig().e(klass.getSimpleName() + " not found in the list of services");
        }
        return service;
    }

    /**
     * Send a conference broadcast.
     *
     * @param message the message
     */
    public void sendConferenceMessage(String message) {
        ConferenceService service = getServiceForKlass(ConferenceService.class);
        if (service != null) {
            service.sendMessage(message);
        }
    }

    /**
     * Gets upload token. Won't be able to upload files and such without it
     */

    public void getUploadToken() {
        UserService service = getServiceForKlass(UserService.class);
        if (service != null) {
            service.getUploadToken();
        }
    }


    public void send(Event command) {
        ObjectMapper om = new ObjectMapper();
        try {
            this.sendConferenceMessage(om.writeValueAsString(command));
        } catch (JsonProcessingException e) {
            Log.e(Event.TAG, "failed to send command", e);
        }
    }

    public Event decode(String message) {
        return EventsFactory.decode(message);
    }

    public Twig getTwig() {
        return _twig;
    }

    public Retrofit getRetrofit() {
        return _voxeet_http.getRetrofit();
    }

    public OkHttpClient getClient() {
        return _voxeet_http.getClient();
    }

    public Context getApplicationContext() {
        return _application_context;
    }

    public VoxeetServerUrl getVoxeetServerUrl() {
        return _voxeet_server_url;
    }

    /**
     * Reset services.
     */
    public void resetServices() {
        Iterator<Class<? extends AbstractVoxeetService>> iterator = _services.keySet().iterator();
        while (iterator.hasNext()) {
            Class<? extends AbstractVoxeetService> next = iterator.next();
            _services.get(next).resetService();
        }
    }

    public EventBus getEventBus() {
        return _event_bus;
    }

    protected VoxeetWebSocket getSocket() {
        return _voxeet_websocket;
    }

    protected VoxeetHttp getVoxeetHttp() {
        return _voxeet_http;
    }

    public void closeSocket() {
        _voxeet_websocket.close(true);
    }

    public boolean isSocketOpen() {
        return _voxeet_websocket.isOpen();
    }

    public void openSdkSession() {
        _voxeet_http.identify(_user_info);
    }

    @Override
    public Promise<Boolean> onCreationSuccess(String conferenceId) {
        AbstractConferenceSdkService service = getServiceForKlass(AbstractConferenceSdkService.class);

        if (service != null) return service.join(conferenceId);
        else {
            return new Promise<>(new PromiseSolver<Boolean>() {
                @Override
                public void onCall(@NonNull Solver<Boolean> solver) {
                    solver.resolve(false);
                }
            });
        }
    }


    protected abstract String getTag();

    /**
     * Register Media. To be called before creating/joining any conference.
     *
     * @param context    the context
     * @param subscriber the subscriber
     */
    public boolean register(Context context, Object subscriber) {
        try {
            Media.Register(context);

            if (!getEventBus().isRegistered(subscriber))
                getEventBus().register(subscriber);
        } catch (EventBusException error) {
            //silent catch
        } catch (UnsatisfiedLinkError error) {
            Toast.makeText(getApplicationContext(), context.getString(R.string.register_conf_error), Toast.LENGTH_SHORT).show();
            return false;
        }

        return true;
    }

    /**
     * Unregister Media. To be called in the on destroy of your activty/fragment
     *
     * @param subscriber the subscriber
     */
    public void unregister(Object subscriber) {
        Media.UnRegister();

        if (getEventBus().isRegistered(subscriber))
            getEventBus().unregister(subscriber);
    }
}
