package voxeet.com.sdk.core;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.neovisionaries.ws.client.WebSocket;
import com.neovisionaries.ws.client.WebSocketException;
import com.neovisionaries.ws.client.WebSocketState;

import java.util.ArrayList;
import java.util.List;

import voxeet.com.sdk.core.network.websocket.ConnectListener;
import voxeet.com.sdk.core.network.websocket.SocketListener;
import voxeet.com.sdk.core.network.websocket.WebSocketProxy;
import voxeet.com.sdk.utils.Twig;

/**
 * Created by RomainBenmansour on 4/15/16.
 */
public class VoxeetWebSocket {

    private static final String TAG = VoxeetWebSocket.class.getSimpleName();

    private Twig twig;

    /**
     * The constant SOCKET_ERROR.
     */
    static final String SOCKET_ERROR = "SOCKET_ERROR";

    /**
     * The constant SOCKET_MESSAGE.
     */
    static final String SOCKET_MESSAGE = "SOCKET_TEXT";

    /**
     * The constant SOCKET_CONNECTED.
     */
    static final String SOCKET_CONNECTED = "SOCKET_CONNECTED";

    /**
     * The constant SOCKET_DISCONNECTED.
     */
    static final String SOCKET_DISCONNECTED = "SOCKET_DISCONNECTED";

    /**
     * The constant SOCKET_STATE_CHANGE.
     */
    static final String SOCKET_STATE_CHANGE = "SOCKET_STATE_CHANGE";


    private Context context;

    private String socketUrl;

    private WebSocketProxy mWebSocketProxy;


    private List<ConnectListener> mConnectListeners;
    private ConnectListener mConnectListener;

    private SocketListener listener = new SocketListener() {

        @Override
        public void onConnect(WebSocket webSocket) {
            VoxeetDispatcher.dispatch(SOCKET_CONNECTED, "Connected");
        }

        @Override
        public void onTextMessage(String message) {
            VoxeetDispatcher.dispatch(SOCKET_MESSAGE, message);
        }

        @Override
        public void onError(WebSocketException exception) {
            exception.printStackTrace();
            VoxeetDispatcher.dispatch(SOCKET_ERROR, "Error " + exception.getMessage());
        }

        @Override
        public void onClose() {
            //TODO close event ?
        }

        @Override
        public void onStateChanged(WebSocketState newState) {
            VoxeetDispatcher.dispatch(SOCKET_STATE_CHANGE, newState.name());
        }

        @Override
        public void onDisconnected() {
            twig.e("Websocket disconnect by server");

            VoxeetDispatcher.dispatch(SOCKET_DISCONNECTED, "on Disconnected by server: ");
        }
    };

    /**
     * Instantiates a new Voxeet web socket.
     *
     * @param context the context
     * @param sdk
     */
    public VoxeetWebSocket(final Context context, VoxeetSdkTemplate sdk, String socketUrl) {
        this.context = context;
        this.twig = sdk.getTwig();

        this.socketUrl = socketUrl;

        mConnectListeners = new ArrayList<>();
        //TODO flush listeners when OK?
        mConnectListener = new ConnectListener() {
            @Override
            public void onConnect(WebSocket websocket) {
                for (ConnectListener listener : mConnectListeners) {
                    listener.onConnect(websocket);
                }
            }

            @Override
            public void onConnectError(Exception e) {
                for (ConnectListener listener : mConnectListeners) {
                    listener.onConnectError(e);
                }
            }

            @Override
            public void onNoNetwork() {
                for (ConnectListener listener : mConnectListeners) {
                    listener.onNoNetwork();
                }
            }
        };
    }

    /**
     * Connect the socket.
     *
     * @param userToken the user token
     */
    public void connect(final String userToken, final String jwtUserToken) {
        connect(userToken, jwtUserToken, null);
    }

    /**
     * Connect the socket.
     *
     * @param userToken           the user token
     * @param jwtUserToken        the user jwt token
     * @param otherSocketListener alternative socket listener
     */
    public void connect(@NonNull final String userToken, @NonNull final String jwtUserToken, @Nullable ConnectListener otherSocketListener) {
        if (mWebSocketProxy == null) {
            if (otherSocketListener != null) {
                mConnectListeners.add(otherSocketListener);
            }
            createProxy();
            mWebSocketProxy.connect(userToken, jwtUserToken, mConnectListener);
        } else {
            switch (mWebSocketProxy.getState()) {
                case CONNECTED:
                    if (otherSocketListener != null) {
                        otherSocketListener.onConnect(mWebSocketProxy.getWebSocket());
                    }
                    break;
                case CONNECTING:
                    if (otherSocketListener != null && !mConnectListeners.contains(otherSocketListener)) {
                        mConnectListeners.add(otherSocketListener);
                    }
                    break;
                case CLOSED:
                    clear();
                    createProxy();
                    mWebSocketProxy.connect(userToken, jwtUserToken, mConnectListener);
                    break;
            }
        }
    }

    private void clear() {
        VoxeetDispatcher.dispatch(SOCKET_STATE_CHANGE, WebSocketState.CLOSED.name());
        if (null != mWebSocketProxy) {
            mWebSocketProxy.cancel();
            mWebSocketProxy.disconnect();
            mWebSocketProxy.removeListener(listener);
            mWebSocketProxy.removeListeners();
        }
        mWebSocketProxy = null;
    }

    private void createProxy() {
        mWebSocketProxy = new WebSocketProxy(context, null, socketUrl);
        mWebSocketProxy.addListener(listener);
    }

    /**
     * Close the web socket.
     */
    public void close(boolean set_disconnect) {
        //TODO manage set_disconnect ?
        Log.d(TAG, "close: mWebSocketProxy := " + mWebSocketProxy);
        clear();
    }

    public boolean isOpen() {
        return mWebSocketProxy != null && mWebSocketProxy.isOpen();
    }

    private boolean sendPing() {
        return mWebSocketProxy != null && mWebSocketProxy.sendPing();
    }

    /**
     * Returns the socket status.
     *
     * @return the boolean
     */
    public boolean isInit() {
        return mWebSocketProxy != null && sendPing();
    }

}
