package voxeet.com.sdk.core;

import android.content.Context;
import android.os.Handler;

import com.neovisionaries.ws.client.WebSocket;
import com.neovisionaries.ws.client.WebSocketAdapter;
import com.neovisionaries.ws.client.WebSocketException;
import com.neovisionaries.ws.client.WebSocketExtension;
import com.neovisionaries.ws.client.WebSocketFactory;
import com.neovisionaries.ws.client.WebSocketFrame;
import com.neovisionaries.ws.client.WebSocketState;

import org.greenrobot.eventbus.EventBus;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import voxeet.com.sdk.events.error.SocketConnectErrorEvent;
import voxeet.com.sdk.utils.DeviceStateUtils;
import voxeet.com.sdk.utils.Twig;

/**
 * Created by RomainBenmansour on 4/15/16.
 */
public class VoxeetWebSocket {

    private static final String TAG = VoxeetWebSocket.class.getSimpleName();

    private static Twig twig;

    /**
     * The constant SOCKET_ERROR.
     */
    public static final String SOCKET_ERROR = "SOCKET_ERROR";

    /**
     * The constant SOCKET_MESSAGE.
     */
    public static final String SOCKET_MESSAGE = "SOCKET_TEXT";

    /**
     * The constant SOCKET_CONNECTED.
     */
    public static final String SOCKET_CONNECTED = "SOCKET_CONNECTED";

    /**
     * The constant SOCKET_DISCONNECTED.
     */
    public static final String SOCKET_DISCONNECTED = "SOCKET_DISCONNECTED";

    /**
     * The constant SOCKET_STATE_CHANGE.
     */
    public static final String SOCKET_STATE_CHANGE = "SOCKET_STATE_CHANGE";

    /**
     * DEFAULT SOCKET RETRY DELAY
     */
    private static long SOCKET_RETRY_DEFAULT = 250;

    /**
     * MAX SOCKET RETRY DELAY
     */
    private static long SOCKET_RETRY_MAX = 8000;

    /**
     * CURRENT SOCKET RETRY DELAY
     */
    private static long SOCKET_RETRY_CURRENT = SOCKET_RETRY_DEFAULT;

    private final int TIMEOUT = 5000;

    private final Context context;

    private final EventBus eventBus;
    private final VoxeetSdkTemplate sdk;
    private String socketUrl;

    private WebSocket webSocket;

    private ExecutorService es;

    private Handler handler = new Handler();

    private Runnable runnable;

    private String userToken;

    private static boolean shouldDisconnect = false;

    private WebSocketAdapter listener = new WebSocketAdapter() {

        @Override
        public void onConnected(WebSocket websocket, Map<String, List<String>> headers) throws Exception {
            super.onConnected(websocket, headers);

            VoxeetDispatcher.dispatch(SOCKET_CONNECTED, "Connected");
        }

        @Override
        public void onTextMessage(WebSocket websocket, String message) throws Exception {
            VoxeetDispatcher.dispatch(SOCKET_MESSAGE, message);
        }

        @Override
        public void onError(WebSocket websocket, WebSocketException cause) throws Exception {
            super.onError(websocket, cause);

            VoxeetDispatcher.dispatch(SOCKET_ERROR, "Error " + cause.getMessage());
        }

        @Override
        public void onDisconnected(WebSocket websocket, WebSocketFrame serverCloseFrame, WebSocketFrame clientCloseFrame, boolean closedByServer) throws Exception {
            super.onDisconnected(websocket, serverCloseFrame, clientCloseFrame, closedByServer);

            twig.e("Websocket disconnect");

            VoxeetDispatcher.dispatch(SOCKET_DISCONNECTED, "on Disconnected by server: " + closedByServer);

            webSocket.removeListener(listener);
            webSocket = null;

            if (!shouldDisconnect)
                handler.postDelayed(runnable, SOCKET_RETRY_CURRENT);
        }

        @Override
        public void onStateChanged(WebSocket websocket, WebSocketState newState) throws Exception {
            super.onStateChanged(websocket, newState);

            VoxeetDispatcher.dispatch(SOCKET_STATE_CHANGE, newState.name());
        }
    };

    /**
     * Instantiates a new Voxeet web socket.
     *
     * @param context the context
     * @param sdk
     */
    public VoxeetWebSocket(final Context context, VoxeetSdkTemplate sdk, String socketUrl) {
        this.context = context;
        this.sdk = sdk;
        this.twig = sdk.getTwig();

        this.socketUrl = socketUrl;

        // Prepare an ExecutorService.
        es = Executors.newSingleThreadExecutor();

        eventBus = EventBus.getDefault();

        runnable = new Runnable() {
            public void run() {
                if (DeviceStateUtils.isNetworkAvailable(context)) { // No point trying to reconnect if no network is available
                    connect(userToken, new SocketListener() {

                        @Override
                        public void onConnect() {
                            twig.i("Socket reconnected successfully");

                            handler.removeCallbacks(runnable);

                            SOCKET_RETRY_CURRENT = SOCKET_RETRY_DEFAULT;
                        }

                        @Override
                        public void onError(String error) {
                            twig.e("Socket failed to reconnect");

                            SOCKET_RETRY_CURRENT = Math.min(SOCKET_RETRY_CURRENT * 2, SOCKET_RETRY_MAX);

                            handler.postDelayed(runnable, SOCKET_RETRY_CURRENT);
                        }
                    });
                } else {
                    twig.e("Socket failed to reconnect");

                    SOCKET_RETRY_CURRENT = Math.min(SOCKET_RETRY_CURRENT * 2, SOCKET_RETRY_MAX);

                    handler.postDelayed(runnable, SOCKET_RETRY_CURRENT);
                }
            }
        };
    }

    private synchronized void initWebSocket() {
        try {
            WebSocketFactory factory = new WebSocketFactory();

            factory.setConnectionTimeout(TIMEOUT);

            webSocket = factory.createSocket(socketUrl);
        } catch (IOException e) {
            twig.e(e);

            webSocket = null;
        }
    }

    /**
     * Connect the socket.
     *
     * @param userToken      the user token
     * @param socketListener the socket listener
     */
    public void connect(final String userToken, SocketListener socketListener) {
        synchronized (this) {
            close(); //reset socket state

            this.userToken = userToken;

            if (DeviceStateUtils.isNetworkAvailable(context)) {
                initWebSocket();

                webSocket.addExtension(WebSocketExtension.PERMESSAGE_DEFLATE);
                webSocket.clearHeaders();
                webSocket.addHeader("Voxeet-Token", userToken.replaceAll("\"", ""));
                webSocket.addListener(listener);

                // Connect to the server asynchronously.
                Future<WebSocket> future = webSocket.connect(es);

                try {
                    // Wait for the opening handshake to complete.
                    future.get();

                    initPing();

                    if (socketListener != null)
                        socketListener.onConnect();

                    shouldDisconnect = false;
                    sdk.getUploadToken();

                    sdk.resetServices();

                    twig.i("Voxeet websocket successfully opened");
                } catch (ExecutionException | InterruptedException e) {
                    if (socketListener != null) // if listener is not null, custom handling of the error
                        socketListener.onError(e.getMessage());
                    else // default handling
                        handler.postDelayed(runnable, SOCKET_RETRY_CURRENT);

                    close();

                    twig.e(e);

                    eventBus.post(new SocketConnectErrorEvent(e.getMessage()));
                }
            } else
                eventBus.post(new SocketConnectErrorEvent("No network"));
        }
    }

    /**
     * Close the web socket.
     */
    public void close() {
        synchronized (this) {
            if (webSocket != null) {
                shouldDisconnect = true;

                webSocket.disconnect();
            }
        }
    }

    public boolean isOpen() {
        return webSocket != null && webSocket.isOpen();
    }

    private boolean sendPing() {
        return webSocket != null && webSocket.sendPing().getState() == WebSocketState.OPEN;
    }

    /**
     * Returns the socket status.
     *
     * @return the boolean
     */
    public boolean isInit() {
        return webSocket != null && sendPing();
    }

    private void initPing() {
        if (webSocket != null)
            webSocket.setPingInterval(30000);
    }

    /**
     * The interface Socket listener which can be used when trying to connect to the server.
     */
    public interface SocketListener {
        /**
         * On connect.
         */
        void onConnect();

        /**
         * On error.
         *
         * @param error the error
         */
        void onError(String error);
    }
}
