package voxeet.com.sdk.core;

import android.app.Application;
import android.provider.Settings;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.firebase.iid.FirebaseInstanceId;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import okhttp3.Authenticator;
import okhttp3.Cookie;
import okhttp3.CookieJar;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.Route;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava.RxJavaCallAdapterFactory;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.schedulers.Schedulers;
import voxeet.com.sdk.converter.JacksonConverterFactory;
import voxeet.com.sdk.core.network.IVoxeetRService;
import voxeet.com.sdk.json.GrandTypeEvent;
import voxeet.com.sdk.json.UserInfo;
import voxeet.com.sdk.models.TokenResponse;
import voxeet.com.sdk.models.UserTokenResponse;
import voxeet.com.sdk.networking.DeviceType;
import voxeet.com.sdk.utils.Auth64;

/**
 * Created by Romain Benmansour on 4/15/16.
 */
public class VoxeetHttp {

    private static final String TAG = VoxeetHttp.class.getSimpleName();
    private final VoxeetSdkTemplate sdk;

    /**
     * The Service.
     */
    protected IVoxeetRService service;

    /**
     * The Token response.
     */
    protected TokenResponse tokenResponse;

    private Application appContext;

    private String serverUrl;

    /**
     * The Retrofit.
     */
    public Retrofit retrofit;

    private OkHttpClient client;

    private String appId;

    private String password;

    private VoxeetServiceListener listener;

    private final HashMap<String, List<Cookie>> cookieStore = new HashMap<>();

    /**
     * Instantiates a new Voxeet http.
     *
     * @param appContext the Application Context
     * @param sdk
     * @param appId      the app id
     * @param password   the password
     * @param listener   the listener
     */
    public VoxeetHttp(Application appContext, VoxeetSdkTemplate sdk, String serverUrl, String appId, String password, VoxeetServiceListener listener) {

        this.sdk = sdk;
        this.tokenResponse = null;
        this.serverUrl = serverUrl;
        this.appContext = appContext;


        this.listener = listener;

        this.appId = appId;

        this.password = password;

        this.initClient();

        this.initRetrofit();

        this.initService();
    }

    /**
     * Overrode retrofit client.
     */
    private void initClient() {
        /*HttpLoggingInterceptor interceptor = new HttpLoggingInterceptor();
        interceptor.setLevel(HttpLoggingInterceptor.Level.BODY);*/

        client = new OkHttpClient.Builder()
                .cookieJar(new CookieJar() {

                    @Override
                    public void saveFromResponse(HttpUrl url, List<Cookie> cookies) {
                        String base = url.scheme() + "://" + url.host() + ":" + url.port();

                        cookieStore.put(base, Lists.<Cookie>newArrayList(Iterables.filter(cookies, new Predicate<Cookie>() {
                            @Override
                            public boolean apply(Cookie input) {
                                return input != null && !input.value().equalsIgnoreCase("deleted") && new Date().before(new Date(input.expiresAt()));
                            }
                        })));
                    }

                    @Override
                    public List<Cookie> loadForRequest(HttpUrl url) {
                        String base = url.scheme() + "://" + url.host() + ":" + url.port();

                        List<Cookie> cookies = cookieStore.get(base);

                        return cookies != null ? cookies : new ArrayList<Cookie>();
                    }
                })
                .addNetworkInterceptor(new Interceptor() {
                    @Override
                    public Response intercept(Chain chain) throws IOException {
                        try {
                            if (tokenResponse == null) {
                                return chain.proceed(chain.request());
                            }


                            return chain.proceed(chain.request().newBuilder().addHeader("Authorization", "bearer " + tokenResponse.getToken()).build());
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                        return null;
                    }
                })
                .authenticator(new Authenticator() {
                    @Override
                    public Request authenticate(Route route, Response response) throws IOException {
                        try {
                            Observable<TokenResponse> token = service.getToken(Auth64.serialize(appId, password), new GrandTypeEvent("client_credentials"));

                            tokenResponse = token.toBlocking().single();
                        } catch (Exception e) {
                            throw new IllegalStateException("Invalid credentials");
                        }

                        return response.request().newBuilder().addHeader("Authorization", "bearer " + tokenResponse.getToken()).build();
                    }
                })
                //.addNetworkInterceptor(interceptor)
                .retryOnConnectionFailure(true)
                .readTimeout(0, TimeUnit.SECONDS)
                .writeTimeout(0, TimeUnit.SECONDS)
                .build();
    }

    private void initRetrofit() {

        retrofit = new Retrofit.Builder()
                .addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(serverUrl)
                .client(client)
                .build();
    }

    private void initService() {
        service = retrofit.create(IVoxeetRService.class);
    }

    /**
     * Identify. Similar to a login
     *
     * @param userInfo the user info
     */
    public void identify(UserInfo userInfo) {

        userInfo.setDeviceIdentifier(Settings.Secure.getString(appContext.getContentResolver(), Settings.Secure.ANDROID_ID));

        if (FirebaseController.getInstance().isEnabled()) {
            sdk.getTwig().i("FirebaseInstanceId.getInstance().getToken(): " + FirebaseInstanceId.getInstance().getToken());

            userInfo.setDevicePushToken(FirebaseInstanceId.getInstance().getToken());
        } else
            sdk.getTwig().i("FirebaseApp is not initialized. Make sure to call FirebaseApp.initializeApp(Context) " +
                    "before initializing the VoxeetSDK if you are planning on using FCM.");

        userInfo.setDeviceType(DeviceType.ANDROID);

        final Observable<UserTokenResponse> user = service.identify(userInfo);
        user.subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<UserTokenResponse>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        e.printStackTrace();
                        listener.onIdentifyError(e.getMessage());
                    }

                    @Override
                    public void onNext(UserTokenResponse userTokenResponse) {
                        sdk.getTwig().i("Successful login with id " + userTokenResponse.getId());

                        VoxeetPreferences.setId(userTokenResponse.getId());

                        VoxeetPreferences.saveLoginCookie(userTokenResponse.getUserToken());

                        listener.onIdentifySuccess(userTokenResponse, tokenResponse);

                        List<Cookie> cookies = getCookies(serverUrl);
                        if (cookies != null && cookies.size() > 0)
                            VoxeetPreferences.saveLoginCookie(cookies.get(0).toString());
                    }
                });
    }

    /**
     * Gets cookies.
     *
     * @param url the url
     * @return the cookies
     */
    public List<Cookie> getCookies(String url) {
        for (Object o : cookieStore.entrySet()) {
            Map.Entry pair = (Map.Entry) o;
            if (((String) pair.getKey()).contains(url)) {
                return cookieStore.get(pair.getKey());
            }
        }
        return null;
    }

    /**
     * Create cookie.
     *
     * @param value the value
     * @return the string
     */
    public String createCookie(String value) {
        String url = serverUrl + ":443";

        Cookie cookie = Cookie.parse(HttpUrl.parse(url), value);
        if (cookie != null && cookie.toString().length() > 0) {
            cookieStore.put(url, Collections.singletonList(cookie));
            return cookie.value();
        }
        return null;
    }

    /**
     * Gets client.
     *
     * @return the client
     */

    public OkHttpClient getClient() {
        return client;
    }

    /**
     * Retrieve cookie.
     *
     * @return the string
     */
    public String retrieveCookie() {
        String cookie;
        if ((cookie = VoxeetPreferences.getLoginCookie()) != null && cookie.length() > 0) {
            cookie = createCookie(cookie);
        } else {
            List<Cookie> cookies = getCookies(serverUrl);
            if (cookies != null && cookies.size() > 0) {
                cookie = cookies.get(0).value();
                VoxeetPreferences.saveLoginCookie(cookies.get(0).toString());
            }
        }
        return cookie;
    }

    public Retrofit getRetrofit() {
        return retrofit;
    }

    /**
     * The interface Voxeet service listener.
     */
    public interface VoxeetServiceListener {

        /**
         * On identify success.
         *
         * @param response      the response
         * @param tokenResponse the token response
         */
        void onIdentifySuccess(UserTokenResponse response, TokenResponse tokenResponse);

        /**
         * On identify error.
         *
         * @param error the error
         */
        void onIdentifyError(String error);
    }
}