package voxeet.com.sdk.core;

import org.greenrobot.eventbus.EventBus;

import voxeet.com.sdk.events.BaseEvent;
import voxeet.com.sdk.events.error.SocketErrorEvent;
import voxeet.com.sdk.events.success.BadgeUpdate;
import voxeet.com.sdk.events.success.ConferenceDestroyedPushEvent;
import voxeet.com.sdk.events.success.ConferenceEndedEvent;
import voxeet.com.sdk.events.success.ConferenceUpdated;
import voxeet.com.sdk.events.success.ConferenceUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUserAddedEvent;
import voxeet.com.sdk.events.success.ConferenceUserSwitchEvent;
import voxeet.com.sdk.events.success.ContactAddedEvent;
import voxeet.com.sdk.events.success.FileConvertedEvent;
import voxeet.com.sdk.events.success.FilePresentationChangeEvent;
import voxeet.com.sdk.events.success.FilePresentationStartEvent;
import voxeet.com.sdk.events.success.FilePresentationStopEvent;
import voxeet.com.sdk.events.success.InvitationReceived;
import voxeet.com.sdk.events.success.MeetingActivityAdded;
import voxeet.com.sdk.events.success.MeetingActivityDeleted;
import voxeet.com.sdk.events.success.MeetingActivityUpdated;
import voxeet.com.sdk.events.success.MeetingAddedOrUpdated;
import voxeet.com.sdk.events.success.MeetingDeleted;
import voxeet.com.sdk.events.success.MeetingEventAddedOrUpdated;
import voxeet.com.sdk.events.success.MeetingReadTimeStampUpdated;
import voxeet.com.sdk.events.success.MessageReceived;
import voxeet.com.sdk.events.success.OfferCreatedEvent;
import voxeet.com.sdk.events.success.OwnConferenceStartedEvent;
import voxeet.com.sdk.events.success.OwnConferenceUserSwitchEvent;
import voxeet.com.sdk.events.success.OwnContactRemoved;
import voxeet.com.sdk.events.success.OwnExternalInvitationSent;
import voxeet.com.sdk.events.success.OwnProfileUpdatedEvent;
import voxeet.com.sdk.events.success.OwnUserInvitedEvent;
import voxeet.com.sdk.events.success.ParticipantUpdatedEvent;
import voxeet.com.sdk.events.success.PeerConnectionStatusUpdatedEvent;
import voxeet.com.sdk.events.success.PlayEmoSphereEvent;
import voxeet.com.sdk.events.success.ProfileUpdated;
import voxeet.com.sdk.events.success.QualityUpdatedEvent;
import voxeet.com.sdk.events.success.RecordingStatusUpdate;
import voxeet.com.sdk.events.success.RenegociationEndedEvent;
import voxeet.com.sdk.events.success.RenegociationUpdate;
import voxeet.com.sdk.events.success.SocketConnectEvent;
import voxeet.com.sdk.events.success.SocketDisconnectEvent;
import voxeet.com.sdk.events.success.SocketStateChangeEvent;
import voxeet.com.sdk.events.success.TypingDetectionEvent;
import voxeet.com.sdk.events.success.UserInvitedEvent;
import voxeet.com.sdk.events.success.WhisperInviteAcceptedEvent;
import voxeet.com.sdk.events.success.WhisperInviteDeclinedEvent;
import voxeet.com.sdk.events.success.WhisperInviteReceivedEvent;
import voxeet.com.sdk.events.success.WhisperLeftEvent;
import voxeet.com.sdk.factories.EventsFactory;
import voxeet.com.sdk.json.BadgeUpdatedEvent;
import voxeet.com.sdk.json.BroadcastEvent;
import voxeet.com.sdk.json.ConferenceDestroyedPush;
import voxeet.com.sdk.json.ConferenceEnded;
import voxeet.com.sdk.json.ConferenceMessageReceived;
import voxeet.com.sdk.json.ConferenceUserAdded;
import voxeet.com.sdk.json.ConferenceUserSwitch;
import voxeet.com.sdk.json.ContactAdded;
import voxeet.com.sdk.json.Event;
import voxeet.com.sdk.json.EventNames;
import voxeet.com.sdk.json.FileAdded;
import voxeet.com.sdk.json.FileAddedEvent;
import voxeet.com.sdk.json.FileConverted;
import voxeet.com.sdk.json.FileDeleted;
import voxeet.com.sdk.json.FileDeletedEvent;
import voxeet.com.sdk.json.FilePresentationChange;
import voxeet.com.sdk.json.FilePresentationStart;
import voxeet.com.sdk.json.FilePresentationStop;
import voxeet.com.sdk.json.InvitationReceivedEvent;
import voxeet.com.sdk.json.MeetActivityAddedEvent;
import voxeet.com.sdk.json.MeetActivityDeletedEvent;
import voxeet.com.sdk.json.MeetActivityUpdatedEvent;
import voxeet.com.sdk.json.MeetingAddedOrUpdatedEvent;
import voxeet.com.sdk.json.MeetingDeletedEvent;
import voxeet.com.sdk.json.MeetingEventAddedOrUpdatedEvent;
import voxeet.com.sdk.json.MeetingReadTimeStampUpdatedEvent;
import voxeet.com.sdk.json.OfferCreated;
import voxeet.com.sdk.json.OwnConferenceCreated;
import voxeet.com.sdk.json.OwnConferenceUserSwitch;
import voxeet.com.sdk.json.OwnContactRemovedEvent;
import voxeet.com.sdk.json.OwnExternalInvitationSentEvent;
import voxeet.com.sdk.json.OwnProfileUpdated;
import voxeet.com.sdk.json.OwnUserInvited;
import voxeet.com.sdk.json.ParticipantUpdated;
import voxeet.com.sdk.json.PeerConnectionStatusUpdated;
import voxeet.com.sdk.json.PlayEmoSphere;
import voxeet.com.sdk.json.ProfileUpdatedEvent;
import voxeet.com.sdk.json.QualityUpdated;
import voxeet.com.sdk.json.RecordingStatusUpdateEvent;
import voxeet.com.sdk.json.TypingDetection;
import voxeet.com.sdk.json.UserInvited;
import voxeet.com.sdk.json.WhisperInvitationAccepted;
import voxeet.com.sdk.json.WhisperInvitationDeclined;
import voxeet.com.sdk.json.WhisperInvitationReceived;
import voxeet.com.sdk.json.WhisperLeft;

/**
 * Created by RomainBenmansour on 4/15/16.
 */
public class VoxeetDispatcher {

    private static final String TAG = VoxeetDispatcher.class.getSimpleName();

    //TODO possible EventBus creation for specific SDK tunnel
    private static EventBus eventBus = EventBus.getDefault();

    /**
     * Map and dispatch events through the eventbus.
     *
     * @param eventName the event name
     * @param message   the message
     */
    public static void dispatch(String eventName, String message) {
        BaseEvent event;
        switch (eventName) {
            case VoxeetWebSocket.SOCKET_DISCONNECTED:
                event = handleDisconnect(message);
                break;
            case VoxeetWebSocket.SOCKET_ERROR:
                event = handleError(message);
                break;
            case VoxeetWebSocket.SOCKET_MESSAGE:
                event = handleMessage(message);
                break;
            case VoxeetWebSocket.SOCKET_CONNECTED:
                event = handleSocketConnect(message);
                break;
            case VoxeetWebSocket.SOCKET_STATE_CHANGE:
                event = handleStateChange(message);
                break;
            default:
                throw new IllegalStateException("Unknown event type");
        }

        eventBus.post(event);
    }

    /**
     * Dispatch a socket error event.
     */
    private static BaseEvent handleError(String message) {
        return new SocketErrorEvent(message);
    }

    /**
     * Dispatch a socket disconnected event.
     */
    private static BaseEvent handleDisconnect(String message) {
        return new SocketDisconnectEvent(message);
    }

    /**
     * Dispatch a socket state change event.
     */
    private static BaseEvent handleStateChange(String message) {
        return new SocketStateChangeEvent(message);
    }

    /**
     * Dispatch a socket connected event.
     */
    private static BaseEvent handleSocketConnect(String message) {
        return new SocketConnectEvent(message);
    }

    /**
     * Dispatch all other events.
     */
    private static BaseEvent handleMessage(String message) {
        BaseEvent busEvent;

        Event event = EventsFactory.decode(message);
        switch (event.getType()) {
            case EventNames.OFFER_CREATED:
                busEvent = new OfferCreatedEvent(message, (OfferCreated) event);
                break;
            case EventNames.CONFERENCE_MESSAGE_RECEIVED:
                busEvent = handleConferenceMessage((ConferenceMessageReceived) event);
                break;
            case EventNames.CONFERENCE_DESTROYED:
                busEvent = new ConferenceDestroyedPushEvent(message, (ConferenceDestroyedPush) event);
                break;
            case EventNames.CONTACT_PROFILE_UPDATED:
                busEvent = new ProfileUpdated(message, (ProfileUpdatedEvent) event);
                break;
            case EventNames.PARTICIPANT_UPDATED:
                ParticipantUpdated participantUpdated = (ParticipantUpdated) event;
                busEvent = new ParticipantUpdatedEvent(message, participantUpdated);
                break;
            case EventNames.OWN_CONFERENCE_CREATED:
                busEvent = new OwnConferenceStartedEvent(message, (OwnConferenceCreated) event);
                break;
            case EventNames.OWN_PROFILE_UPDATED:
                busEvent = new OwnProfileUpdatedEvent(message, (OwnProfileUpdated) event);
                break;
            case EventNames.OWN_EXTERNAL_INVITATION:
                busEvent = new OwnExternalInvitationSent(message, (OwnExternalInvitationSentEvent) event);
                break;
            case EventNames.OWN_CONTACT_REMOVED:
                busEvent = new OwnContactRemoved(message, (OwnContactRemovedEvent) event);
                break;
            case EventNames.MEETING_ADDED_OR_UPDATED:
                busEvent = new MeetingAddedOrUpdated(message, (MeetingAddedOrUpdatedEvent) event);
                break;
            case EventNames.MEETING_DELETED:
                busEvent = new MeetingDeleted(message, (MeetingDeletedEvent) event);
                break;
            case EventNames.BADGE_UPDATE_EVENT:
                busEvent = new BadgeUpdate(message, (BadgeUpdatedEvent) event);
                break;
            case EventNames.RENEGOCIATION_ENDED:
                busEvent = new RenegociationUpdate(message, (RenegociationEndedEvent) event);
                break;
            case EventNames.BROADCAST_EVENT:
                busEvent = handleBroadcast((BroadcastEvent) event);
                break;
            case EventNames.RECORDING_STATUS_UPDATE:
                busEvent = new RecordingStatusUpdate(message, (RecordingStatusUpdateEvent) event);
                break;
            case EventNames.MEET_ACTIVITY_DELETED:
                busEvent = new MeetingActivityDeleted(message, (MeetActivityDeletedEvent) event);
                break;
            case EventNames.MEET_ACTIVITY_UPDATED:
                busEvent = new MeetingActivityUpdated(message, (MeetActivityUpdatedEvent) event);
                break;
            case EventNames.MEET_ACTIVITY_ADDED:
                busEvent = new MeetingActivityAdded(message, (MeetActivityAddedEvent) event);
                break;
            case EventNames.MEET_READ_TIMESTAMP_UPDATED:
                busEvent = new MeetingReadTimeStampUpdated(message, (MeetingReadTimeStampUpdatedEvent) event);
                break;
            case EventNames.INVITATION_RECEIVED:
                busEvent = new InvitationReceived(message, (InvitationReceivedEvent) event);
                break;
            case EventNames.MEETING_EVENT_ADDED_OR_UPDATED:
                busEvent = new MeetingEventAddedOrUpdated(message, (MeetingEventAddedOrUpdatedEvent) event);
                break;
            case EventNames.FILE_DELETED:
                busEvent = new FileDeleted(message, (FileDeletedEvent) event);
                break;
            case EventNames.FILE_ADDED:
                busEvent = new FileAdded(message, (FileAddedEvent) event);
                break;
            case EventNames.QUALITY_UPDATED:
                busEvent = new QualityUpdatedEvent(message, (QualityUpdated) event);
                break;
            case EventNames.WHISPER_INVITATION_RECEIVED:
                busEvent = new WhisperInviteReceivedEvent(message, (WhisperInvitationReceived) event);
                break;
            case EventNames.WHISPER_INVITATION_ACCEPTED:
                busEvent = new WhisperInviteAcceptedEvent(message, (WhisperInvitationAccepted) event);
                break;
            case EventNames.WHISPER_INVITATION_DECLINED:
                busEvent = new WhisperInviteDeclinedEvent(message, (WhisperInvitationDeclined) event);
                break;
            case EventNames.WHISPER_LEFT:
                busEvent = new WhisperLeftEvent(message, (WhisperLeft) event);
                break;
            case EventNames.OWN_USER_INVITED:
                busEvent = new OwnUserInvitedEvent(message, (OwnUserInvited) event);
                break;
            case EventNames.USER_INVITED:
                busEvent = new UserInvitedEvent(message, (UserInvited) event);
                break;
            case EventNames.PARTICIPANT_SWITCH:
                busEvent = new ConferenceUserSwitchEvent(message, (ConferenceUserSwitch) event);
                break;
            case EventNames.PARTICIPANT_ADDED:
                busEvent = new ConferenceUserAddedEvent(message, (ConferenceUserAdded) event);
                break;
            case EventNames.OWN_USER_SWITCH:
                busEvent = new OwnConferenceUserSwitchEvent(message, (OwnConferenceUserSwitch) event);
                break;
            case EventNames.CONFERENCE_ENDED:
                busEvent = new ConferenceEndedEvent(message, (ConferenceEnded) event);
                break;
            case EventNames.CONFERENCE_UPDATED:
                busEvent = new ConferenceUpdatedEvent(message, (ConferenceUpdated) event);
                break;
            case EventNames.FILE_CONVERTED:
                busEvent = new FileConvertedEvent(message, (FileConverted) event);
                break;
            case EventNames.CONTACT_ADDED:
                busEvent = new ContactAddedEvent(message, (ContactAdded) event);
                break;
            case EventNames.PEER_CONNECTION_UPDATED:
                busEvent = new PeerConnectionStatusUpdatedEvent(message, (PeerConnectionStatusUpdated) event);
                break;
            default:
                throw new IllegalStateException("Unknown type of event");
        }
        return busEvent;
    }

    /**
     * Dispatch conference related events
     */
    private static BaseEvent handleConferenceMessage(ConferenceMessageReceived event) {
        Event event1 = EventsFactory.decode(event.getMessage());
        if (event1 == null)
            return new MessageReceived(event.getUserId(), event.getConferenceId(), event.getMessage());
        else {
            switch (event1.getType()) {
                case EventNames.FILE_PRESENTATION_CHANGE:
                    return new FilePresentationChangeEvent(event.getMessage(), event.getUserId(), event.getConferenceId(), (FilePresentationChange) event1);
                case EventNames.FILE_PRESENTATION_START:
                    return new FilePresentationStartEvent(event.getMessage(), event.getUserId(), event.getConferenceId(), (FilePresentationStart) event1);
                case EventNames.FILE_PRESENTATION_STOP:
                    return new FilePresentationStopEvent(event.getMessage(), event.getUserId(), event.getConferenceId(), (FilePresentationStop) event1);
                case EventNames.PLAY_EMO_SPHERE:
                    return new PlayEmoSphereEvent(event.getMessage(), event.getUserId(), event.getConferenceId(), (PlayEmoSphere) event1);
                default:
                    return new MessageReceived(event.getUserId(), event.getConferenceId(), event.getMessage());
            }
        }
    }

    /**
     * Dispatch a conference broadcast based event.
     */
    private static BaseEvent handleBroadcast(BroadcastEvent event) {
        Event event1 = EventsFactory.decode(event.getMessage());
        switch (event1.getType()) {
            case EventNames.TYPING_DETECTION:
                return new TypingDetectionEvent(event.getMessage(), (TypingDetection) event1, event.getUserId(), event.getMeetingId());
            default:
                throw new IllegalStateException("Unknown broadcast type event");
        }
    }
}
