package voxeet.com.sdk.core.abs;

import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;
import android.util.Log;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.voxeet.android.media.Media;
import com.voxeet.android.media.MediaException;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.VideoRenderer;
import com.voxeet.android.media.peer.PendingPeerCallback;
import com.voxeet.android.media.peer.SdpCandidate;
import com.voxeet.android.media.peer.SdpDescription;
import com.voxeet.android.media.peer.SdpMessage;
import com.voxeet.android.media.settings.AudioSettings;
import com.voxeet.android.media.video.CameraEnumerationAndroid;
import com.voxeet.android.media.video.EglBase;
import com.voxeet.android.media.video.VideoCapturerAndroid;
import com.voxeet.kernel.R;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import okhttp3.ResponseBody;
import retrofit2.Response;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.schedulers.Schedulers;
import voxeet.com.sdk.audio.MessagingEnvironment;
import voxeet.com.sdk.core.VoxeetPreferences;
import voxeet.com.sdk.core.VoxeetSdkTemplate;
import voxeet.com.sdk.core.services.AbstractVoxeetClientService;
import voxeet.com.sdk.core.services.SdkConferenceService;
import voxeet.com.sdk.events.ConferenceTimeoutUserJoinedEvent;
import voxeet.com.sdk.events.error.CameraSwitchErrorEvent;
import voxeet.com.sdk.events.error.ConferenceCreatedError;
import voxeet.com.sdk.events.error.ConferenceJoinedError;
import voxeet.com.sdk.events.error.ConferenceLeftError;
import voxeet.com.sdk.events.error.GetConferenceHistoryErrorEvent;
import voxeet.com.sdk.events.error.GetConferenceStatusErrorEvent;
import voxeet.com.sdk.events.error.ParticipantAddedErrorEvent;
import voxeet.com.sdk.events.error.ReplayConferenceErrorEvent;
import voxeet.com.sdk.events.error.SdkLogoutErrorEvent;
import voxeet.com.sdk.events.error.SubscribeConferenceErrorEvent;
import voxeet.com.sdk.events.success.AddConferenceParticipantResultEvent;
import voxeet.com.sdk.events.success.CameraSwitchSuccessEvent;
import voxeet.com.sdk.events.success.ConferenceCreationSuccess;
import voxeet.com.sdk.events.success.ConferenceDestroyedPushEvent;
import voxeet.com.sdk.events.success.ConferenceEndedEvent;
import voxeet.com.sdk.events.success.ConferenceJoinedSuccessEvent;
import voxeet.com.sdk.events.success.ConferenceLeftSuccessEvent;
import voxeet.com.sdk.events.success.ConferencePreJoinedEvent;
import voxeet.com.sdk.events.success.ConferenceRefreshedEvent;
import voxeet.com.sdk.events.success.ConferenceUserAddedEvent;
import voxeet.com.sdk.events.success.ConferenceUserCallDeclinedEvent;
import voxeet.com.sdk.events.success.ConferenceUserJoinedEvent;
import voxeet.com.sdk.events.success.ConferenceUserLeftEvent;
import voxeet.com.sdk.events.success.ConferenceUserQualityUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUserUpdatedEvent;
import voxeet.com.sdk.events.success.ConferenceUsersInvitedEvent;
import voxeet.com.sdk.events.success.DeclineConferenceResultEvent;
import voxeet.com.sdk.events.success.GetConferenceHistoryEvent;
import voxeet.com.sdk.events.success.GetConferenceStatusEvent;
import voxeet.com.sdk.events.success.InvitationReceived;
import voxeet.com.sdk.events.success.OfferCreatedEvent;
import voxeet.com.sdk.events.success.OwnConferenceStartedEvent;
import voxeet.com.sdk.events.success.ParticipantAddedEvent;
import voxeet.com.sdk.events.success.ParticipantUpdatedEvent;
import voxeet.com.sdk.events.success.QualityUpdatedEvent;
import voxeet.com.sdk.events.success.RecordingStatusUpdate;
import voxeet.com.sdk.events.success.ReplayConferenceEvent;
import voxeet.com.sdk.events.success.ResumeConference;
import voxeet.com.sdk.events.success.ScreenStreamAddedEvent;
import voxeet.com.sdk.events.success.ScreenStreamRemovedEvent;
import voxeet.com.sdk.events.success.SdkLogoutSuccessEvent;
import voxeet.com.sdk.events.success.SendBroadcastResultEvent;
import voxeet.com.sdk.events.success.StartRecordingResultEvent;
import voxeet.com.sdk.events.success.StartVideoAnswerEvent;
import voxeet.com.sdk.events.success.StopRecordingResultEvent;
import voxeet.com.sdk.events.success.StopVideoAnswerEvent;
import voxeet.com.sdk.events.success.SubscribeConferenceEvent;
import voxeet.com.sdk.events.success.UnSubscribeConferenceAnswerEvent;
import voxeet.com.sdk.factories.VoxeetIntentFactory;
import voxeet.com.sdk.json.BroadcastEvent;
import voxeet.com.sdk.json.ConferenceUserAdded;
import voxeet.com.sdk.json.DeviceEvent;
import voxeet.com.sdk.json.OfferCreated;
import voxeet.com.sdk.json.RecordingStatusUpdateEvent;
import voxeet.com.sdk.json.SdkConferenceInvitation;
import voxeet.com.sdk.json.SdkConferenceReplayBody;
import voxeet.com.sdk.json.StartVideoResponse;
import voxeet.com.sdk.json.StopVideoResponse;
import voxeet.com.sdk.json.UserInfo;
import voxeet.com.sdk.models.ConferenceQuality;
import voxeet.com.sdk.models.ConferenceResponse;
import voxeet.com.sdk.models.ConferenceType;
import voxeet.com.sdk.models.ConferenceUserStatus;
import voxeet.com.sdk.models.HistoryConference;
import voxeet.com.sdk.models.OfferCandidate;
import voxeet.com.sdk.models.OfferDescription;
import voxeet.com.sdk.models.RecordingStatus;
import voxeet.com.sdk.models.SubscribeConference;
import voxeet.com.sdk.models.abs.ConferenceUser;
import voxeet.com.sdk.models.impl.DefaultConference;
import voxeet.com.sdk.models.impl.DefaultConferenceUser;
import voxeet.com.sdk.networking.DeviceType;
import voxeet.com.sdk.utils.ConferenceListener;
import voxeet.com.sdk.utils.Twig;
import voxeet.com.sdk.utils.Validate;

/**
 * Created by RomainBenmansour on 4/18/16.
 *
 * @see SdkConferenceService
 */
public abstract class AbstractConferenceSdkService<T, COP extends AbstractConferenceSdkObservableProvider<T, DemoEvent>, DemoEvent>
        extends AbstractVoxeetClientService<T>
        implements SdkConferenceService, ConferenceListener {

    private final String TAG = AbstractConferenceSdkService.class.getSimpleName();
    private VoxeetSdkTemplate sdk;
    private ConferenceListener listener;
    private AbstractConferenceSdkObservableProvider conference_observable_provider;

    protected Context context;

    private Twig twig;

    private DefaultConference conference;

    private String conferenceId = null;

    private String conferenceAlias = null;

    private EventBus eventBus;

    private Media media;

    private EglBase eglBase = EglBase.create();

    private final MessagingEnvironment environment;

    private boolean listenerMode = false;

    private boolean isVideoOn = false; // default camera setting

    private boolean isMicrophoneOn = true;

    private boolean isRecording = false;

    private boolean inConference = false;

    private long timeOutTimer = -1;

    private Runnable timeoutRunnable = new Runnable() {
        @Override
        public void run() {
            twig.i("No conference user joined after " + timeOutTimer
                    + "ms. Now Attempting to leave the conference");

            leave();

            //conference was left after Xms
            //note that if you change the timeout after calling the join, the
            //value returned is absolutely inaccurate
            eventBus.post(new ConferenceTimeoutUserJoinedEvent(timeOutTimer));
        }
    };

    private Media.MediaStreamListener mediaStreamListener = new Media.MediaStreamListener() {
        @Override
        public void onStreamAdded(final String peer, final MediaStream stream) {
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    DefaultConferenceUser user = updateConferenceParticipants(peer, ConferenceUserStatus.ON_AIR);
                    if (user != null) {
                        getTwig().i("New conference user joined with id: " + user.getUserId());

                        if (!peer.equalsIgnoreCase(VoxeetPreferences.id()) && timeOutTimer != -1) {
                            twig.i("Cancelling timeout timer");

                            removeTimeoutCallbacks();
                        }

                        eventBus.post(new ConferenceUserJoinedEvent(user, stream));
                    }
                }
            });
        }

        @Override
        public void onStreamUpdated(final String peer, final MediaStream stream) {
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    DefaultConferenceUser user = findUserById(peer);
                    if (user != null) {
                        getTwig().i("Conference user updated with id: " + user.getUserId());

                        eventBus.post(new ConferenceUserUpdatedEvent(user, stream));
                    }
                }
            });
        }

        @Override
        public void onStreamRemoved(final String peer) {
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    DefaultConferenceUser user = findUserById(peer);
                    if (user != null) {

                        startTransactionConferenceUser();
                        user.setConferenceStatus(ConferenceUserStatus.LEFT);

                        if (conference.getConferenceUsers().contains(user))
                            conference.getConferenceUsers().remove(user);

                        getTwig().i("Conference user left with id: " + user.getUserId());
                        commitTransactionConferenceUser();

                        eventBus.post(new ConferenceUserLeftEvent(user));
                    }
                }
            });
        }

        @Override
        public void onScreenStreamAdded(final String peer, final MediaStream stream) {
            getTwig().i("Screen share stream added: " + peer);

            eventBus.post(new ScreenStreamAddedEvent(peer, stream));
        }

        @Override
        public void onScreenStreamRemoved(String peer) {
            getTwig().i("Screen share stream removed: " + peer);

            eventBus.post(new ScreenStreamRemovedEvent(peer));
        }
    };
    private boolean hasRequestedInvitation = false;
    private String mDefaultCamera;

    public AbstractConferenceSdkService(VoxeetSdkTemplate instance, COP conference_observable_provider, long timeout, T service) {
        super(instance, service);

        mDefaultCamera = CameraEnumerationAndroid.getNameOfFrontFacingDevice();

        this.conference_observable_provider = conference_observable_provider;

        conference_observable_provider.setRetrofitInstantiatedProvider(getService());

        this.listener = instance;
        this.sdk = instance;
        this.timeOutTimer = timeout;

        this.twig = instance.getTwig();

        this.eventBus = EventBus.getDefault();

        this.context = instance.getApplicationContext();

        this.environment = new MessagingEnvironment.Builder(context)
                //.setServerHost(R.string.connectionManagerHost_prod)
                .setServerHost(R.string.session_manager_url)
                .setServerPort(R.string.connectionManagerPort)
                .setStunHost(R.string.stunHost)
                .setStunPort(R.string.stunPort)
                .build();

        register();
    }

    @Override
    public void setListenerMode(boolean isListener) {
        isMicrophoneOn = !isListener;
    }

    @Override
    public void listenConference(String conferenceId) {
        listenerMode = true;

        getTwig().i("Listener mode set to true");

        join(conferenceId);
    }

    protected void setMediaStreamListener() {
        Validate.notNull(media, "media");

        if (media != null) {
            media.setMediaStreamListener(mediaStreamListener);
        }
    }

    @Override
    public boolean attachMediaStream(MediaStream stream, VideoRenderer.Callbacks render) {
        if (media != null) {
            media.attachMediaStream(render, stream);
            return true;
        }
        return false;
    }

    @Override
    public boolean unAttachMediaStream(MediaStream stream, VideoRenderer.Callbacks render) {
        if (media != null) {
            media.unattachMediaStream(render, stream);
            return true;
        }
        return false;
    }

    @Override
    public void decline(final String conferenceId) {
        final Observable<Response<ResponseBody>> user = conference_observable_provider.getDeclineObservable(conferenceId);
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<Response<ResponseBody>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e(e);

                        eventBus.post(new DeclineConferenceResultEvent(handleError(e), false));
                    }

                    @Override
                    public void onNext(Response<ResponseBody> response) {
                        getTwig().i("IConference declined with id: " + conferenceId);

                        eventBus.post(new DeclineConferenceResultEvent(response.code() == 200));
                    }
                });
    }

    @Override
    public void register() {
        registerEventBus();
    }

    @Override
    public void unregister() {
        unRegisterEventBus();
    }

    protected DefaultConferenceUser updateConferenceParticipants(String userId, ConferenceUserStatus status) {
        DefaultConferenceUser user = findUserById(userId);
        if (user != null) {
            startTransactionConferenceUser();
            user.setIsOwner(conference.getOwnerProfile() == null || conference.getOwnerProfile().getUserId().equals(user.getProfile().getUserId()));
            user.setConferenceStatus(status);
            user.setQuality(ConferenceQuality.ULTRA_HIGH.StringValue());
            commitTransactionConferenceUser();

            return user;
        }
        return null;
    }

    protected void initMedia() {
        Validate.notNull(context, "context");

        Validate.notNull(environment, "environment");

        Validate.notNull(VoxeetPreferences.id(), "user id");

        if (media == null) {
            try {
                media = new Media(VoxeetPreferences.id(), context, isVideoOn, isMicrophoneOn, new AudioSettings(environment.getStunHost(), environment.getStunPort()));
                setMediaStreamListener();

                getTwig().i("Media started with video set to: " + isVideoOn + " and microphone to: " + isMicrophoneOn);
            } catch (MediaException e) {
                getTwig().e(e);
            }
        }
    }

    public void setDefaultCamera(String cameraName) {
        mDefaultCamera = cameraName;
    }

    public void startVideo() {
        Validate.notNull(media, "media");

        media.startVideo(mDefaultCamera);

        final Observable<StartVideoResponse> stopVideo = conference_observable_provider.getStartVideoObservable(conferenceId, VoxeetPreferences.id());
        stopVideo.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<StartVideoResponse>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        twig.e(e);

                        eventBus.post(new StartVideoAnswerEvent(false));
                    }

                    @Override
                    public void onNext(StartVideoResponse response) {
                        media.startVideo(mDefaultCamera);

                        isVideoOn = true;

                        createVideoAsnwer(response.getUserId(), response.getDescription(), response.getCandidates());

                        eventBus.post(new StartVideoAnswerEvent(true));
                    }
                });
    }

    public void stopVideo() {
        String conferenceId = getCurrentConferenceId();

        Validate.notNull(conferenceId, "conference Id");

        Validate.notNull(media, "media");

        media.stopVideo();

        final Observable<StopVideoResponse> stopVideo = conference_observable_provider.getStopVideoObservable(conferenceId, VoxeetPreferences.id());
        stopVideo.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<StopVideoResponse>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e(e);

                        eventBus.post(new StopVideoAnswerEvent(false));
                    }

                    @Override
                    public void onNext(StopVideoResponse response) {
                        media.stopVideo();

                        isVideoOn = false;

                        createVideoAsnwer(response.getUserId(), response.getDescription(), response.getCandidates());

                        //removed in bintay_v2
                        //eventBus.post(new StopVideoAnswerEvent(true));
                    }
                });
    }


    private void createVideoAsnwer(final String userId, final OfferDescription offerDescription, final List<OfferCandidate> offerCandidates) {
        Validate.notNull(media, "media");

        SdpDescription description = new SdpDescription(offerDescription.getType(), offerDescription.getSdp());

        List<SdpCandidate> candidates = new ArrayList<>();
        if (offerCandidates != null) {
            for (OfferCandidate candidate : offerCandidates) {
                candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
            }
        }

        media.createAnswerForPeer(userId,
                description.getSsrc(),
                description,
                candidates,
                VoxeetPreferences.id().equalsIgnoreCase(userId),
                new PendingPeerCallback() {
                    @Override
                    public void onMessage(SdpMessage message) {
                        answer(userId, message);
                    }
                });
    }

    @Override
    public void toggleVideo() {
        if (isVideoOn)
            stopVideo();
        else if (Validate.hasCameraPermissions(context))
            startVideo();
    }

    @Override
    public boolean create() {
        getTwig().i("Attempting to create conference");

        conference_observable_provider.getCreateConferenceObservable()
                .subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<ConferenceResponse>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        twig.e(e);

                        eventBus.post(new ConferenceCreatedError(handleError(e)));
                    }

                    @Override
                    public void onNext(ConferenceResponse response) {
                        conference = createConference();

                        conferenceId = response.getConfId();
                        conferenceAlias = response.getConfAlias();

                        conference.setConferenceId(conferenceId);
                        conference.setConferenceAlias(conferenceAlias);

                        getTwig().i("IConference created with id: " + conferenceId);

                        eventBus.post(new ConferenceCreationSuccess(response.getConfId(), response.getConfAlias()));

                        onCreationSuccess(response.getConfId());
                    }
                });
        return true;
    }

    @Override
    public void demo() {
        getTwig().i("Attempting to create demo conference");

        final Observable<DemoEvent> user = conference_observable_provider.getCreateDemoObservable();
        user.subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<DemoEvent>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e(e);

                        eventBus.post(new ConferenceCreatedError(handleError(e)));
                    }

                    @Override
                    public void onNext(DemoEvent response) {
                        AbstractConferenceSdkService.this.onCreateDemoSuccess(response);
                    }
                });
    }

    protected abstract void onCreateDemoSuccess(DemoEvent response);

    @Override
    public void invite(List<String> voxeetIds, final List<String> externalIds) {
        //remove the timeout, a new one should be started when user will be invited
        removeTimeoutCallbacks();

        final rx.Observable<Response<ResponseBody>> user = conference_observable_provider.getInviteObservable(conferenceId, new SdkConferenceInvitation(voxeetIds, externalIds));
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<Response<ResponseBody>>() {
                    @Override
                    public void onCompleted() {
                        //Do nothing
                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e(e);

                        eventBus.post(new AddConferenceParticipantResultEvent(handleError(e), false));
                    }

                    @Override
                    public void onNext(Response<ResponseBody> response) {
                        eventBus.post(new AddConferenceParticipantResultEvent(response.code() == 200));

                        if (response.code() == 200) {
                            for (String userId : externalIds) {
                                if (!userId.equals(VoxeetPreferences.id())) {
                                    getTwig().i("Conference participant with id: " + userId + " invited");

                                    eventBus.post(new ConferenceRefreshedEvent(updateConferenceParticipants(userId, ConferenceUserStatus.IN_PROGRESS)));
                                }
                            }
                        }

                        if (timeOutTimer != -1) {
                            getTwig().i("scheduling timer to leave the conference in " + timeOutTimer);

                            handler.postDelayed(timeoutRunnable, timeOutTimer);
                        }
                    }
                });
    }

    @Override
    public void logout() {
        if (VoxeetPreferences.isLoggedIn()) {
            twig.i("Attempting to logout");

            final Observable<Response<ResponseBody>> user = conference_observable_provider.getLogOutObservable(VoxeetPreferences.token());
            user.subscribeOn(Schedulers.newThread())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(new Subscriber<Response<ResponseBody>>() {
                        @Override
                        public void onCompleted() {
                        }

                        @Override
                        public void onError(Throwable e) {
                            getTwig().e(e);

                            eventBus.post(new SdkLogoutErrorEvent(handleError(e)));
                        }

                        @Override
                        public void onNext(Response<ResponseBody> response) {
                            String message;
                            if (response.code() == 200) {
                                message = "Logout success";

                                getTwig().i(message);

                                VoxeetPreferences.onLogout();

                                getVoxeetSDK().closeSocket();

                                eventBus.post(new SdkLogoutSuccessEvent(message));
                            } else {
                                message = "Logout failed";

                                getTwig().e(message);

                                eventBus.post(new SdkLogoutErrorEvent(message));
                            }
                        }
                    });
        } else {
            final String message = "Already logged out";
            getTwig().e(message);
            eventBus.post(new SdkLogoutSuccessEvent("Already logged out"));
        }
    }


    @Override
    public void replay(final String conferenceId, final long offset) {
        listenerMode = true;
        setListenerMode(false);

        final rx.Observable<ReplayConferenceEvent> user = conference_observable_provider.getReplayObservable(conferenceId, new SdkConferenceReplayBody(offset));
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<ReplayConferenceEvent>() {
                    @Override
                    public void onCompleted() {
                        //Do nothing
                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Failed to replay conference");

                        eventBus.post(new ReplayConferenceErrorEvent(handleError(e)));
                    }

                    @Override
                    public void onNext(ReplayConferenceEvent response) {
                        getTwig().i("Success server answer for replaying conference with id: " +
                                conferenceId + "at offset: " + offset);

                        join(response.getConferenceId());
                    }
                });
    }

    public void startRecording() {
        Validate.notNull(conferenceId, "conference Id");

        final Observable<Response<ResponseBody>> user = conference_observable_provider.getStartRecordingObservable(conferenceId);
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<Response<ResponseBody>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Failed to start conference recording");

                        eventBus.post(new StartRecordingResultEvent(handleError(e), false));
                    }

                    @Override
                    public void onNext(Response<ResponseBody> response) {
                        if (response.code() == 200)
                            getTwig().i("IConference recording started ");

                        eventBus.post(new StartRecordingResultEvent(response.code() == 200));
                    }
                });
    }

    public void stopRecording() {
        Validate.notNull(conferenceId, "conference Id");

        final Observable<Response<ResponseBody>> user = conference_observable_provider.getStopRecordingObservable(conferenceId);
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<Response<ResponseBody>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Failed to cancel conference recording");

                        eventBus.post(new StopRecordingResultEvent(handleError(e), false));
                    }

                    @Override
                    public void onNext(Response<ResponseBody> response) {
                        if (response.code() == 200)
                            getTwig().i("Conference recording stopped");

                        eventBus.post(new StopRecordingResultEvent(response.code() == 200));
                    }
                });
    }

    @Override
    public boolean muteConference(boolean mute) {
        if (mute && media != null && !media.isMuted()) {
            twig.i("Conference muted");

            media.muteRecording();
        } else if (media != null) {
            twig.i("Conference unmuted");

            media.unMuteRecording();
        }
        return true;
    }

    @Override
    public boolean isMuted() {
        return media != null && media.isMuted();
    }

    @Override
    public boolean isUserMuted(String userId) {
        DefaultConferenceUser user = findUserById(userId);

        return user != null && user.isMuted();
    }

    @Override
    public void getConferenceStatus(String conferenceId) {
        final Observable<GetConferenceStatusEvent> user = conference_observable_provider.getConferenceStatusObservable(conferenceId);
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<GetConferenceStatusEvent>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Error while getting conference history");

                        eventBus.post(new GetConferenceStatusErrorEvent(handleError(e)));
                    }

                    @Override
                    public void onNext(GetConferenceStatusEvent response) {
                        for (DefaultConferenceUser conferenceUser : response.getConferenceUsers()) {
                            getTwig().e(conferenceUser.getUserId() + " / " + conferenceUser.getStatus());
                        }
                        getTwig().i("Dispatching conference status result");

                        eventBus.post(response);
                    }
                });
    }

    @Override
    public String getCurrentConferenceId() {
        Validate.runningOnUiThread();

        //TODO when toggling video, conference.getConferenceId() is null but not conferenceId
        if (conference != null && conference.getConferenceId() != null)
            return conference.getConferenceId();

        return conferenceId;
    }

    @Override
    public int getConferenceRoomSize() {
        if (conference != null)
            return conference.getConferenceRoomSize();
        return 0;
    }

    @Override
    public void conferenceHistory(String conferenceId) {
        final Observable<List<HistoryConference>> user = conference_observable_provider.getConferenceHistoryObservable(conferenceId);
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<List<HistoryConference>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Error while retrieving conference history");

                        eventBus.post(new GetConferenceHistoryErrorEvent(handleError(e)));
                    }

                    @Override
                    public void onNext(List<HistoryConference> response) {
                        getTwig().i("Dispatching conference history response");

                        eventBus.post(new GetConferenceHistoryEvent(response));
                    }
                });
    }

    @Override
    public String currentSpeaker() {
        //TODO difference between both implementation
        //in one it was checking for more than 2 participants AND peerVuMeter > 100 not in the other
        //for contextual purposes, I removed the "if there are no more than 2 people, return the current one"
        //this predicate would make the only active user set to the current user !
        if (media == null || conference == null || conference.getConferenceUsers() == null) {
            return VoxeetPreferences.id();
        } else {
            String currentSpeaker = null;

            for (DefaultConferenceUser user : conference.getConferenceUsers()) {
                if (user.getUserId() != null) {
                    int peerVuMeter = media.getPeerVuMeter(user.getUserId());
                    if (currentSpeaker == null || (peerVuMeter > 100 && media.getPeerVuMeter(currentSpeaker) < peerVuMeter))
                        currentSpeaker = user.getUserId();
                }
            }
            return currentSpeaker;
        }
    }

    @Override
    public int getSdkPeerVuMeter(String peerId) {
        Validate.runningOnUiThread();

        if (media != null)
            return media.getPeerVuMeter(peerId);
        return 0;
    }

    @Override
    public DefaultConferenceUser findUserById(final String userId) {
        Validate.notNull(userId, "user id");

        if (conference != null && conference.getConferenceUsers() != null) {
            return Iterables.find(conference.getConferenceUsers(), new Predicate<DefaultConferenceUser>() {
                @Override
                public boolean apply(DefaultConferenceUser input) {
                    return userId.equalsIgnoreCase(input.getUserId());
                }
            }, null);
        }

        return null;
    }

    @Override
    public boolean muteUser(String userId, boolean shouldMute) {
        Validate.notNull(userId, "user id");
        Validate.runningOnUiThread();

        DefaultConferenceUser user = findUserById(userId);
        if (user != null) {
            float muteValue;
            if (shouldMute)
                muteValue = MUTE_FACTOR;
            else
                muteValue = UNMUTE_FACTOR;

            startTransactionConferenceUser();
            user.setMuted(shouldMute);

            getTwig().i("Setting mute property for conference participant with id " + userId + " to " + shouldMute);

            media.changePeerGain(userId, muteValue);
            commitTransactionConferenceUser();

            return true;
        }
        return false;
    }

    @Override
    public String getAliasId() {
        return conferenceAlias;
    }

    @Override
    public void switchCamera() {
        Validate.notNull(media, "media");

        media.switchCamera(new VideoCapturerAndroid.CameraSwitchHandler() {
            @Override
            public void onCameraSwitchDone(boolean isFrontCamera) {
                getTwig().i("Successfully switched camera");

                eventBus.post(new CameraSwitchSuccessEvent(isFrontCamera));
            }

            @Override
            public void onCameraSwitchError(String errorDescription) {
                getTwig().e("Failed to switch camera");

                eventBus.post(new CameraSwitchErrorEvent(errorDescription));
            }
        });
    }

    @Override
    public String getConferenceId() {
        return conferenceId;
    }

    @Override
    public List<Media.AudioRoute> getAvailableRoutes() {
        return media != null ? media.availableRoutes() : new ArrayList<Media.AudioRoute>();
    }

    @Override
    public Media.AudioRoute currentRoute() {
        Media.AudioRoute route = Media.AudioRoute.ROUTE_PHONE;
        if (media != null) {
            route = media.outputRoute();
        }

        return route;
    }

    @Override
    public boolean setAudioRoute(Media.AudioRoute route) {
        Validate.runningOnUiThread();

        if (media != null) {
            getTwig().i("Changing audio route to " + route);

            media.setOutputRoute(route);

            return true;
        }
        return false;
    }

    @Override
    public void subscribe(String conferenceId) {
        final Observable<SubscribeConference> user = conference_observable_provider.getSubscribeObservable(conferenceId);
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<SubscribeConference>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Failed to subscribe to this conference's events");

                        eventBus.post(new SubscribeConferenceErrorEvent(handleError(e)));
                    }

                    @Override
                    public void onNext(SubscribeConference response) {
                        getTwig().i("You are no subscribed to this conference's events");

                        eventBus.post(new SubscribeConferenceEvent(response));
                    }
                });
    }

    @Override
    public void unSubscribe() {
        final Observable<Response<ResponseBody>> user = conference_observable_provider.getUnSubscribeObservable(conferenceId);
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<Response<ResponseBody>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Failed to unsubscribe from this conference's events");

                        eventBus.post(new UnSubscribeConferenceAnswerEvent(false, handleError(e)));
                    }

                    @Override
                    public void onNext(Response<ResponseBody> response) {
                        getTwig().i("You are no longer subscribed to this conference's events");

                        eventBus.post(new UnSubscribeConferenceAnswerEvent(response.code() == 200));
                    }
                });
    }

    @Override
    public void join(final String conferenceId) {
        Validate.notNull(conferenceId, "conference Id");

        //remove the timeout, a new one should be started when user will be invited
        removeTimeoutCallbacks();

        if (inConference) {
            twig.e("Already joining/joined a conference");

            return;
        }

        this.inConference = true;

        this.conferenceId = conferenceId;

        initMedia();

        final Observable<ResumeConference> user = conference_observable_provider.joinConference(conferenceId, new DeviceEvent(DeviceType.ANDROID, listenerMode));
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<ResumeConference>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        e.printStackTrace();
                        getTwig().e("Failed to Join conference with id " + conferenceId);

                        inConference = false;

                        closeMedia();

                        eventBus.post(new ConferenceJoinedError(handleError(e)));
                    }

                    @Override
                    public void onNext(ResumeConference response) {
                        if (conference == null) conference = createConference();

                        conference.setConferenceId(response.getConferenceId());
                        conference.setConferenceAlias(response.getConferenceId());

                        AbstractConferenceSdkService.this.conferenceId = conference.getConferenceId();
                        AbstractConferenceSdkService.this.conferenceAlias = conference.getConferenceAlias();


                        if (response.getConferenceId() != null) {
                            AbstractConferenceSdkService.this.conferenceId = response.getConferenceId();
                        }

                        if (response.getConferenceAlias() != null) {
                            AbstractConferenceSdkService.this.conferenceAlias = response.getConferenceAlias();
                        }

                        eventBus.post(new ConferencePreJoinedEvent(getConferenceId(), getAliasId()));

                        inConference = true;

                        setAudioRoute(Media.AudioRoute.ROUTE_PHONE);

                        getTwig().i("Joined conference with id " + getConferenceId());


                        eventBus.post(new ConferenceJoinedSuccessEvent(getConferenceId(), getAliasId()));
                    }
                });
    }

    @Override
    public EglBase.Context getEglContext() {
        Validate.notNull(eglBase, "egl base");

        return eglBase.getEglBaseContext();
    }

    @Override
    public boolean setTimeOut(long timeout) {
        getTwig().i("Timeout set to " + timeout);

        this.timeOutTimer = timeout;

        return true;
    }

    @Override
    public long getTimeout() {
        return timeOutTimer;
    }

    @Override
    public void getInvitedUsers() {
        final Observable<GetConferenceStatusEvent> user = conference_observable_provider.getConferenceStatusObservable(conferenceId);
        user.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<GetConferenceStatusEvent>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Error while retrieving connecting users");

                        eventBus.post(new ConferenceUsersInvitedEvent(null));
                    }

                    @Override
                    public void onNext(GetConferenceStatusEvent response) {
                        Iterable<DefaultConferenceUser> users = Iterables.filter(response.getConferenceUsers(), new Predicate<DefaultConferenceUser>() {
                            @Override
                            public boolean apply(DefaultConferenceUser input) {
                                return input.getStatus().equalsIgnoreCase(ConferenceUserStatus.CONNECTING.toString());
                            }
                        });

                        twig.i("Successfully retrieved connecting users");

                        eventBus.post(new ConferenceUsersInvitedEvent(users));
                    }
                });
    }

    protected void answer(final String peer, SdpMessage message) {
        Validate.notNull(conferenceId, "conference Id");

        final Observable<Response<ResponseBody>> user = conference_observable_provider.answerConference(conferenceId, peer, message.getDescription());
        user.subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<Response<ResponseBody>>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("Offer created for " + peer + " : KO");

                        eventBus.post(new ParticipantAddedErrorEvent(handleError(e)));
                    }

                    @Override
                    public void onNext(Response<ResponseBody> response) {
                        if (response.code() != 200) {
                            getTwig().i("Offer created for " + peer + " : KO");

                            eventBus.post(new ParticipantAddedErrorEvent(response.code() + ""));
                        } else {
                            getTwig().i("Offer created for " + peer + " : OK");
                        }
                    }
                });
    }

    @Override
    public void sendBroadcastMessage(String message) {
        Validate.notNull(conferenceId, " conference Id");

        Validate.notNull(message, "message");

        final Observable<Response<ResponseBody>> user = conference_observable_provider.broadcastMessage(conferenceId, new BroadcastEvent(message));
        user.subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<Response<ResponseBody>>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("The broadcast message has not been sent");

                        eventBus.post(new SendBroadcastResultEvent(handleError(e), false));
                    }

                    @Override
                    public void onNext(Response<ResponseBody> response) {
                        getTwig().i("The broadcast message has been sent");

                        eventBus.post(new SendBroadcastResultEvent(response.code() == 200));
                    }
                });
    }

    /**
     * This method attempts to leave the current conference
     * <p>
     * The previous version used a boolean, the new method fingerprint is now void
     */
    @Override
    public void leave() {
        //remove the timeout, a new one should be started when user will be invited
        //in the next conferece
        removeTimeoutCallbacks();

        getTwig().i("Attempting to leave conference with conference id " + conferenceId);

        final Observable<Response<ResponseBody>> user = conference_observable_provider.leaveConference(conferenceId);
        user.subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<Response<ResponseBody>>() {

                    private boolean success = false;

                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        getTwig().e("having throwable", e);
                        e.printStackTrace();
                        getTwig().e("Something went wrong while leaving the conference");

                        closeMedia();

                        if (!success) {
                            eventBus.post(new ConferenceLeftError(handleError(e)));
                        }
                    }

                    @Override
                    public void onNext(Response<ResponseBody> response) {
                        getTwig().i("Conference left successfully");

                        int leftInConference = 0;
                        DefaultConference conference = getConference();
                        if (conference != null) {
                            leftInConference = getConference().getConferenceUsers().size() - 1;
                        }

                        closeMedia();

                        if (response.code() == 200 || response.code() == 404) {
                            eventBus.post(new ConferenceLeftSuccessEvent(leftInConference));
                            success = true; //when having 404, the above onError is sent !
                        } else
                            eventBus.post(new ConferenceLeftError(response.code() + ""));
                    }
                });
    }


    protected void closeMedia() {
        if (media != null) {
            try {
                if (conference != null) {
                    for (DefaultConferenceUser user : conference.getConferenceUsers()) {
                        if (user != null && user.getUserId() != null)
                            media.removePeer(user.getUserId());
                    }
                }
            } catch (Exception ex) {
                twig.e(ex);
            } finally {
                media.stop();
                media = null;

                isMicrophoneOn = true;

                getTwig().i("Microphone value restored to true");

                isVideoOn = false;

                twig.i("Video value restored to false");

                listenerMode = false;

                getTwig().i("Listener mode set back to false");
            }
        }
        //media = null has been removed in bintay_v2
        //media = null;

        getTwig().i("Media closed");

        inConference = false;

        conference = null;

    }

    protected void handleAnswer(final OfferCreated offer) throws MediaException {
        if (media != null) {
            SdpDescription description = new SdpDescription(offer.getDescription().getType(), offer.getDescription().getSdp());

            List<SdpCandidate> candidates = new ArrayList<>();
            for (OfferCandidate candidate : offer.getCandidates()) {
                candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
            }

            media.createAnswerForPeer(offer.getUserId(),
                    description.getSsrc(),
                    description,
                    candidates,
                    offer.isMaster(),
                    new PendingPeerCallback() {
                        @Override
                        public void onMessage(SdpMessage message) {
                            try {
                                if (conference == null) {
                                    conference = createConference();
                                }

                                if (conference == null)
                                    conference = createConference();

                                if (offer.getUserId().contains("11111"))
                                    conference.getConferenceUsers().add(createConferenceUser(offer.getUserId(), offer.getDevice(), new UserInfo("Julie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_1.png")));
                                else if (offer.getUserId().contains("22222"))
                                    conference.getConferenceUsers().add(createConferenceUser(offer.getUserId(), offer.getDevice(), new UserInfo("Sophie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png")));
                                else if ((offer.getUserId().contains("33333")))
                                    conference.getConferenceUsers().add(createConferenceUser(offer.getUserId(), offer.getDevice(), new UserInfo("Mike", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_3.png")));
                                else
                                    conference.getConferenceUsers().add(createConferenceUser(offer.getUserId(), offer.getDevice(), new UserInfo(offer.getName(), offer.getExternalId(), offer.getAvatarUrl())));


                                changePeerPosition(offer.getUserId(), 0, 0.5);

                                answer(offer.getUserId(), message);
                            } catch (Exception e) {
                                Log.d(TAG, "onMessage: unlockPeerOperation" + e.getMessage());
                            }
                        }
                    });
        }
    }

    @Override
    public void toggleRecording() {
        if (isRecording)
            stopRecording();
        else
            startRecording();
    }

    @Override
    public boolean changePeerPosition(final String userId, final double angle, final double distance) {
        if (media != null) {
            media.changePeerPosition(userId, angle, distance);
            return true;
        }
        return false;
    }

    //TODO CHECK WETHER SDK OR NOT
    //removed in bintay_v2
    /*@Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceUserSwitchEvent event) {
        if (media != null && event.getEvent().getUserId() != null)
            media.removePeer(event.getEvent().getUserId());
    }*/

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceived invitation) {
        DefaultConference conference = invitation.getEvent().getConference();
        if (!invitation.getEvent().getUserId().equals(VoxeetPreferences.id())
                || conference.getConferenceType().equals(ConferenceType.SCHEDULED.value())) {
            getTwig().i("Invitation received from " + invitation.getEvent().getUserId());

            hasRequestedInvitation = true;

            getConferenceStatus(conference.getConferenceId());
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final GetConferenceStatusEvent event) {
        if (!isLive() && hasRequestedInvitation && event.getConferenceUsers().size() > 0) {
            DefaultConferenceUser conferenceUser = event.getConferenceUsers().get(0);
            Map<String, String> infos = new HashMap<>();
            infos.put(VoxeetIntentFactory.CONF_ID, event.getConferenceId());
            infos.put(VoxeetIntentFactory.INVITER_ID, conferenceUser.getUserInfo().getExternalId());
            infos.put(VoxeetIntentFactory.INVITER_NAME, conferenceUser.getUserInfo().getName());
            infos.put(VoxeetIntentFactory.INVITER_URL, conferenceUser.getUserInfo().getAvatarUrl());

            getTwig().i("Starting default incoming view");

            Intent intent = VoxeetIntentFactory.buildFrom(context, VoxeetPreferences.getDefaultActivity(), infos);
            if (intent != null)
                context.startActivity(intent);

            hasRequestedInvitation = false;
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(OwnConferenceStartedEvent ownConferenceStartedEvent) {
        if (conference == null)
            conference = createConference();

        getTwig().i("Own conference started");

        conference.setConferenceInfos(ownConferenceStartedEvent.event().getConferenceInfos());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final OfferCreatedEvent event) {
        try {
            handleAnswer(event.offer());
        } catch (MediaException e) {
            e.printStackTrace();
            getTwig().e(e);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserAddedEvent event) {
        ConferenceUserAdded content = event.getEvent();
        if (null != content) {
            DefaultConferenceUser user = findUserById(event.getEvent().getUserId());

            if (null == user) {
                user = new DefaultConferenceUser(content.getUserId(), null);
                user.setIsOwner(false);
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));

                conference.getConferenceUsers().add(user);
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantUpdatedEvent event) {
        DefaultConferenceUser user = findUserById(event.getUserId());
        final ConferenceUserStatus status = ConferenceUserStatus.valueOf(event.getStatus());

        boolean was_online = false;

        try {
            was_online = ConferenceUserStatus.ON_AIR.equals(user.getConferenceStatus());
        } catch (Exception e) {
            //Exception while converting the status
            //report this issue to voxeet
        }


        if (user != null) {
            startTransactionConferenceUser();
            user.setStatus(event.getStatus());
            commitTransactionConferenceUser();
        } else {
            getTwig().i("Not in conference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
        }

        switch(status) {
            case LEFT:
            case DECLINE:
                if (media != null && user != null) {
                    getTwig().i("In conference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
                    media.removePeer(event.getUserId());
                }
                break;
            default:
                getTwig().i("status not managed updated to " + event.getStatus());

        }

        if(ConferenceUserStatus.DECLINE.equals(status) || !was_online) {
            if (!event.getUserId().equals(VoxeetPreferences.id())) {
                getTwig().i("Conference user with id: " + event.getUserId() + " declined the call");

                eventBus.post(new ConferenceUserCallDeclinedEvent(event.getConfId(), event.getUserId(), event.getStatus()));
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantAddedEvent event) {
        DefaultConferenceUser user = event.getUser();
        if (user != null) {
            twig.i("Conference user joined with id: " + user.getUserId());

            MediaStream stream = null;
            if (null != media) {
                HashMap<String, MediaStream> streams = media.getMapOfStreams();
                if (streams.containsKey(user.getUserId())) {
                    stream = media.getMapOfStreams().get(user.getUserId());
                }
            }

            eventBus.post(new ConferenceUserJoinedEvent(user, stream));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceDestroyedPushEvent event) {
        closeMedia();

        getTwig().i("Conference has ended");
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceEndedEvent event) {
        closeMedia();

        getTwig().i("Conference has ended");

        eventBus.post(event.getEvent());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final QualityUpdatedEvent event) {
        for (DefaultConferenceUser conferenceUser : event.getEvent().getUser()) {
            DefaultConferenceUser user = findUserById(conferenceUser.getUserId());
            if (user != null) {
                twig.i("Quality updated for " + user.getUserId());

                startTransactionConferenceUser();
                user.setQuality(conferenceUser.getQuality());
                commitTransactionConferenceUser();

                eventBus.post(new ConferenceUserQualityUpdatedEvent(user));
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final RecordingStatusUpdate event) {
        RecordingStatusUpdateEvent recordingStatusUpdateEvent = event.getEvent();
        DefaultConferenceUser user = findUserById(recordingStatusUpdateEvent.getUserId());
        if (user != null) {
            RecordingStatus status = RecordingStatus.valueOf(recordingStatusUpdateEvent.getRecordingStatus());
            if (status == RecordingStatus.RECORDING) {
                isRecording = true;
                conference.setStartRecordTimestamp(new Date(recordingStatusUpdateEvent.getTimeStamp()));
                conference.setRecordingStatus(RecordingStatus.RECORDING);
                conference.setRecordingUser(recordingStatusUpdateEvent.getUserId());
                user.setIsRecordingOwner(true);
            } else {
                isRecording = false;
                conference.setStartRecordTimestamp(null);
                conference.setRecordingStatus(RecordingStatus.NOT_RECORDING);
                conference.setRecordingUser(null);
                user.setIsRecordingOwner(false);
            }
        }

        getTwig().i("Conference's recording status has changed to: " + event.getEvent().getRecordingStatus());

        eventBus.post(recordingStatusUpdateEvent);
    }

    public DefaultConference getConference() {
        return conference;
    }

    public int getPeerVuMeter(String peerId) {
        Validate.runningOnUiThread();

        if (media != null)
            return media.getPeerVuMeter(peerId);
        return 0;
    }

    @Override
    public List<ConferenceUser> getConferenceUsers() {
        List<? extends ConferenceUser> users = conference.getConferenceUsers();
        return (List<ConferenceUser>) users;
    }

    @Override
    public boolean isLive() {
        return conference != null;
    }

    @Override
    public boolean isListenerMode() {
        return listenerMode;
    }

    @Override
    public void onCreationSuccess(String conferenceId) {
        this.join(conferenceId);
    }

    //new DefaultConference()
    //new DefaultConferenceRealm()
    protected abstract DefaultConference createConference();

    //conference.getConferenceUsers().add(new DefaultConferenceUser(offer.getUserId(), offer.getDevice(), new UserInfo("Sophie", offer.getExternalId(), "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png")));
    protected abstract DefaultConferenceUser createConferenceUser(String userId, String device, UserInfo userInfo);

    protected Twig getTwig() {
        return twig;
    }

    protected void setConference(DefaultConference conference) {
        this.conference = conference;
        this.conferenceId = conference.getConferenceId();
    }

    protected void setConferenceAlias(String alias) {
        conferenceAlias = alias;
    }

    protected EventBus getEventBus() {
        return eventBus;
    }

    protected ConferenceListener getConferenceListener() {
        return listener;
    }

    protected Media getMedia() {
        return media;
    }


    public ConferenceUser getUser(final String userId) {
        if (conference != null && conference.getConferenceUsers() != null) {
            return Iterables.find(conference.getConferenceUsers(), new Predicate<ConferenceUser>() {
                @Override
                public boolean apply(ConferenceUser input) {
                    return userId.equals(input.getUserId());
                }
            }, null);
        }
        return null;
    }

    protected VoxeetSdkTemplate getVoxeetSDK() {
        return sdk;
    }

    protected abstract void startTransactionConferenceUser();

    protected abstract void commitTransactionConferenceUser();

    @NonNull
    public HashMap<String, MediaStream> getMapOfStreams() {
        return media != null ? media.getMapOfStreams() : new HashMap<String, MediaStream>();
    }

    /**
     * Remove any previous timeout callback
     * <p>
     * To be used when it is mandatory to remove such behaviour
     * for instance, when joining a new conference
     * <p>
     * It happens since we do not want to manage every edge case :
     * - decline
     * - network error
     * - etc...
     */
    private void removeTimeoutCallbacks() {
        handler.removeCallbacks(timeoutRunnable);
    }
}
