package voxeet.com.sdk.promise;

import android.os.Handler;
import android.support.annotation.NonNull;

/**
 * Promise class
 *
 * Accessible methods :
 *
 * then() -> pass the from type to a result type
 * error() -> auto call for execute()
 * execute() -> resolve the promise, the good practice is to use error()
 * setResolveBlock() -> call to resolve() will block or post to next
 */

public class Promise<TYPE> {

    private final Handler mHandler;
    private AbstractPromiseCallbackImpl mPromiseCallback;
    private PromiseSolver<TYPE> mPromiseSolver;
    private Promise mParent;
    private Promise mChild;
    private TYPE mResult;
    private boolean mDone;
    private boolean mRejected;
    private boolean mPosted;
    private boolean mIsSync;



    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Constructors
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    public Promise() {
        mRejected = false;
        mPosted = false;
        mHandler = new Handler();
    }

    public Promise(PromiseSolver<TYPE> solver) {
        this();

        solver.setParent(this);
        mPromiseSolver = solver;

    }

    private Promise(Promise<TYPE> parent) {
        this();

        mParent = parent;
    }

    public Promise(Promise parent, AbstractPromiseCallbackImpl success_promise) {
        this(parent);

        success_promise.setPromise(this);
        mPromiseCallback = success_promise;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public methods
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Chain the promise with a sub promise
     *
     * @param success_promise a non null onSuccess() implementation
     * @return a valid promise to chain
     */
    public <TYPE_RESULT> Promise<TYPE_RESULT> then(@NonNull SuccessPromise<TYPE, TYPE_RESULT> success_promise) {
        mChild = new Promise<TYPE_RESULT>(this, success_promise);
        return mChild.setResolveBlock(mIsSync);
    }

    /**
     * Chain the error_promise to an error manager
     * It will start resolving right away
     *
     * @param error_promise a non null onError() implementation
     */
    public void error(@NonNull ErrorPromise error_promise) {
        mChild = new Promise<TYPE>(this, error_promise)
                .setResolveBlock(mIsSync);
        execute();
    }

    /**
     * Execute the given promise
     */
    public void execute() {
        if (mParent != null) {
            mParent.execute();
        } else if (null != mPromiseSolver) {
            mPromiseSolver.onCall();
        }
    }

    /**
     * Specify to use a blocking or post on the current thread/looper mode
     *
     * @param is_sync true or false only
     * @return the current object to chain calls directly
     */
    public Promise<TYPE> setResolveBlock(boolean is_sync) {
        mIsSync = is_sync;
        return this;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Internal methods
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    void resolve(TYPE result) {
        if (mRejected) return; //no call to resolve if already rejected
        if (mDone) throwError();
        mDone = true;

        setResult(result);
    }

    void reject(Throwable error) {
        mRejected = true;
        if (mDone) throwError();
        mDone = true;

        if (mChild != null) {
            mChild.reject(error);
        } else if (mPromiseCallback != null) {
            mPromiseCallback.onError(error);
        } else {
            error.printStackTrace();
        }
    }

    private void execute(TYPE result) {
        if (mPromiseCallback != null) {
            mPromiseCallback.onSuccess(result);
        }
    }

    private void setResult(TYPE result) {
        mPosted = true;
        mResult = result;

        if (mRejected) return; //no call to resolve if already rejected
        tryNext();
    }


    private void throwError() {
        throw new IllegalPromiseStateException("The promise has already been resolved/rejected");
    }

    private void tryNext() {
        if (mIsSync) {
            if (mPosted && mChild != null) {
                mChild.execute(mResult);
            }
        } else {
            mHandler.post(new Runnable() {
                @Override
                public void run() {
                    if (mPosted && mChild != null) {
                        mChild.execute(mResult);
                    }
                }
            });
        }
    }
}
