package voxeet.com.sdk.core.preferences;

import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.HashMap;

/**
 * Class to wrap preferences
 * <p>
 * prevent issues with flush/ed keys by surrounding cache
 */

public class SharedPreferencesCacheWrapper {
    @NonNull
    private HashMap<String, String> _strings;

    @NonNull
    private HashMap<String, Boolean> _booleans;

    @NonNull
    private HashMap<String, Long> _longs;

    @NonNull
    private HashMap<String, Integer> _ints;

    private SharedPreferences _preferences;

    private SharedPreferencesCacheWrapper() {
        _strings = new HashMap<>();
        _booleans = new HashMap<>();
        _longs = new HashMap<>();
        _ints = new HashMap<>();
    }

    public SharedPreferencesCacheWrapper(SharedPreferences preferences) {
        this();

        _preferences = preferences;
    }

    @NonNull
    public Editor edit() {
        return new Editor(this);
    }

    @Nullable
    public String getString(@NonNull String key, @Nullable String def) {
        if (_strings.containsKey(key))
            return _strings.get(key);
        String result = _preferences.getString(key, def);

        return result;
    }

    public boolean getBoolean(@NonNull String key, boolean def) {
        if (_booleans.containsKey(key))
            return _booleans.get(key);
        return _preferences.getBoolean(key, def);
    }

    public int getInt(@NonNull String key, int def) {
        if (_ints.containsKey(key))
            return _ints.get(key);
        return _preferences.getInt(key, def);
    }

    public long getLong(@NonNull String key, long def) {
        if (_longs.containsKey(key))
            return _longs.get(key);
        return _preferences.getLong(key, def);
    }

    public boolean contains(@NonNull String key) {
        if (_strings.containsKey(key)) return true;
        if (_booleans.containsKey(key)) return true;
        if (_longs.containsKey(key)) return true;
        return _preferences.contains(key);
    }

    public class Editor {
        private SharedPreferencesCacheWrapper _wrapper;
        private SharedPreferences.Editor _editor;

        private Editor() {

        }

        protected Editor(SharedPreferencesCacheWrapper wrapper) {
            this();

            _wrapper = wrapper;
            _editor = wrapper._preferences.edit();
        }

        public Editor putString(@NonNull String key, @Nullable String value) {
            _wrapper._strings.put(key, value);
            _editor.putString(key, value);
            return this;
        }

        public Editor putLong(@NonNull String key, long value) {
            _wrapper._longs.put(key, value);
            _editor.putLong(key, value);
            return this;
        }

        public Editor putBoolean(@NonNull String key, boolean value) {
            _wrapper._booleans.put(key, value);
            _editor.putBoolean(key, value);
            return this;
        }

        public Editor putInt(@NonNull String key, int value) {
            _wrapper._ints.put(key, value);
            _editor.putInt(key, value);
            return this;
        }

        /**
         * Apply the values to the preferences
         */
        public void apply() {
            //when calling apply everything is fine
            //TODO if revert is called, restore values
            _editor.apply();
        }
    }


}