package voxeet.com.sdk.core.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.Media;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.VideoRenderer;
import com.voxeet.android.media.peer.SdpDescription;
import com.voxeet.android.media.video.EglBase;

import java.util.List;

import okhttp3.OkHttpClient;
import retrofit2.Retrofit;
import voxeet.com.sdk.core.AbstractVoxeetService;
import voxeet.com.sdk.json.Event;
import voxeet.com.sdk.models.ConferenceType;
import voxeet.com.sdk.models.abs.Conference;

/**
 * Created by RomainBenmansour on 4/18/16.
 */
public abstract class ConferenceService extends AbstractVoxeetService {

    /**
     * Instantiates a new Voxeet service.
     *
     * @param context       the context
     * @param client        the client
     * @param retrofit
     * @param service_klass
     */
    protected ConferenceService(@NonNull Context context, @Nullable OkHttpClient client, @Nullable Retrofit retrofit, @Nullable Class service_klass) {
        super(context, client, retrofit, service_klass);
    }

    /**
     * Used to leave the ongoing conference.
     */
    abstract protected void leave();

    /**
     * Join a conference using the pstn
     *
     * @param nodeId is the id of the conference
     */
    public abstract void pstn(String nodeId);

    /**
     * Rate the conference
     *
     * @param conferenceId is the id of the conference you wish to rate
     * @param rating       the rate from 0 to 5
     * @param comment      the comment associated with the rating
     */
    public abstract void rate(String conferenceId, int rating, String comment);

    /**
     * Joins the conference as a guest meaning you don't have to be logged
     *
     * @param pincode          the conference alias id
     * @param deviceIdentifier is the phone unique identifier
     */
    public abstract void guest(String pincode, String deviceIdentifier);

    /**
     * Joins the conference
     *
     * @param conferenceId joins the conference associated with this id
     */
    abstract protected void join(String conferenceId);

    /**
     * Joins the conference.
     *
     * @param conferenceId joins the conference associated with this id
     * @param type         the type of conference you trying to join
     */
    public abstract void resumeConference(String conferenceId, final ConferenceType type);

    /**
     * Sends a message to each conference user present in the conference
     *
     * @param message the message
     */
    public abstract void sendMessage(String message);

    /**
     * Sends event to each conference user present in the conference.
     *
     * @param event the event
     */
    public abstract void sendEvent(Event event);

    /**
     * Creates a conference.
     *
     * @param meetingId    the meeting id
     * @param contactIds   the users' contact ids
     * @param contactMails the contact mails if they don't have ids
     * @param message      the message to display with the invitation
     * @param meetNow      Scheduled (true) or default conference
     */
    public abstract void create(String meetingId, List<String> contactIds, List<String> contactMails, String message, boolean meetNow);

    /**
     * Answer to the created offer received.
     *
     * @param userId      the user id of the offer created
     * @param description the description corresponding to the the offer
     */
    abstract protected void answer(final String userId, SdpDescription description);

    /**
     * Creates a demo conference.
     */
    public abstract void demo();

    /**
     * Replays a conference.
     *
     * @param conferenceId the conference id of the conference to be replayed
     */
    public abstract void replay(String conferenceId);

    /**
     * Switch camera.
     */
    public abstract void switchCamera();

    /**
     * Basically the same as declining a call.
     *
     * @param conferenceId the conference id of the conference you wish to decline
     */
    public abstract void later(String conferenceId);

    /**
     * Declines call invitation.
     *
     * @param conferenceId the conference id of the conference you wish to decline
     */
    public abstract void decline(String conferenceId);

    /**
     * Add users to the conference.
     *
     * @param contactIds   the contact ids
     * @param contactMails the contact mails
     */
    public abstract void invite(List<String> contactIds, List<String> contactMails);

    /**
     * Recall a list of participants.
     *
     * @param ids the ids
     */
    public abstract void recall(List<String> ids);

    /**
     * Sends reminder to a list of participants.
     *
     * @param ids the ids
     */
    public abstract void sendReminder(List<String> ids);

    /**
     * Starts recording the conference.
     */
    public abstract void startRecording();

    /**
     * Stops recording the conference.
     */
    public abstract void stopRecording();

    /**
     * Leaves whisper.
     */
    public abstract void leaveWhisper();

    /**
     * Declines whisper.
     *
     * @param userId the user id
     */
    public abstract void declineWhisper(String userId);

    /**
     * Accepts the whisper invitation.
     *
     * @param userId the user id
     */
    public abstract void acceptWhisper(String userId);

    /**
     * Invite someone to whisper.
     *
     * @param userId the user id
     */
    public abstract void inviteWhisper(String userId);

    /**
     * Check if a conference is live
     *
     * @return true if a conference is live
     */
    public abstract boolean isLive();

    /**
     * Register the media subscribes to the eventbus.
     */
    abstract protected void register();

    /**
     * Unregister the media subscribes to the eventbus.
     */
    abstract protected void unregister();

    /**
     * Mute conference recording.
     *
     * @param mute the mute value
     */
    abstract protected void muteConference(boolean mute);

    /**
     * Is muted boolean.
     *
     * @return if you are muted or not
     */
    abstract protected boolean isMuted();

    /**
     * Gets available audio routes.
     *
     * @return the available routes
     */
    public abstract List<Media.AudioRoute> getAvailableRoutes();

    /**
     * Get the current audio route.
     *
     * @return the audio route
     */
    abstract protected Media.AudioRoute currentRoute();

    /**
     * Is recording owner boolean.
     *
     * @return if you are the recording owner or not
     */
    abstract protected boolean isRecordingOwner();

    /**
     * Is conference owner boolean.
     *
     * @return if you are the conference owner or not
     */
    abstract protected boolean isConferenceOwner();

    /**
     * Sets audio route.
     *
     * @param route the route
     */
    public abstract void setAudioRoute(Media.AudioRoute route);

    /**
     * Change peer position.
     *
     * @param userId   the user id
     * @param angle    the angle (between 0 and 1)
     * @param distance the distance
     */
    public abstract boolean changePeerPosition(String userId, double angle, double distance);

    /**
     * Gets current conference id.
     *
     * @return the current conference id
     */
    public abstract String getCurrentConferenceId();

    /**
     * Gets conference.
     *
     * @return the current conference object
     */
    public abstract Conference getConference();

    /**
     * Gets conference room size.
     *
     * @return the conference room size
     */
    public abstract int getConferenceRoomSize();

    /**
     * Mute user.
     *
     * @param userId     the user id
     * @param shouldMute the should mute
     */
    public abstract void muteUser(String userId, boolean shouldMute);

    /**
     * Gets peer vu meter.
     *
     * @param peerId the peer id
     * @return the peer vu meter
     */
    public abstract int getPeerVuMeter(String peerId);

    /**
     * Attach media stream.
     *
     * @param stream the stream
     * @param render the renderer
     */
    public abstract void attachMediaStream(MediaStream stream, VideoRenderer.Callbacks render);

    /**
     * Un attach media stream.
     *
     * @param stream the stream
     * @param render the renderer
     */
    public abstract void unAttachMediaStream(MediaStream stream, VideoRenderer.Callbacks render);

    /**
     * Toggle video.
     */
    public abstract void toggleVideo();

    /**
     * Gets egl context.
     *
     * @return the egl context
     */
    public abstract EglBase.Context getEglContext();

    /**
     * Current speaker string.
     *
     * @return the string
     */
    abstract protected String currentSpeaker();

    //TODO remove isConfPremium if not used in public sdk
    //public abstract boolean isConfPremium();
}
