package voxeet.com.sdk.core.services;

import com.voxeet.android.media.Media.AudioRoute;

import org.webrtc.EglBase;
import org.webrtc.MediaStream;
import org.webrtc.VideoRenderer;

import java.util.List;

import voxeet.com.sdk.models.abs.ConferenceUser;


/**
 * Created by RomainBenmansour on 4/18/16.
 */
public interface SdkConferenceService {

    /**
     * The constant MUTE_FACTOR.
     */
    public static final int MUTE_FACTOR = 10000;

    /**
     * The constant UNMUTE_FACTOR.
     */
    public static final int UNMUTE_FACTOR = 1;


    /**
     * Sets listener mode. To be called before starting a conference
     *
     * @param isListener true to be a listener and not use the user microphone, false otherwise
     */
    void setListenerMode(boolean isListener);

    /**
     * Joins a conference as a listener meaning no camera and microphone for this user.
     *
     * @param conferenceId id of the conference to join as a listener
     */
    public abstract void listenConference(String conferenceId);

    /**
     * Gets sdk peer vu meter.
     *
     * @param peerId the peer id
     * @return the sdk peer vu meter
     */
    int getSdkPeerVuMeter(String peerId);

    /**
     * Gets current conference id.
     *
     * @return the current conference id. Can be null
     */
    String getCurrentConferenceId();

    /**
     * Gets conference room size.
     *
     * @return the conference room size
     */
    int getConferenceRoomSize();

    /**
     * Conference history given for a conference.
     *
     * @param conferenceId the conference id
     */
    void conferenceHistory(String conferenceId);

    /**
     * Get Current speaker id.
     *
     * @return the string
     */
    String currentSpeaker();

    /**
     * Find user by id conference user.
     *
     * @param userId the user id
     * @return the conference user associated with the user id if it exists
     */
    ConferenceUser findUserById(String userId);

    /**
     * Send broadcast message.
     *
     * @param string the string
     */
    void sendBroadcastMessage(String string);

    /**
     * Leave the conference.
     */
    void leave();

    /**
     * Join the sdk conference.
     *
     * @param conferenceId the conference id
     */
    void join(String conferenceId);

    /**
     * Register the media and subscribe to the eventbus.
     */
    void register();

    /**
     * Unregister the media and unsubscribe from the eventbus.
     */
    void unregister();

    /**
     * Create a new conference.
     */
    boolean create();

    /**
     * Create a demo conference.
     */
    void demo();

    /**
     * Logout.
     */
    public abstract void logout();

    /**
     * Replay a conference.
     *
     * @param conferenceId the conference id
     */
    void replay(String conferenceId, long offset);

    /**
     * Mute conference recordings.
     *
     * @param mute the mute
     */
    boolean muteConference(boolean mute);

    /**
     * Check if the user is muted.
     *
     * @return the boolean
     */
    boolean isMuted();

    /**
     * CHeck if the user associated with the userId is muted or not.
     *
     * @param userId the user id
     * @return the boolean
     */
    boolean isUserMuted(String userId);

    /**
     * Gets conference status.
     *
     * @param conferenceId the conference id
     */
    void getConferenceStatus(String conferenceId);

    /**
     * Mute an user.
     *  @param userId     the user id
     * @param shouldMute the should mute
     */
    boolean muteUser(String userId, boolean shouldMute);

    /**
     * Gets alias id for the conference.
     *
     * @return the alias id
     */
    String getAliasId();

    /**
     * Switch camera.
     */
    void switchCamera();

    /**
     * Gets conference id.
     *
     * @return the conference id if it exists. null otherwise
     */
    String getConferenceId();

    /**
     * Gets audio available routes.
     *
     * @return the available routes
     */
    List<AudioRoute> getAvailableRoutes();

    /**
     * Current audio route.
     *
     * @return the audio route
     */
    AudioRoute currentRoute();

    /**
     * Sets audio route.
     *
     * @param route the route
     */
    boolean setAudioRoute(AudioRoute route);

    /**
     * Toggle recording.
     */
    void toggleRecording();

    /**
     * Change peer position.
     *
     * @param userId   the user id
     * @param angle    the angle
     * @param distance the distance
     * @return true if media was initialized
     */
    boolean changePeerPosition(String userId, double angle, double distance);

    /**
     * Gets conference users.
     *
     * @return the conference users
     */
    List<ConferenceUser> getConferenceUsers();

    /**
     * Toggle video.
     */
    void toggleVideo();

    /**
     * Check if conference is live.
     *
     * @return the boolean
     */
    boolean isLive();

    /**
     * Check if conference is listening
     */
    boolean isListenerMode();

    /**
     * Subscribe to a conference's events.
     *
     * @param conferenceId the conference id
     */
    void subscribe(String conferenceId);

    /**
     * Unsubscribe from a conference's events.
     */
    void unSubscribe();

    /**
     * Attach media stream.
     *  @param stream the stream
     * @param render the render
     */
    boolean attachMediaStream(MediaStream stream, VideoRenderer.Callbacks render);

    /**
     * Unattach media stream.
     *  @param stream the stream
     * @param render the render
     */
    boolean unAttachMediaStream(MediaStream stream, VideoRenderer.Callbacks render);

    void decline(String conferenceId);

    void invite(List<String> voxeetIds, List<String> externalIds);

    /**
     * Gets egl context.
     *
     * @return the egl context
     */
    EglBase.Context getEglContext();

    /**
     * Define a timeout to leave a call if none joins the conference
     *
     * @param timeout the time to wait before leaving the call
     */
    boolean setTimeOut(long timeout);

    /**
     * @return the time to wait before leaving the call
     */
    long getTimeout();

    /**
     * @return asynchronously list of invited users to the conference
     */
    void getInvitedUsers();
}
