package voxeet.com.sdk.core;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.fasterxml.jackson.databind.ObjectMapper;

import org.greenrobot.eventbus.EventBus;

import java.io.IOException;
import java.util.Date;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import okhttp3.OkHttpClient;
import okhttp3.ResponseBody;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava.HttpException;

/**
 * Created by RomainB on 5/13/16.
 *
 * @param <T> is the class which will be instantiated
 */
public abstract class AbstractVoxeetService<T> {

    private static final String TAG = AbstractVoxeetService.class.getSimpleName();

    /**
     * The Client.
     */
    protected OkHttpClient client;

    /**
     * The Event bus.
     */
    protected EventBus eventBus;

    /**
     * The Retrofit.
     */
    protected Retrofit retrofit;

    private T service;

    /**
     * last time we asked the API for data
     */
    private Date lastUpdate;

    private boolean isLoaded = false;

    private int UPDATE_TIME = 5 * 60 * 1000;

    /**
     * The Handler.
     */
    protected final static Handler handler = new Handler(Looper.getMainLooper());

    /**
     * The Executor service.
     */
    protected ExecutorService executorService;

    /**
     * The Context.
     */
    protected Context context;

    private AbstractVoxeetService() {

        this.eventBus = EventBus.getDefault();

        this.lastUpdate = null;

        this.executorService = Executors.newSingleThreadExecutor();
    }

    /**
     * Instantiates a new Voxeet service.
     *
     * @param context the context
     * @param client  the client
     */
    protected AbstractVoxeetService(@NonNull Context context, @Nullable OkHttpClient client, @Nullable Retrofit retrofit, @Nullable Class<T> service_klass) {
        this(context, client, retrofit, service_klass != null ? retrofit.create(service_klass) : (T) null);
    }

    protected AbstractVoxeetService(@NonNull Context context, @Nullable OkHttpClient client, @Nullable Retrofit retrofit, @Nullable T service_klass_instantiated) {
        this();

        this.retrofit = retrofit;

        this.context = context;

        this.client = client;

        this.service = service_klass_instantiated;

        //initService(service_klass);
    }

    /**
     * Update time stamp.
     */
    protected void updateTimeStamp() {
        this.lastUpdate = new Date();
    }

    /**
     * Retuns if the API has been reached at least once successfully
     *
     * @return true if it's been loaded, false otherwise
     */
    protected boolean isLoaded() {
        return isLoaded;
    }

    /**
     * Sets loaded.
     *
     * @param loaded the loaded
     */
    protected void setLoaded(boolean loaded) {
        isLoaded = loaded;
    }

    /**
     * Should update boolean.
     *
     * @return the boolean
     */
    protected boolean shouldUpdate() {
        if (lastUpdate == null)
            return true;
        else
            return (new Date().getTime() - lastUpdate.getTime() > UPDATE_TIME);
    }

    final protected T getService() {
        return service;
    }

    /**
     * Retrieves the error string from the request.
     *
     * @param e the exception
     * @return the error content
     */
    protected String handleError(Throwable e) {
        if (e instanceof HttpException) {
            try {
                ResponseBody body = ((HttpException) e).response().errorBody();
                return body.string();
            } catch (IOException exception) {
                Log.e(TAG, "error: " + exception.getMessage());
            }
        }
        return e.getMessage();
    }

    /**
     * Gets error body.
     *
     * @param message the message
     * @return the error body
     */
    protected static VoxeetJsonError getErrorBody(String message) {
        ObjectMapper om = new ObjectMapper();
        VoxeetJsonError command = null;
        try {
            command = om.readValue(message, VoxeetJsonError.class);
        } catch (IOException e) {
            Log.e(TAG, "failed to get error body", e);
        }

        return command;
    }

    /**
     * Reset service.
     */
    public void resetService() {
        lastUpdate = null;

        isLoaded = false;
    }

    protected void registerEventBus() {
        if (!eventBus.isRegistered(this))
            eventBus.register(this);
    }

    protected void unRegisterEventBus() {
        if (eventBus.isRegistered(this))
            eventBus.unregister(this);
    }

    protected void postOnMainThread(@NonNull Runnable runnable) {
        handler.post(runnable);
    }
}
