/*
 * Decompiled with CFR 0.152.
 */
package com.viae.maven.sonar.services;

import com.viae.maven.sonar.exceptions.SonarQualityException;
import com.viae.maven.sonar.services.SonarQualityGateService;
import com.viae.maven.sonar.utils.JsonUtil;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.json.simple.JSONArray;
import org.sonar.wsclient.SonarClient;
import org.sonar.wsclient.base.HttpException;

public class SonarQualityGateServiceImpl
implements SonarQualityGateService {
    public static final String QUALITY_GATE_QUERY_URL = "/api/resources/index?metrics=quality_gate_details&format=json&resource=%s";
    private static final int SLEEP_INTERVAL = 100;
    private static final String LEVEL_ERROR = "ERROR";
    private static final String FIELD_LEVEL = "level";
    private static final String FIELD_CONDITIONS = "conditions";
    private final Log logger;

    public SonarQualityGateServiceImpl(Log logger) {
        this.logger = logger;
    }

    @Override
    public void validateQualityGate(SonarClient client, String projectKey, String qualityGateName, LocalDateTime executionStart, int secondsToWait) throws SonarQualityException {
        this.waitForNewPublishingOfSonarResults(client, projectKey, qualityGateName, executionStart, secondsToWait);
        this.handleQualityGateState(client, projectKey);
    }

    public void handleQualityGateState(SonarClient client, String projectKey) throws SonarQualityException {
        Validate.notNull((Object)client, (String)"The given sonar client can't be null", (Object[])new Object[0]);
        Validate.notBlank((CharSequence)projectKey, (String)"The given project key can't be blank", (Object[])new Object[0]);
        try {
            String url = String.format(QUALITY_GATE_QUERY_URL, projectKey);
            this.logger.info((CharSequence)String.format("Retrieve quality gate details from: %s", url));
            String qualityGateDetailsData = client.get(url, new Object[0]);
            this.logger.info((CharSequence)String.format("Resulting quality gate state: %s", qualityGateDetailsData));
            if (StringUtils.isNotBlank((CharSequence)qualityGateDetailsData)) {
                String level;
                String msr = JsonUtil.getOnMainLevel(qualityGateDetailsData, "msr");
                String data = StringUtils.isNotBlank((CharSequence)msr) ? JsonUtil.getOnMainLevel(msr, "data") : "";
                String string = level = StringUtils.isNotBlank((CharSequence)data) ? JsonUtil.getOnMainLevel(data, FIELD_LEVEL) : "";
                if (LEVEL_ERROR.equals(level.toUpperCase())) {
                    StringJoiner joiner = new StringJoiner("\n");
                    joiner.add("");
                    joiner.add("############################");
                    joiner.add("############################");
                    joiner.add("### quality gate not met ###");
                    joiner.add("############################");
                    joiner.add("############################");
                    JSONArray conditionsResponse = JsonUtil.parseArray(JsonUtil.getOnMainLevel(data, FIELD_CONDITIONS));
                    if (conditionsResponse != null) {
                        joiner.add("Conditions:");
                        conditionsResponse.forEach(condition -> joiner.add(condition.toString()));
                    }
                    throw new SonarQualityException(joiner.toString());
                }
            }
        }
        catch (Exception e) {
            throw new SonarQualityException(String.format("Error while getting quality gate data:\n%s", ExceptionUtils.getStackTrace((Throwable)e)), e);
        }
    }

    protected final void waitForNewPublishingOfSonarResults(SonarClient client, String projectKey, String qualityGateName, LocalDateTime executionStart, int secondsToWait) throws SonarQualityException {
        Validate.notNull((Object)client, (String)"The given sonar client can't be null", (Object[])new Object[0]);
        Validate.notBlank((CharSequence)projectKey, (String)"The given project key can't be blank", (Object[])new Object[0]);
        LocalDateTime start = LocalDateTime.now();
        if (executionStart != null) {
            LocalDateTime lastRunTimeStamp = this.getLastRunTimeStamp(client, projectKey, qualityGateName);
            while (!lastRunTimeStamp.isAfter(executionStart) || !this.qualityGateDetailsExists(client, projectKey)) {
                long duration = Duration.between(start, LocalDateTime.now()).getSeconds();
                if (duration > (long)secondsToWait) {
                    throw new SonarQualityException(String.format("We waited for %s seconds, but no update on last run (i.e. date field) occurred.", duration));
                }
                this.sleep();
                lastRunTimeStamp = this.getLastRunTimeStamp(client, projectKey, qualityGateName);
            }
        }
    }

    public boolean qualityGateDetailsExists(SonarClient client, String projectKey) {
        boolean exists = false;
        String url = String.format(QUALITY_GATE_QUERY_URL, projectKey);
        try {
            client.get(url, new Object[0]);
        }
        catch (HttpException e) {
            this.logger.info((CharSequence)String.format("url %s does not exist", url));
        }
        return exists;
    }

    private void sleep() {
        try {
            Thread.sleep(100L);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void linkQualityGateToProject(SonarClient client, String projectKey, String qualityGateName) throws SonarQualityException {
        Validate.notNull((Object)client, (String)"The given Sonar client can't be null", (Object[])new Object[0]);
        Validate.notBlank((CharSequence)projectKey, (String)"The given project key can't be null", (Object[])new Object[0]);
        Validate.notBlank((CharSequence)qualityGateName, (String)"The given quality gate name can't be null", (Object[])new Object[0]);
        try {
            String resourceDataJson = client.get(String.format("/api/resources?format=json&resource=%s", projectKey), new Object[0]);
            String errorCode = JsonUtil.getOnMainLevel(resourceDataJson, "err_code");
            if ("404".equals(errorCode)) {
                this.verifySonarProjectExistsAndIsLinkedToQualityGate(client, projectKey, qualityGateName);
                resourceDataJson = client.get(String.format("/api/resources?format=json&resource=%s", projectKey), new Object[0]);
            }
            String projectId = JsonUtil.getIdOnMainLevel(resourceDataJson);
            this.doLinkQualityGateToProject(client, Integer.parseInt(projectId), qualityGateName);
        }
        catch (HttpException e) {
            this.logger.error((Throwable)e);
            if (e.status() == 404) {
                this.verifySonarProjectExistsAndIsLinkedToQualityGate(client, projectKey, qualityGateName);
            }
            throw new SonarQualityException("Sonar HTTP exception", (Exception)((Object)e));
        }
    }

    private void doLinkQualityGateToProject(SonarClient client, int projectId, String qualityGateName) throws SonarQualityException {
        String qualityGateJson = client.get(String.format("/api/qualitygates/show?name=%s", qualityGateName), new Object[0]);
        String qualityGateId = JsonUtil.getIdOnMainLevel(qualityGateJson);
        if (StringUtils.isNotBlank((CharSequence)qualityGateId)) {
            ConcurrentHashMap<String, Object> map = new ConcurrentHashMap<String, Object>();
            map.put("gateId", qualityGateId);
            map.put("projectId", projectId);
            client.post("/api/qualitygates/select", map);
        }
    }

    @Override
    public LocalDateTime getLastRunTimeStamp(SonarClient client, String projectKey, String qualityGateName) throws SonarQualityException {
        Validate.notNull((Object)client, (String)"The given Sonar client can't be null", (Object[])new Object[0]);
        Validate.notBlank((CharSequence)projectKey, (String)"The given project key can't be null", (Object[])new Object[0]);
        LocalDateTime result = null;
        try {
            String dateStringValue;
            String resourceDataJson = client.get(String.format("/api/resources?format=json&resource=%s", projectKey), new Object[0]);
            String errorCode = JsonUtil.getOnMainLevel(resourceDataJson, "err_code");
            if ("404".equals(errorCode)) {
                this.verifySonarProjectExistsAndIsLinkedToQualityGate(client, projectKey, qualityGateName);
                resourceDataJson = client.get(String.format("/api/resources?format=json&resource=%s", projectKey), new Object[0]);
            }
            if (StringUtils.isNotBlank((CharSequence)(dateStringValue = JsonUtil.getOnMainLevel(resourceDataJson, "date")))) {
                result = LocalDateTime.parse(dateStringValue, DATE_TIME_FORMATTER);
            }
        }
        catch (HttpException e) {
            if (e.status() == 404) {
                result = LocalDateTime.now();
            }
            throw new SonarQualityException("Sonar HTTP exception", (Exception)((Object)e));
        }
        return result;
    }

    @Override
    public String composeSonarProjectKey(MavenProject project, String projectKey, String branchName) {
        String resultingKey = String.format("%s:%s", project.getGroupId(), project.getArtifactId());
        if (projectKey != null) {
            resultingKey = projectKey;
        }
        if (StringUtils.isNotBlank((CharSequence)branchName)) {
            resultingKey = resultingKey + ":" + branchName;
        }
        return resultingKey;
    }

    private int verifySonarProjectExistsAndIsLinkedToQualityGate(SonarClient client, String projectKey, String qualityGateName) throws SonarQualityException {
        int projectId;
        String lookupProjectData = client.get(String.format("/api/projects?key=", projectKey), new Object[0]);
        this.logger.info((CharSequence)String.format("Lookup project data result: %s", lookupProjectData));
        if (StringUtils.isBlank((CharSequence)JsonUtil.getOnMainLevel(lookupProjectData, "err_code"))) {
            projectId = this.getProjectId(lookupProjectData);
            this.logger.info((CharSequence)("Retrieved project id: " + projectId));
        } else {
            projectId = this.createProject(client, projectKey);
            this.logger.info((CharSequence)("Created project with id: " + projectId));
        }
        this.logger.info((CharSequence)"wait for project to be published");
        try {
            Thread.sleep(10000L);
        }
        catch (InterruptedException e) {
            this.logger.error((Throwable)e);
        }
        this.logger.info((CharSequence)String.format("creating project %s resulted in project id %s", projectKey, projectId));
        this.logger.info((CharSequence)String.format("link project %s to quality gate %s", projectKey, qualityGateName));
        this.doLinkQualityGateToProject(client, projectId, qualityGateName);
        return projectId;
    }

    int getProjectId(String lookupProjectData) throws SonarQualityException {
        int projectId;
        try {
            if (Optional.ofNullable(lookupProjectData).orElse("").trim().startsWith("[")) {
                JSONArray jsonArray = JsonUtil.parseArray(lookupProjectData);
                projectId = Integer.parseInt(JsonUtil.getIdOnMainLevel(jsonArray.get(0).toString()));
            } else {
                projectId = Integer.parseInt(JsonUtil.getIdOnMainLevel(lookupProjectData));
            }
        }
        catch (Exception e) {
            throw new SonarQualityException(String.format("Could not get project id from json %s", lookupProjectData), e);
        }
        return projectId;
    }

    private int createProject(SonarClient client, String projectKey) throws SonarQualityException {
        this.logger.info((CharSequence)String.format("create project '%s'", projectKey));
        ConcurrentHashMap<String, String> map = new ConcurrentHashMap<String, String>();
        map.put("key", projectKey);
        map.put("name", projectKey);
        String postResult = client.post("/api/projects/create", map);
        this.logger.info((CharSequence)String.format("Result of creation call: %s", postResult));
        int projectId = Integer.parseInt(JsonUtil.getIdOnMainLevel(postResult));
        return projectId;
    }
}

