/*
 * Decompiled with CFR 0.152.
 */
package org.webrtc;

import android.annotation.TargetApi;
import android.media.MediaCodec;
import android.media.MediaFormat;
import android.os.Bundle;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Deque;
import java.util.concurrent.LinkedBlockingDeque;
import org.webrtc.BitrateAdjuster;
import org.webrtc.EncodedImage;
import org.webrtc.Logging;
import org.webrtc.ThreadUtils;
import org.webrtc.VideoCodecStatus;
import org.webrtc.VideoCodecType;
import org.webrtc.VideoEncoder;
import org.webrtc.VideoFrame;

@TargetApi(value=19)
class HardwareVideoEncoder
implements VideoEncoder {
    private static final String TAG = "HardwareVideoEncoder";
    private static final int VIDEO_ControlRateConstant = 2;
    private static final String KEY_BITRATE_MODE = "bitrate-mode";
    private static final int MAX_VIDEO_FRAMERATE = 30;
    private static final int MAX_ENCODER_Q_SIZE = 2;
    private static final int MEDIA_CODEC_RELEASE_TIMEOUT_MS = 5000;
    private static final int DEQUEUE_OUTPUT_BUFFER_TIMEOUT_US = 100000;
    private final String codecName;
    private final VideoCodecType codecType;
    private final int colorFormat;
    private final ColorFormat inputColorFormat;
    private final int keyFrameIntervalSec;
    private final long forcedKeyFrameMs;
    private long lastKeyFrameMs;
    private final BitrateAdjuster bitrateAdjuster;
    private int adjustedBitrate;
    private final Deque<EncodedImage.Builder> outputBuilders;
    private Thread outputThread;
    private volatile boolean running = false;
    private volatile Exception shutdownException = null;
    private MediaCodec codec;
    private VideoEncoder.Callback callback;
    private int width;
    private int height;
    private ByteBuffer configBuffer = null;

    public HardwareVideoEncoder(String codecName, VideoCodecType codecType, int colorFormat, int keyFrameIntervalSec, int forceKeyFrameIntervalMs, BitrateAdjuster bitrateAdjuster) {
        this.codecName = codecName;
        this.codecType = codecType;
        this.colorFormat = colorFormat;
        this.inputColorFormat = ColorFormat.valueOf(colorFormat);
        this.keyFrameIntervalSec = keyFrameIntervalSec;
        this.forcedKeyFrameMs = forceKeyFrameIntervalMs;
        this.bitrateAdjuster = bitrateAdjuster;
        this.outputBuilders = new LinkedBlockingDeque<EncodedImage.Builder>();
    }

    @Override
    public VideoCodecStatus initEncode(VideoEncoder.Settings settings, VideoEncoder.Callback callback) {
        return this.initEncodeInternal(settings.width, settings.height, settings.startBitrate, settings.maxFramerate, callback);
    }

    private VideoCodecStatus initEncodeInternal(int width, int height, int bitrateKbps, int fps, VideoEncoder.Callback callback) {
        Logging.d((String)TAG, (String)("initEncode: " + width + " x " + height + ". @ " + bitrateKbps + "kbps. Fps: " + fps));
        this.width = width;
        this.height = height;
        if (bitrateKbps != 0 && fps != 0) {
            this.bitrateAdjuster.setTargets(bitrateKbps * 1000, fps);
        }
        this.adjustedBitrate = this.bitrateAdjuster.getAdjustedBitrateBps();
        this.callback = callback;
        this.lastKeyFrameMs = -1L;
        try {
            this.codec = MediaCodec.createByCodecName((String)this.codecName);
        }
        catch (IOException | IllegalArgumentException e) {
            Logging.e((String)TAG, (String)("Cannot create media encoder " + this.codecName));
            return VideoCodecStatus.ERROR;
        }
        try {
            MediaFormat format = MediaFormat.createVideoFormat((String)this.codecType.mimeType(), (int)width, (int)height);
            format.setInteger("bitrate", this.adjustedBitrate);
            format.setInteger(KEY_BITRATE_MODE, 2);
            format.setInteger("color-format", this.colorFormat);
            format.setInteger("frame-rate", this.bitrateAdjuster.getAdjustedFramerate());
            format.setInteger("i-frame-interval", this.keyFrameIntervalSec);
            Logging.d((String)TAG, (String)("Format: " + format));
            this.codec.configure(format, null, null, 1);
            this.codec.start();
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"initEncode failed", (Throwable)e);
            this.release();
            return VideoCodecStatus.ERROR;
        }
        this.running = true;
        this.outputThread = this.createOutputThread();
        this.outputThread.start();
        return VideoCodecStatus.OK;
    }

    @Override
    public VideoCodecStatus release() {
        try {
            this.running = false;
            if (!ThreadUtils.joinUninterruptibly((Thread)this.outputThread, (long)5000L)) {
                Logging.e((String)TAG, (String)"Media encoder release timeout");
                VideoCodecStatus videoCodecStatus = VideoCodecStatus.TIMEOUT;
                return videoCodecStatus;
            }
            if (this.shutdownException != null) {
                Logging.e((String)TAG, (String)"Media encoder release exception", (Throwable)this.shutdownException);
                VideoCodecStatus videoCodecStatus = VideoCodecStatus.ERROR;
                return videoCodecStatus;
            }
        }
        finally {
            this.codec = null;
            this.outputThread = null;
            this.outputBuilders.clear();
        }
        return VideoCodecStatus.OK;
    }

    @Override
    public VideoCodecStatus encode(VideoFrame videoFrame, VideoEncoder.EncodeInfo encodeInfo) {
        ByteBuffer buffer;
        int index;
        VideoCodecStatus status;
        if (this.codec == null) {
            return VideoCodecStatus.UNINITIALIZED;
        }
        int frameWidth = videoFrame.getWidth();
        int frameHeight = videoFrame.getHeight();
        if ((frameWidth != this.width || frameHeight != this.height) && (status = this.resetCodec(frameWidth, frameHeight)) != VideoCodecStatus.OK) {
            return status;
        }
        try {
            index = this.codec.dequeueInputBuffer(0L);
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"dequeueInputBuffer failed", (Throwable)e);
            return VideoCodecStatus.FALLBACK_SOFTWARE;
        }
        if (index == -1) {
            Logging.e((String)TAG, (String)"Dropped frame, no input buffers available");
            return VideoCodecStatus.OK;
        }
        if (this.outputBuilders.size() > 2) {
            Logging.e((String)TAG, (String)"Dropped frame, encoder queue full");
            return VideoCodecStatus.OK;
        }
        try {
            buffer = this.codec.getInputBuffers()[index];
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"getInputBuffers failed", (Throwable)e);
            return VideoCodecStatus.FALLBACK_SOFTWARE;
        }
        VideoFrame.I420Buffer i420 = videoFrame.getBuffer().toI420();
        this.inputColorFormat.fillBufferFromI420(buffer, i420);
        boolean requestedKeyFrame = false;
        for (EncodedImage.FrameType frameType : encodeInfo.frameTypes) {
            if (frameType != EncodedImage.FrameType.VideoFrameKey) continue;
            requestedKeyFrame = true;
        }
        long presentationTimestampUs = (videoFrame.getTimestampNs() + 500L) / 1000L;
        long presentationTimestampMs = (presentationTimestampUs + 500L) / 1000L;
        if (requestedKeyFrame || this.shouldForceKeyFrame(presentationTimestampMs)) {
            this.requestKeyFrame(presentationTimestampMs);
        }
        int bufferSize = videoFrame.getBuffer().getHeight() * videoFrame.getBuffer().getWidth() * 3 / 2;
        EncodedImage.Builder builder = EncodedImage.builder().setCaptureTimeMs(presentationTimestampMs).setCompleteFrame(true).setEncodedWidth(videoFrame.getWidth()).setEncodedHeight(videoFrame.getHeight()).setRotation(videoFrame.getRotation());
        this.outputBuilders.offer(builder);
        try {
            this.codec.queueInputBuffer(index, 0, bufferSize, presentationTimestampUs, 0);
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"queueInputBuffer failed", (Throwable)e);
            this.outputBuilders.pollLast();
            return VideoCodecStatus.FALLBACK_SOFTWARE;
        }
        return VideoCodecStatus.OK;
    }

    @Override
    public VideoCodecStatus setChannelParameters(short packetLoss, long roundTripTimeMs) {
        return VideoCodecStatus.OK;
    }

    @Override
    public VideoCodecStatus setRateAllocation(VideoEncoder.BitrateAllocation bitrateAllocation, int framerate) {
        if (framerate > 30) {
            framerate = 30;
        }
        this.bitrateAdjuster.setTargets(bitrateAllocation.getSum(), framerate);
        return this.updateBitrate();
    }

    @Override
    public VideoEncoder.ScalingSettings getScalingSettings() {
        return null;
    }

    @Override
    public String getImplementationName() {
        return "HardwareVideoEncoder: " + this.codecName;
    }

    private VideoCodecStatus resetCodec(int newWidth, int newHeight) {
        VideoCodecStatus status = this.release();
        if (status != VideoCodecStatus.OK) {
            return status;
        }
        return this.initEncodeInternal(newWidth, newHeight, 0, 0, this.callback);
    }

    private boolean shouldForceKeyFrame(long presentationTimestampMs) {
        return this.forcedKeyFrameMs > 0L && presentationTimestampMs > this.lastKeyFrameMs + this.forcedKeyFrameMs;
    }

    private void requestKeyFrame(long presentationTimestampMs) {
        try {
            Bundle b = new Bundle();
            b.putInt("request-sync", 0);
            this.codec.setParameters(b);
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"requestKeyFrame failed", (Throwable)e);
            return;
        }
        this.lastKeyFrameMs = presentationTimestampMs;
    }

    private Thread createOutputThread() {
        return new Thread(){

            @Override
            public void run() {
                while (HardwareVideoEncoder.this.running) {
                    HardwareVideoEncoder.this.deliverEncodedImage();
                }
                HardwareVideoEncoder.this.releaseCodecOnOutputThread();
            }
        };
    }

    private void deliverEncodedImage() {
        try {
            MediaCodec.BufferInfo info = new MediaCodec.BufferInfo();
            int index = this.codec.dequeueOutputBuffer(info, 100000L);
            if (index < 0) {
                return;
            }
            ByteBuffer codecOutputBuffer = this.codec.getOutputBuffers()[index];
            codecOutputBuffer.position(info.offset);
            codecOutputBuffer.limit(info.offset + info.size);
            if ((info.flags & 2) != 0) {
                Logging.d((String)TAG, (String)("Config frame generated. Offset: " + info.offset + ". Size: " + info.size));
                this.configBuffer = ByteBuffer.allocateDirect(info.size);
                this.configBuffer.put(codecOutputBuffer);
            } else {
                ByteBuffer frameBuffer;
                boolean isKeyFrame;
                this.bitrateAdjuster.reportEncodedFrame(info.size);
                if (this.adjustedBitrate != this.bitrateAdjuster.getAdjustedBitrateBps()) {
                    this.updateBitrate();
                }
                boolean bl = isKeyFrame = (info.flags & 1) != 0;
                if (isKeyFrame && this.codecType == VideoCodecType.H264) {
                    Logging.d((String)TAG, (String)("Prepending config frame of size " + this.configBuffer.capacity() + " to output buffer with offset " + info.offset + ", size " + info.size));
                    frameBuffer = ByteBuffer.allocateDirect(info.size + this.configBuffer.capacity());
                    this.configBuffer.rewind();
                    frameBuffer.put(this.configBuffer);
                } else {
                    frameBuffer = ByteBuffer.allocateDirect(info.size);
                }
                frameBuffer.put(codecOutputBuffer);
                frameBuffer.rewind();
                EncodedImage.FrameType frameType = EncodedImage.FrameType.VideoFrameDelta;
                if (isKeyFrame) {
                    Logging.d((String)TAG, (String)"Sync frame generated");
                    frameType = EncodedImage.FrameType.VideoFrameKey;
                }
                EncodedImage.Builder builder = this.outputBuilders.poll();
                builder.setBuffer(frameBuffer).setFrameType(frameType);
                this.callback.onEncodedFrame(builder.createEncodedImage(), new VideoEncoder.CodecSpecificInfo());
            }
            this.codec.releaseOutputBuffer(index, false);
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"deliverOutput failed", (Throwable)e);
        }
    }

    private void releaseCodecOnOutputThread() {
        Logging.d((String)TAG, (String)"Releasing MediaCodec on output thread");
        try {
            this.codec.stop();
        }
        catch (Exception e) {
            Logging.e((String)TAG, (String)"Media encoder stop failed", (Throwable)e);
        }
        try {
            this.codec.release();
        }
        catch (Exception e) {
            Logging.e((String)TAG, (String)"Media encoder release failed", (Throwable)e);
            this.shutdownException = e;
        }
        Logging.d((String)TAG, (String)"Release on output thread done");
    }

    private VideoCodecStatus updateBitrate() {
        this.adjustedBitrate = this.bitrateAdjuster.getAdjustedBitrateBps();
        try {
            Bundle params = new Bundle();
            params.putInt("video-bitrate", this.adjustedBitrate);
            this.codec.setParameters(params);
            return VideoCodecStatus.OK;
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"updateBitrate failed", (Throwable)e);
            return VideoCodecStatus.ERROR;
        }
    }

    private static enum ColorFormat {
        I420{

            @Override
            void fillBufferFromI420(ByteBuffer buffer, VideoFrame.I420Buffer i420) {
                buffer.put(i420.getDataY());
                buffer.put(i420.getDataU());
                buffer.put(i420.getDataV());
            }
        }
        ,
        NV12{

            @Override
            void fillBufferFromI420(ByteBuffer buffer, VideoFrame.I420Buffer i420) {
                buffer.put(i420.getDataY());
                ByteBuffer u = i420.getDataU();
                ByteBuffer v = i420.getDataV();
                boolean i = false;
                while (u.hasRemaining() && v.hasRemaining()) {
                    buffer.put(u.get());
                    buffer.put(v.get());
                }
            }
        };


        abstract void fillBufferFromI420(ByteBuffer var1, VideoFrame.I420Buffer var2);

        static ColorFormat valueOf(int colorFormat) {
            switch (colorFormat) {
                case 19: {
                    return I420;
                }
                case 21: 
                case 2141391872: 
                case 2141391876: {
                    return NV12;
                }
            }
            throw new IllegalArgumentException("Unsupported colorFormat: " + colorFormat);
        }
    }
}

