/*
 * Decompiled with CFR 0.152.
 */
package org.webrtc;

import android.annotation.TargetApi;
import android.graphics.Matrix;
import android.media.MediaCodec;
import android.media.MediaFormat;
import android.os.SystemClock;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Deque;
import java.util.concurrent.LinkedBlockingDeque;
import org.webrtc.EncodedImage;
import org.webrtc.I420BufferImpl;
import org.webrtc.Logging;
import org.webrtc.MediaCodecUtils;
import org.webrtc.ThreadUtils;
import org.webrtc.VideoCodecStatus;
import org.webrtc.VideoCodecType;
import org.webrtc.VideoDecoder;
import org.webrtc.VideoFrame;

@TargetApi(value=16)
class HardwareVideoDecoder
implements VideoDecoder {
    private static final String TAG = "HardwareVideoDecoder";
    private static final String MEDIA_FORMAT_KEY_STRIDE = "stride";
    private static final String MEDIA_FORMAT_KEY_SLICE_HEIGHT = "slice-height";
    private static final String MEDIA_FORMAT_KEY_CROP_LEFT = "crop-left";
    private static final String MEDIA_FORMAT_KEY_CROP_RIGHT = "crop-right";
    private static final String MEDIA_FORMAT_KEY_CROP_TOP = "crop-top";
    private static final String MEDIA_FORMAT_KEY_CROP_BOTTOM = "crop-bottom";
    private static final int MEDIA_CODEC_RELEASE_TIMEOUT_MS = 5000;
    private static final int DEQUEUE_INPUT_TIMEOUT_US = 500000;
    private static final int DEQUEUE_OUTPUT_BUFFER_TIMEOUT_US = 100000;
    private final String codecName;
    private final VideoCodecType codecType;
    private final Deque<FrameInfo> frameInfos;
    private int colorFormat;
    private Thread outputThread;
    private ThreadUtils.ThreadChecker outputThreadChecker;
    private ThreadUtils.ThreadChecker decoderThreadChecker;
    private volatile boolean running = false;
    private volatile Exception shutdownException = null;
    private final Object activeOutputBuffersLock = new Object();
    private int activeOutputBuffers = 0;
    private final Object dimensionLock = new Object();
    private int width;
    private int height;
    private int stride;
    private int sliceHeight;
    private boolean hasDecodedFirstFrame;
    private boolean keyFrameRequired;
    private VideoDecoder.Callback callback;
    private MediaCodec codec = null;

    HardwareVideoDecoder(String codecName, VideoCodecType codecType, int colorFormat) {
        if (!this.isSupportedColorFormat(colorFormat)) {
            throw new IllegalArgumentException("Unsupported color format: " + colorFormat);
        }
        this.codecName = codecName;
        this.codecType = codecType;
        this.colorFormat = colorFormat;
        this.frameInfos = new LinkedBlockingDeque<FrameInfo>();
    }

    @Override
    public VideoCodecStatus initDecode(VideoDecoder.Settings settings, VideoDecoder.Callback callback) {
        this.decoderThreadChecker = new ThreadUtils.ThreadChecker();
        return this.initDecodeInternal(settings.width, settings.height, callback);
    }

    private VideoCodecStatus initDecodeInternal(int width, int height, VideoDecoder.Callback callback) {
        this.decoderThreadChecker.checkIsOnValidThread();
        if (this.outputThread != null) {
            Logging.e((String)TAG, (String)"initDecodeInternal called while the codec is already running");
            return VideoCodecStatus.ERROR;
        }
        this.callback = callback;
        this.width = width;
        this.height = height;
        this.stride = width;
        this.sliceHeight = height;
        this.hasDecodedFirstFrame = false;
        this.keyFrameRequired = true;
        try {
            this.codec = MediaCodec.createByCodecName((String)this.codecName);
        }
        catch (IOException | IllegalArgumentException e) {
            Logging.e((String)TAG, (String)("Cannot create media decoder " + this.codecName));
            return VideoCodecStatus.ERROR;
        }
        try {
            MediaFormat format = MediaFormat.createVideoFormat((String)this.codecType.mimeType(), (int)width, (int)height);
            format.setInteger("color-format", this.colorFormat);
            this.codec.configure(format, null, null, 0);
            this.codec.start();
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"initDecode failed", (Throwable)e);
            this.release();
            return VideoCodecStatus.ERROR;
        }
        this.running = true;
        this.outputThread = this.createOutputThread();
        this.outputThread.start();
        return VideoCodecStatus.OK;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public VideoCodecStatus decode(EncodedImage frame, VideoDecoder.DecodeInfo info) {
        ByteBuffer buffer;
        int index;
        VideoCodecStatus status;
        int height;
        int width;
        this.decoderThreadChecker.checkIsOnValidThread();
        if (this.codec == null || this.callback == null) {
            return VideoCodecStatus.UNINITIALIZED;
        }
        if (frame.buffer == null) {
            Logging.e((String)TAG, (String)"decode() - no input data");
            return VideoCodecStatus.ERR_PARAMETER;
        }
        int size = frame.buffer.remaining();
        if (size == 0) {
            Logging.e((String)TAG, (String)"decode() - input buffer empty");
            return VideoCodecStatus.ERR_PARAMETER;
        }
        Object object = this.dimensionLock;
        synchronized (object) {
            width = this.width;
            height = this.height;
        }
        if (frame.encodedWidth * frame.encodedHeight > 0 && (frame.encodedWidth != width || frame.encodedHeight != height) && (status = this.reinitDecode(frame.encodedWidth, frame.encodedHeight)) != VideoCodecStatus.OK) {
            return status;
        }
        if (this.keyFrameRequired) {
            if (frame.frameType != EncodedImage.FrameType.VideoFrameKey) {
                Logging.e((String)TAG, (String)"decode() - key frame required first");
                return VideoCodecStatus.ERROR;
            }
            if (!frame.completeFrame) {
                Logging.e((String)TAG, (String)"decode() - complete frame required first");
                return VideoCodecStatus.ERROR;
            }
        }
        try {
            index = this.codec.dequeueInputBuffer(500000L);
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"dequeueInputBuffer failed", (Throwable)e);
            return VideoCodecStatus.ERROR;
        }
        if (index < 0) {
            Logging.e((String)TAG, (String)"decode() - no HW buffers available; decoder falling behind");
            return VideoCodecStatus.ERROR;
        }
        try {
            buffer = this.codec.getInputBuffers()[index];
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"getInputBuffers failed", (Throwable)e);
            return VideoCodecStatus.ERROR;
        }
        if (buffer.capacity() < size) {
            Logging.e((String)TAG, (String)"decode() - HW buffer too small");
            return VideoCodecStatus.ERROR;
        }
        buffer.put(frame.buffer);
        this.frameInfos.offer(new FrameInfo(SystemClock.elapsedRealtime(), frame.rotation));
        try {
            this.codec.queueInputBuffer(index, 0, size, frame.captureTimeMs * 1000L, 0);
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"queueInputBuffer failed", (Throwable)e);
            this.frameInfos.pollLast();
            return VideoCodecStatus.ERROR;
        }
        if (this.keyFrameRequired) {
            this.keyFrameRequired = false;
        }
        return VideoCodecStatus.OK;
    }

    @Override
    public boolean getPrefersLateDecoding() {
        return true;
    }

    @Override
    public String getImplementationName() {
        return "HardwareVideoDecoder: " + this.codecName;
    }

    @Override
    public VideoCodecStatus release() {
        try {
            this.running = false;
            if (!ThreadUtils.joinUninterruptibly((Thread)this.outputThread, (long)5000L)) {
                Logging.e((String)TAG, (String)"Media encoder release timeout", (Throwable)new RuntimeException());
                VideoCodecStatus videoCodecStatus = VideoCodecStatus.TIMEOUT;
                return videoCodecStatus;
            }
            if (this.shutdownException != null) {
                Logging.e((String)TAG, (String)"Media encoder release error", (Throwable)new RuntimeException(this.shutdownException));
                this.shutdownException = null;
                VideoCodecStatus videoCodecStatus = VideoCodecStatus.ERROR;
                return videoCodecStatus;
            }
        }
        finally {
            this.codec = null;
            this.callback = null;
            this.outputThread = null;
            this.frameInfos.clear();
        }
        return VideoCodecStatus.OK;
    }

    private VideoCodecStatus reinitDecode(int newWidth, int newHeight) {
        this.decoderThreadChecker.checkIsOnValidThread();
        VideoCodecStatus status = this.release();
        if (status != VideoCodecStatus.OK) {
            return status;
        }
        return this.initDecodeInternal(newWidth, newHeight, this.callback);
    }

    private Thread createOutputThread() {
        return new Thread("HardwareVideoDecoder.outputThread"){

            @Override
            public void run() {
                HardwareVideoDecoder.this.outputThreadChecker = new ThreadUtils.ThreadChecker();
                while (HardwareVideoDecoder.this.running) {
                    HardwareVideoDecoder.this.deliverDecodedFrame();
                }
                HardwareVideoDecoder.this.releaseCodecOnOutputThread();
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void deliverDecodedFrame() {
        this.outputThreadChecker.checkIsOnValidThread();
        try {
            VideoFrame.I420Buffer frameBuffer;
            int sliceHeight;
            int stride;
            int height;
            int width;
            MediaCodec.BufferInfo info = new MediaCodec.BufferInfo();
            int result = this.codec.dequeueOutputBuffer(info, 100000L);
            if (result == -2) {
                this.reformat(this.codec.getOutputFormat());
                return;
            }
            if (result < 0) {
                Logging.v((String)TAG, (String)("dequeueOutputBuffer returned " + result));
                return;
            }
            FrameInfo frameInfo = this.frameInfos.poll();
            Integer decodeTimeMs = null;
            int rotation = 0;
            if (frameInfo != null) {
                decodeTimeMs = (int)(SystemClock.elapsedRealtime() - frameInfo.decodeStartTimeMs);
                rotation = frameInfo.rotation;
            }
            this.hasDecodedFirstFrame = true;
            Object object = this.dimensionLock;
            synchronized (object) {
                width = this.width;
                height = this.height;
                stride = this.stride;
                sliceHeight = this.sliceHeight;
            }
            if (info.size < width * height * 3 / 2) {
                Logging.e((String)TAG, (String)("Insufficient output buffer size: " + info.size));
                return;
            }
            if (info.size < stride * height * 3 / 2 && sliceHeight == height && stride > width) {
                stride = info.size * 2 / (height * 3);
            }
            ByteBuffer buffer = this.codec.getOutputBuffers()[result];
            buffer.position(info.offset);
            buffer.limit(info.size);
            if (this.colorFormat == 19) {
                if (sliceHeight % 2 == 0) {
                    frameBuffer = this.createBufferFromI420(buffer, result, info.offset, stride, sliceHeight, width, height);
                } else {
                    frameBuffer = new I420BufferImpl(width, height);
                    HardwareVideoDecoder.copyI420(buffer, info.offset, frameBuffer, stride, sliceHeight, width, height);
                    this.codec.releaseOutputBuffer(result, false);
                }
            } else {
                frameBuffer = new I420BufferImpl(width, height);
                HardwareVideoDecoder.nv12ToI420(buffer, info.offset, frameBuffer, stride, sliceHeight, width, height);
                this.codec.releaseOutputBuffer(result, false);
            }
            long presentationTimeNs = info.presentationTimeUs * 1000L;
            VideoFrame frame = new VideoFrame(frameBuffer, rotation, presentationTimeNs, new Matrix());
            this.callback.onDecodedFrame(frame, decodeTimeMs, null);
            frame.release();
        }
        catch (IllegalStateException e) {
            Logging.e((String)TAG, (String)"deliverDecodedFrame failed", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void reformat(MediaFormat format) {
        int newHeight;
        int newWidth;
        this.outputThreadChecker.checkIsOnValidThread();
        Logging.d((String)TAG, (String)("Decoder format changed: " + format.toString()));
        if (format.containsKey(MEDIA_FORMAT_KEY_CROP_LEFT) && format.containsKey(MEDIA_FORMAT_KEY_CROP_RIGHT) && format.containsKey(MEDIA_FORMAT_KEY_CROP_BOTTOM) && format.containsKey(MEDIA_FORMAT_KEY_CROP_TOP)) {
            newWidth = 1 + format.getInteger(MEDIA_FORMAT_KEY_CROP_RIGHT) - format.getInteger(MEDIA_FORMAT_KEY_CROP_LEFT);
            newHeight = 1 + format.getInteger(MEDIA_FORMAT_KEY_CROP_BOTTOM) - format.getInteger(MEDIA_FORMAT_KEY_CROP_TOP);
        } else {
            newWidth = format.getInteger("width");
            newHeight = format.getInteger("height");
        }
        Object object = this.dimensionLock;
        synchronized (object) {
            if (this.hasDecodedFirstFrame && (this.width != newWidth || this.height != newHeight)) {
                this.stopOnOutputThread(new RuntimeException("Unexpected size change. Configured " + this.width + "*" + this.height + ". New " + newWidth + "*" + newHeight));
                return;
            }
            this.width = newWidth;
            this.height = newHeight;
        }
        if (format.containsKey("color-format")) {
            this.colorFormat = format.getInteger("color-format");
            Logging.d((String)TAG, (String)("Color: 0x" + Integer.toHexString(this.colorFormat)));
            if (!this.isSupportedColorFormat(this.colorFormat)) {
                this.stopOnOutputThread(new IllegalStateException("Unsupported color format: " + this.colorFormat));
                return;
            }
        }
        object = this.dimensionLock;
        synchronized (object) {
            if (format.containsKey(MEDIA_FORMAT_KEY_STRIDE)) {
                this.stride = format.getInteger(MEDIA_FORMAT_KEY_STRIDE);
            }
            if (format.containsKey(MEDIA_FORMAT_KEY_SLICE_HEIGHT)) {
                this.sliceHeight = format.getInteger(MEDIA_FORMAT_KEY_SLICE_HEIGHT);
            }
            Logging.d((String)TAG, (String)("Frame stride and slice height: " + this.stride + " x " + this.sliceHeight));
            this.stride = Math.max(this.width, this.stride);
            this.sliceHeight = Math.max(this.height, this.sliceHeight);
        }
    }

    private void releaseCodecOnOutputThread() {
        this.outputThreadChecker.checkIsOnValidThread();
        Logging.d((String)TAG, (String)"Releasing MediaCodec on output thread");
        this.waitOutputBuffersReleasedOnOutputThread();
        try {
            this.codec.stop();
        }
        catch (Exception e) {
            Logging.e((String)TAG, (String)"Media decoder stop failed", (Throwable)e);
        }
        try {
            this.codec.release();
        }
        catch (Exception e) {
            Logging.e((String)TAG, (String)"Media decoder release failed", (Throwable)e);
            this.shutdownException = e;
        }
        this.codec = null;
        this.callback = null;
        this.outputThread = null;
        this.frameInfos.clear();
        Logging.d((String)TAG, (String)"Release on output thread done");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void waitOutputBuffersReleasedOnOutputThread() {
        this.outputThreadChecker.checkIsOnValidThread();
        Object object = this.activeOutputBuffersLock;
        synchronized (object) {
            while (this.activeOutputBuffers > 0) {
                Logging.d((String)TAG, (String)"Waiting for all frames to be released.");
                try {
                    this.activeOutputBuffersLock.wait();
                }
                catch (InterruptedException e) {
                    Logging.e((String)TAG, (String)"Interrupted while waiting for output buffers to be released.", (Throwable)e);
                    return;
                }
            }
        }
    }

    private void stopOnOutputThread(Exception e) {
        this.outputThreadChecker.checkIsOnValidThread();
        this.running = false;
        this.shutdownException = e;
    }

    private boolean isSupportedColorFormat(int colorFormat) {
        for (int supported : MediaCodecUtils.DECODER_COLOR_FORMATS) {
            if (supported != colorFormat) continue;
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private VideoFrame.I420Buffer createBufferFromI420(final ByteBuffer buffer, final int outputBufferIndex, int offset, final int stride, int sliceHeight, final int width, final int height) {
        final int uvStride = stride / 2;
        int chromaWidth = (width + 1) / 2;
        final int chromaHeight = (height + 1) / 2;
        final int yPos = offset;
        final int uPos = yPos + stride * sliceHeight;
        final int vPos = uPos + uvStride * sliceHeight / 2;
        Object object = this.activeOutputBuffersLock;
        synchronized (object) {
            ++this.activeOutputBuffers;
        }
        return new VideoFrame.I420Buffer(){
            private int refCount = 1;

            @Override
            public ByteBuffer getDataY() {
                ByteBuffer data = buffer.slice();
                data.position(yPos);
                data.limit(yPos + this.getStrideY() * height);
                return data;
            }

            @Override
            public ByteBuffer getDataU() {
                ByteBuffer data = buffer.slice();
                data.position(uPos);
                data.limit(uPos + this.getStrideU() * chromaHeight);
                return data;
            }

            @Override
            public ByteBuffer getDataV() {
                ByteBuffer data = buffer.slice();
                data.position(vPos);
                data.limit(vPos + this.getStrideV() * chromaHeight);
                return data;
            }

            @Override
            public int getStrideY() {
                return stride;
            }

            @Override
            public int getStrideU() {
                return uvStride;
            }

            @Override
            public int getStrideV() {
                return uvStride;
            }

            @Override
            public int getWidth() {
                return width;
            }

            @Override
            public int getHeight() {
                return height;
            }

            @Override
            public VideoFrame.I420Buffer toI420() {
                return this;
            }

            @Override
            public void retain() {
                ++this.refCount;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void release() {
                --this.refCount;
                if (this.refCount == 0) {
                    HardwareVideoDecoder.this.codec.releaseOutputBuffer(outputBufferIndex, false);
                    Object object = HardwareVideoDecoder.this.activeOutputBuffersLock;
                    synchronized (object) {
                        HardwareVideoDecoder.this.activeOutputBuffers--;
                        HardwareVideoDecoder.this.activeOutputBuffersLock.notifyAll();
                    }
                }
            }
        };
    }

    private static void copyI420(ByteBuffer src, int offset, VideoFrame.I420Buffer frameBuffer, int stride, int sliceHeight, int width, int height) {
        int uvStride = stride / 2;
        int chromaWidth = (width + 1) / 2;
        int chromaHeight = sliceHeight % 2 == 0 ? (height + 1) / 2 : height / 2;
        int yPos = offset;
        int uPos = yPos + stride * sliceHeight;
        int vPos = uPos + uvStride * sliceHeight / 2;
        HardwareVideoDecoder.copyPlane(src, yPos, stride, frameBuffer.getDataY(), 0, frameBuffer.getStrideY(), width, height);
        HardwareVideoDecoder.copyPlane(src, uPos, uvStride, frameBuffer.getDataU(), 0, frameBuffer.getStrideU(), chromaWidth, chromaHeight);
        HardwareVideoDecoder.copyPlane(src, vPos, uvStride, frameBuffer.getDataV(), 0, frameBuffer.getStrideV(), chromaWidth, chromaHeight);
        if (sliceHeight % 2 != 0) {
            int strideU = frameBuffer.getStrideU();
            int endU = chromaHeight * strideU;
            HardwareVideoDecoder.copyRow(frameBuffer.getDataU(), endU - strideU, frameBuffer.getDataU(), endU, chromaWidth);
            int strideV = frameBuffer.getStrideV();
            int endV = chromaHeight * strideV;
            HardwareVideoDecoder.copyRow(frameBuffer.getDataV(), endV - strideV, frameBuffer.getDataV(), endV, chromaWidth);
        }
    }

    private static void nv12ToI420(ByteBuffer src, int offset, VideoFrame.I420Buffer frameBuffer, int stride, int sliceHeight, int width, int height) {
        int yPos = offset;
        int uvPos = yPos + stride * sliceHeight;
        int chromaWidth = (width + 1) / 2;
        int chromaHeight = (height + 1) / 2;
        HardwareVideoDecoder.copyPlane(src, yPos, stride, frameBuffer.getDataY(), 0, frameBuffer.getStrideY(), width, height);
        int dstUPos = 0;
        int dstVPos = 0;
        for (int i = 0; i < chromaHeight; ++i) {
            for (int j = 0; j < chromaWidth; ++j) {
                frameBuffer.getDataU().put(dstUPos + j, src.get(uvPos + j * 2));
                frameBuffer.getDataV().put(dstVPos + j, src.get(uvPos + j * 2 + 1));
            }
            dstUPos += frameBuffer.getStrideU();
            dstVPos += frameBuffer.getStrideV();
            uvPos += stride;
        }
    }

    private static void copyPlane(ByteBuffer src, int srcPos, int srcStride, ByteBuffer dst, int dstPos, int dstStride, int width, int height) {
        for (int i = 0; i < height; ++i) {
            HardwareVideoDecoder.copyRow(src, srcPos, dst, dstPos, width);
            srcPos += srcStride;
            dstPos += dstStride;
        }
    }

    private static void copyRow(ByteBuffer src, int srcPos, ByteBuffer dst, int dstPos, int width) {
        for (int i = 0; i < width; ++i) {
            dst.put(dstPos + i, src.get(srcPos + i));
        }
    }

    private static class FrameInfo {
        final long decodeStartTimeMs;
        final int rotation;

        FrameInfo(long decodeStartTimeMs, int rotation) {
            this.decodeStartTimeMs = decodeStartTimeMs;
            this.rotation = rotation;
        }
    }
}

