package org.chromium.content_shell;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import com.vcc.securitysdk.AuthenResponse;
import com.vcc.securitysdk.OnAuthPlayerListener;
import com.vcc.securitysdk.SecureSDK;

import org.json.JSONObject;

import static com.vcc.securitysdk.SecureSDK.AUTHEN_SUCCESSFULLY;
import static com.vcc.securitysdk.SecureSDK.CONFIG_PREBUILD;
import static com.vcc.securitysdk.SecureSDK.MANIFEST_PERMISSION;
import static com.vcc.securitysdk.SecureSDK.PARAMESTER_REQUIRED;
import static com.vcc.securitysdk.SecureSDK.PLAYER_DENIED_SERVER;

public class SecureManager {
    public static final int NON_PERMISSION = -1;
    public static final int PERMISSION_ACCEPTED = 1;
    public static final int PERMISSION_DENIED = 0;

    private static final String TAG = SecureManager.class.getSimpleName();
    private static SecureManager secureManager;
    private Context context;
    private String appKey;
    private String secretKey;
    private String playerId;
    private String versionName;
    private int permission = NON_PERMISSION;
    private OnConfigListener onConfigListener;
    private boolean isRequest = false;

    public static SecureManager shareInstance() {
        if (secureManager == null) {
            secureManager = new SecureManager();
            return secureManager;
        } else
            return secureManager;
    }

    public int getPermission() {
        return permission;
    }

    public void checkStateSecure(Context context, OnConfigListener configListener) {
        if (secureManager != null) {
            if (isRequest)
                return;
            isRequest = true;
            SecureSDK.authPlayer(context, appKey, secretKey, playerId, versionName, new OnAuthPlayerListener() {
                @Override
                public void onAuthorizing() {

                }

                @Override
                public void onAuthFailed(int errorcode, String message) {
                    isRequest = false;
                    ConfigErrorType errorType = ConfigErrorType.CONFIG_PREBUILD;
                    permission = NON_PERMISSION;
                    switch (errorcode) {
                        case CONFIG_PREBUILD:
                            errorType = ConfigErrorType.CONFIG_PREBUILD;
                            break;
                        case MANIFEST_PERMISSION:
                            errorType = ConfigErrorType.MANIFEST_PERMISSION;
                            break;
                        case PARAMESTER_REQUIRED:
                            errorType = ConfigErrorType.PARAMESTER_REQUIRED;
                            break;
                        case PLAYER_DENIED_SERVER:
                            permission = PERMISSION_DENIED;
                            errorType = ConfigErrorType.PLAYER_AUTHENTICATION;
                            break;
                    }
                    pushPermissionUnrequest(errorType, message);
                }

                @Override
                public void onAuthSuccessfully(int type, JSONObject jsonObject) {
                    isRequest = false;
                    if (type == NON_PERMISSION) {
                        permission = NON_PERMISSION;
                        if (getOnConfigListener() != null) {
                            getOnConfigListener().onInitSuccessfully();
                        }
                        return;
                    }
                    permission = PERMISSION_ACCEPTED;
                }
            });
        }
    }

    private void pushPermissionUnrequest(ConfigErrorType errorType, String message) {
        if (getOnConfigListener() == null) {
            android.util.Log.e("Player sdk: ", message);
        } else {
            getOnConfigListener().onInitFailed(errorType, message);
        }
    }

    public OnConfigListener getOnConfigListener() {
        return onConfigListener;
    }

    public void setOnConfigListener(OnConfigListener configListener) {
        this.onConfigListener = configListener;
    }

    /**
     * @deprecated
     */
    private void reauthPlayer() {
        AuthenResponse authenResponse = SecureSDK.reauthPlayer(context, appKey, secretKey, playerId, versionName);
        int errorcode = authenResponse.getErrorCode();
        switch (errorcode) {
            case AUTHEN_SUCCESSFULLY:
                permission = PERMISSION_ACCEPTED;
                if (getOnConfigListener() != null) {
                    getOnConfigListener().onInitSuccessfully();
                }
                break;
            case CONFIG_PREBUILD:
            case MANIFEST_PERMISSION:
            case PARAMESTER_REQUIRED:
                permission = NON_PERMISSION;
                break;
            case PLAYER_DENIED_SERVER:
                permission = PERMISSION_DENIED;
                break;
        }
    }

    /**
     * @return return state your permission. You have permission to using this player
     * true if authenticated otherwise
     */
    public boolean isAllowUsingBrowser() {
        if (permission == PERMISSION_DENIED) {
            if (getOnConfigListener() != null)
                getOnConfigListener().onInitFailed(ConfigErrorType.PLAYER_AUTHENTICATION, "You haven't permission to player content.");
            else
                android.util.Log.e(TAG, "You haven't permisison to player content. You must register OnConfigListener interface to listen state");
            return false;
        }
        if (permission == NON_PERMISSION) {
            if (isInternetAvailable(getContext())) {
                shareInstance().reauthPlayer();
                return getPermission() == PERMISSION_ACCEPTED;
            } else {
                return true;
            }
        }
        return true;
    }

    boolean isInternetAvailable(Context context) {
        ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo netInfo = cm.getActiveNetworkInfo();
        return netInfo != null && netInfo.isConnectedOrConnecting();
    }

    public Context getContext() {
        return context;
    }

    public void setContext(Context context) {
        this.context = context;
    }

    public void configKey(String appKey, String secretKey, String playerId, String versionName) {
        this.appKey = appKey;
        this.secretKey = secretKey;
        this.playerId = playerId;
        this.versionName = versionName;
    }
}
