package org.chromium.content_shell;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.DatabaseErrorHandler;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import com.vccorp.object.LinkVisited;
import com.vccorp.object.User;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

public class DatabaseHandler extends SQLiteOpenHelper {

    private static final String DATABASE_NAME = "VivaHistory";
    private static final int DATABASE_VERSION = 1;
    // Table User
    private static final String TABLE_NAME_USER = "User";
    private static final String VIVA_BROWSER_UIDKEY = "uid";
    private static final String VIVA_BROWSER_NAMEKEY = "name";
    private static final String VIVA_BROWSER_EMAILKEY = "email";
    private static final String VIVA_BROWSER_EXTRAKEY = "extra";
    private static final String VIVA_BROWSER_TIMECREATEKEY = "timeCreated";

    // Table Link`
    private static final String VIVA_BROWSER_IDKEY = "id";
    private static final String TABLE_NAME_VISIT = "LinkVisited";
    private static final String KEY_LINK_ID = "linkId";
    private static final String VIVA_BROWSER_DOMAINNAMEKEY = "domainName";
    private static final String VIVA_BROWSER_URLKEY = "url";
    private static final String VIVA_BROWSER_TITLE = "title";
    private static final String VIVA_BROWSER_REF_IDKEY = "pid";
    private static final String VIVA_BROWSER_TIMEVISITEDKEY = "timeVisited";


    public DatabaseHandler(Context context) {
        super(context, DATABASE_NAME, null, DATABASE_VERSION);
    }

    public DatabaseHandler(Context context, String name, SQLiteDatabase.CursorFactory factory, int version) {
        super(context, name, factory, version);
    }

    public DatabaseHandler(Context context, String name, SQLiteDatabase.CursorFactory factory, int version, DatabaseErrorHandler errorHandler) {
        super(context, name, factory, version, errorHandler);
    }

    public DatabaseHandler(Context context, String name, int version, SQLiteDatabase.OpenParams openParams) {
        super(context, name, version, openParams);
    }


    @Override
    public void onCreate(SQLiteDatabase db) {
        createTable(db);

    }

    private void createTable(SQLiteDatabase db) {
        String createUserTable = String.format("CREATE TABLE IF NOT EXISTS %s (%s INTEGER PRIMARY KEY, %s TEXT, %s TEXT, %s TEXT, %s TEXT)", TABLE_NAME_USER, VIVA_BROWSER_UIDKEY,
                VIVA_BROWSER_NAMEKEY, VIVA_BROWSER_EMAILKEY, VIVA_BROWSER_EXTRAKEY, VIVA_BROWSER_TIMECREATEKEY);
        String createLinkVisitedTable = String.format("CREATE TABLE IF NOT EXISTS %s (%s INTEGER PRIMARY KEY " +
                        "AUTOINCREMENT, %s TEXT, %s TEXT, %s TEXT, %s TEXT, %s TEXT)", TABLE_NAME_VISIT,
                VIVA_BROWSER_IDKEY,
                VIVA_BROWSER_DOMAINNAMEKEY,
                VIVA_BROWSER_URLKEY,
                VIVA_BROWSER_TITLE,
                VIVA_BROWSER_REF_IDKEY,
                VIVA_BROWSER_TIMEVISITEDKEY);
        db.execSQL(createUserTable);
        db.execSQL(createLinkVisitedTable);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        db.execSQL("DROP TABLE IF EXISTS " + TABLE_NAME_USER);
        db.execSQL("DROP TABLE IF EXISTS " + TABLE_NAME_VISIT);
    }

    boolean addUser(User user) {
        ContentValues values = new ContentValues();
        SQLiteDatabase db = this.getWritableDatabase();
        Cursor cursor = db.rawQuery("SELECT " + VIVA_BROWSER_UIDKEY + " FROM " + TABLE_NAME_USER + " where " + VIVA_BROWSER_UIDKEY + "= " + user.getUid(), null);
        if (cursor.getCount() > 0) {
            db.close();
            return updateUser(user);
        } else {
            values.put(VIVA_BROWSER_UIDKEY, user.getUid());
            values.put(VIVA_BROWSER_NAMEKEY, user.getName());
            values.put(VIVA_BROWSER_EMAILKEY, user.getEmail());
            values.put(VIVA_BROWSER_EXTRAKEY, user.getExtra());
            values.put(VIVA_BROWSER_TIMECREATEKEY, user.getTimeCreate());

            long result = db.insert(TABLE_NAME_USER, null, values);
            db.close();
            if (result > 0) {
                return true;
            } else return false;

        }

    }

    private String convertMinisecondToDate(String currentTimeMillis) {
        @SuppressLint("SimpleDateFormat") SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(Long.parseLong(currentTimeMillis));
        return simpleDateFormat.format(calendar.getTime());

    }

    private boolean updateUser(User user) {
        SQLiteDatabase db = this.getWritableDatabase();
        ContentValues values = new ContentValues();
        values.put(VIVA_BROWSER_NAMEKEY, user.getName());
        values.put(VIVA_BROWSER_EMAILKEY, user.getExtra());
        values.put(VIVA_BROWSER_EXTRAKEY, user.getExtra());
        values.put(VIVA_BROWSER_TIMECREATEKEY, convertMinisecondToDate(user.getTimeCreate()));

        int result = db.update(TABLE_NAME_USER, values, VIVA_BROWSER_UIDKEY + " = ?", new String[]{String.valueOf(user.getUid())});
        db.close();
        return result > 0;
    }

    boolean removeUserByUserId(int userId) {
        SQLiteDatabase db = this.getWritableDatabase();
        int result = db.delete(TABLE_NAME_USER, VIVA_BROWSER_UIDKEY + " = ?", new String[]{String.valueOf(userId)});
        db.close();
        return result > 0;
    }

    boolean removeUserByName(String name) {
        SQLiteDatabase db = this.getWritableDatabase();
        int result = db.delete(TABLE_NAME_USER, VIVA_BROWSER_NAMEKEY + " = ?", new String[]{String.valueOf(name)});
        db.close();
        return result > 0;
    }

    boolean removeUserByEmail(String email) {
        SQLiteDatabase db = this.getWritableDatabase();
        int result = db.delete(TABLE_NAME_USER, VIVA_BROWSER_EMAILKEY + " = ?", new String[]{String.valueOf(email)});
        db.close();
        return result > 0;
    }


    public User getUserInfo(int userId) {
        SQLiteDatabase db = this.getReadableDatabase();

        Cursor cursor = db.query(TABLE_NAME_USER, null, VIVA_BROWSER_UIDKEY + " = ?", new String[]{String.valueOf(userId)}, null, null, null);
        if (cursor != null)
            cursor.moveToFirst();

        User user = new User(cursor.getInt(0), cursor.getString(1), cursor.getString(2)
                , cursor.getString(3), cursor.getString(4));
        return user;
    }

    protected List<User> getAllUserInfo() {
        List<User> userList = new ArrayList<>();
        String query = "SELECT * FROM " + TABLE_NAME_USER;

        SQLiteDatabase db = this.getReadableDatabase();
        Cursor cursor = db.rawQuery(query, null);
        cursor.moveToFirst();

        while (!cursor.isAfterLast()) {
            User user = new User(cursor.getInt(0), cursor.getString(1), cursor.getString(2)
                    , cursor.getString(3), cursor.getString(4));
            userList.add(user);
            cursor.moveToNext();
        }
        return userList;
    }

    private String getFirstRowOfTable() {
        SQLiteDatabase db = this.getReadableDatabase();

        Cursor cursor = db.rawQuery("SELECT * FROM " + TABLE_NAME_VISIT + " ORDER BY " + VIVA_BROWSER_TIMEVISITEDKEY + " ASC LIMIT 1", null);
        if (cursor != null)
            cursor.moveToFirst();
        if (cursor.getCount() > 0) {
            String date = cursor.getString(5);
            return date;
        }

        return convertMinisecondToDate(String.valueOf(System.currentTimeMillis()));
    }

    protected List<LinkVisited> getAllHistoryOfUser(int userId) {
        List<LinkVisited> arrLinkVisiteds = new ArrayList<>();
        SQLiteDatabase db = this.getReadableDatabase();
        Cursor cursor = db.rawQuery("SELECT * FROM " + TABLE_NAME_VISIT + " where " + VIVA_BROWSER_UIDKEY + "= " + userId, null);
        cursor.moveToFirst();

        while (!cursor.isAfterLast()) {
            LinkVisited user = new LinkVisited(cursor.getString(0), cursor.getString(1), cursor.getString(2)
                    , cursor.getInt(3), cursor.getString(4));
            arrLinkVisiteds.add(user);
            cursor.moveToNext();
        }
        return arrLinkVisiteds;
    }


    protected List<LinkVisited> getAllHistory() {
        List<LinkVisited> arrLinkVisiteds = new ArrayList<>();
        SQLiteDatabase db = this.getReadableDatabase();
        Cursor cursor = db.rawQuery("SELECT * FROM " + TABLE_NAME_VISIT, null);
        cursor.moveToFirst();

        while (!cursor.isAfterLast()) {
            LinkVisited user = new LinkVisited(cursor.getString(0), cursor.getString(1), cursor.getString(2)
                    , cursor.getInt(3), cursor.getString(4));
            arrLinkVisiteds.add(user);
            cursor.moveToNext();
        }
        return arrLinkVisiteds;
    }

    protected List<LinkVisited> getHistoryByRange(Date start, Date end) {
        List<LinkVisited> arrLinkVisiteds = new ArrayList<>();
        SQLiteDatabase db = this.getReadableDatabase();
        String sql = "SELECT * FROM " + TABLE_NAME_VISIT + " WHERE " + VIVA_BROWSER_TIMEVISITEDKEY + " BETWEEN '" + start + "' AND '" + end + "'";
        Cursor cursor = db.rawQuery(sql, null);
        cursor.moveToFirst();

        while (!cursor.isAfterLast()) {
            LinkVisited user = new LinkVisited(cursor.getString(0), cursor.getString(1), cursor.getString(2)
                    , cursor.getInt(3), cursor.getString(4));
            arrLinkVisiteds.add(user);
            cursor.moveToNext();
        }
        return arrLinkVisiteds;
    }


    public void addLinkVisited(LinkVisited linkVisited) {
        ContentValues values = new ContentValues();
        SQLiteDatabase db = this.getWritableDatabase();
        values.put(VIVA_BROWSER_DOMAINNAMEKEY, linkVisited.getLinkDomain());
        values.put(VIVA_BROWSER_URLKEY, linkVisited.getLinkURL());
        values.put(VIVA_BROWSER_TITLE, linkVisited.getLinkTitle());
        values.put(VIVA_BROWSER_REF_IDKEY, linkVisited.getRefUid());
        values.put(VIVA_BROWSER_TIMEVISITEDKEY, linkVisited.getTimeVisited());

        long result = db.insert(TABLE_NAME_VISIT, null, values);

        db.close();
    }

    public void removeHistory30Days() {
        String dateBeforeCreated = getFirstRowOfTable();
        @SuppressLint("SimpleDateFormat") SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        try {
            Date dateBefore = simpleDateFormat.parse(dateBeforeCreated);
            Date dateAfter = simpleDateFormat.parse(convertMinisecondToDate(String.valueOf(System.currentTimeMillis())));

            Calendar calendar = Calendar.getInstance();
            calendar.setTime(dateBefore);
            calendar.add(Calendar.DATE, 30);
            String output = simpleDateFormat.format(calendar.getTime());
            Date after30 = simpleDateFormat.parse(output);
            if (after30.compareTo(dateAfter) < 0) {
                removeHistoryByRange(dateBefore, after30);
            }


        } catch (ParseException e) {
            e.printStackTrace();
        }

//        removeHistoryByRange();
    }

    public void removeHistoryByRange(Date start, Date end) {
        @SuppressLint("SimpleDateFormat") SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        String dateStart = simpleDateFormat.format(start);
        String dateEnd = simpleDateFormat.format(end);
        SQLiteDatabase db = this.getWritableDatabase();
        String sql = "DELETE FROM " + TABLE_NAME_VISIT + " WHERE " + VIVA_BROWSER_TIMEVISITEDKEY + " BETWEEN '" + dateStart + "' AND '" + dateEnd + "'";
        db.execSQL(sql);
    }


    public void removeHistoryByUserId(int userId) {
        SQLiteDatabase db = this.getWritableDatabase();
        String sql = "DELETE FROM " + TABLE_NAME_VISIT + " WHERE " + VIVA_BROWSER_REF_IDKEY + " = " + userId;
        db.execSQL(sql);
    }
}
