// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package com.vccorp.device.battery;

import com.vccorp.base.Log;

import com.vccorp.device.battery.BatteryStatusManager.BatteryStatusCallback;
import com.vccorp.device.mojom.BatteryMonitor;
import com.vccorp.device.mojom.BatteryStatus;
import com.vccorp.services.service_manager.InterfaceFactory;

import org.chromium.base.ThreadUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

/**
 * Factory that creates instances of BatteryMonitor implementations and notifies them about battery
 * status changes.
 */
public class BatteryMonitorFactory implements InterfaceFactory<BatteryMonitor> {
    private static final String TAG = "BattMonitorFactory";

    // Backing source of battery information.
    private final BatteryStatusManager mManager;
    // Monitors currently interested in the battery status notifications.
    private final HashSet<BatteryMonitorImpl> mSubscribedMonitors =
            new HashSet<BatteryMonitorImpl>();

    private final BatteryStatusCallback mCallback = new BatteryStatusCallback() {
        @Override
        public void onBatteryStatusChanged(BatteryStatus batteryStatus) {
            ThreadUtils.assertOnUiThread();

            List<BatteryMonitorImpl> monitors = new ArrayList<>(mSubscribedMonitors);
            for (BatteryMonitorImpl monitor : monitors) {
                monitor.didChange(batteryStatus);
            }
        }
    };

    public BatteryMonitorFactory() {
        mManager = new BatteryStatusManager(mCallback);
    }

    @Override
    public BatteryMonitor createImpl() {
        ThreadUtils.assertOnUiThread();

        if (mSubscribedMonitors.isEmpty() && !mManager.start()) {
            Log.e(TAG, "BatteryStatusManager failed to start.");
        }
        // TODO(ppi): record the "BatteryStatus.StartAndroid" histogram here once we have a Java API
        //            for UMA - http://crbug.com/442300.

        BatteryMonitorImpl monitor = new BatteryMonitorImpl(this);
        mSubscribedMonitors.add(monitor);
        return monitor;
    }

    void unsubscribe(BatteryMonitorImpl monitor) {
        ThreadUtils.assertOnUiThread();

        assert mSubscribedMonitors.contains(monitor);
        mSubscribedMonitors.remove(monitor);
        if (mSubscribedMonitors.isEmpty()) {
            mManager.stop();
        }
    }
}
