package com.vcc.securitysdk;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.support.v4.content.ContextCompat;
import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.security.MessageDigest;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

public class SecureSDK {
    //region Parameters for authen player
    public static final int MANIFEST_PERMISSION = 0;
    public static final int PLAYER_DENIED_SERVER = 1;
    public static final int PARAMESTER_REQUIRED = 2;
    public static final int CONFIG_PREBUILD = 3;
    public static final int AUTHEN_SUCCESSFULLY = 200;
    static final long MILLISECOND = 1000;
    private static final String PLATFORM = "android";
    private static final String LIBRAY_NAME = "SecureSDK";
    private static final int PERMISSION_ACCEPTED = 1;
    private static final int NON_PERMISSION = -1;
    private static Thread threadRequest;
    private static boolean request = false;
    //endregion

    //region Methods for init authentication
    static native String verifyPlayerSdk(
            String appKey,
            String playerId,
            String expiredTime,
            String enc,
            String currentVersion);

    private static boolean checkPermission(Context context, String permissionStr) {
        // Here, thisActivity is the current activity
        return ContextCompat.checkSelfPermission(context, permissionStr) == PackageManager.PERMISSION_GRANTED;
    }


    private static Date addDays(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DATE, 1); //minus number would decrement the days
        return cal.getTime();
    }

    private static boolean check30Days() {
        Date dateSince = new Date(2019, 1, 22);
        Date dateNow = new Date();
        Calendar cal = Calendar.getInstance();
        cal.setTime(dateSince);
        cal.add(Calendar.DATE, 30);
        return cal.getTime().compareTo(dateNow) >= 0;
    }

    private static String hashMd5(String expiredTime) {
        String content = expiredTime + "-player-verified";
        final String MD5 = "MD5";
        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest
                    .getInstance(MD5);
            digest.update(content.getBytes());
            byte messageDigest[] = digest.digest();

            // Create Hex String
            StringBuilder hexString = new StringBuilder();
            for (byte aMessageDigest : messageDigest) {
                String h = Integer.toHexString(0xFF & aMessageDigest);
                while (h.length() < 2)
                    h = "0" + h;
                hexString.append(h);
            }
            return hexString.toString();

        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    private static void destroyThread() {
        try {
            if (threadRequest != null) {
                threadRequest.interrupt();
                threadRequest.stop();
                threadRequest = null;
            }
        } catch (Exception ignored) {
        }
    }
    //endregion

    /**
     * Call this method to verify your player
     *
     * @param context   Context of your player is place
     * @param appKey    AppKey for your app
     * @param secretKey SecretKey for your app
     * @param playerId  PlayerId for your app
     */
    public static AuthenResponse reauthPlayer(Context context, String appKey, String secretKey, String playerId, String currentVersion) {
        try {
            System.loadLibrary(LIBRAY_NAME);
        } catch (Exception e) {
            android.util.Log.e("Player sdk: ", "Cannot load extension. Please try again.");
            AuthenResponse authenResponse = new AuthenResponse();
            authenResponse.setErrorCode(CONFIG_PREBUILD);
            authenResponse.setJsonResponse(null);
            return authenResponse;
        } catch (Error e2) {
            android.util.Log.e("Player sdk: ", "Cannot load extension. Please try again.");
            AuthenResponse authenResponse = new AuthenResponse();
            authenResponse.setErrorCode(CONFIG_PREBUILD);
            authenResponse.setJsonResponse(null);
            return authenResponse;
        }

        if (!checkPermission(context, Manifest.permission.INTERNET)) {
            android.util.Log.e("Player sdk: ", "OnPlayerCallback cannot null");
            AuthenResponse authenResponse = new AuthenResponse();
            authenResponse.setErrorCode(MANIFEST_PERMISSION);
            authenResponse.setJsonResponse(null);
            return authenResponse;
        }

        if (TextUtils.isEmpty(appKey)
                || TextUtils.isEmpty(secretKey)) {
            android.util.Log.e("Player sdk: ", "AppKey, Secret Key must not empty! OnPlayerCallback cannot null");
            AuthenResponse authenResponse = new AuthenResponse();
            authenResponse.setErrorCode(PARAMESTER_REQUIRED);
            authenResponse.setJsonResponse(null);
            return authenResponse;
        }

        String encrypt;

        Date currentDay = new Date();
        final Date expriedTime = addDays(currentDay);

        try {
            Map<String, Object> claims = new HashMap<>();
            claims.put("appkey", appKey);
            claims.put("player", playerId);
            claims.put("platform", PLATFORM);
            claims.put("package_name", context.getPackageName());

            String key = appKey + "-" + secretKey;

            encrypt = Jwts.builder()
                    .setClaims(claims)
                    .setHeaderParam("typ", "JWT")
                    .setIssuedAt(currentDay)
                    .setExpiration(expriedTime)
                    .signWith(SignatureAlgorithm.HS256, key.getBytes())
                    .compact();
        } catch (Exception e) {
            android.util.Log.e("Player sdk: ", "OnPlayerCallback cannot null");
            AuthenResponse authenResponse = new AuthenResponse();
            authenResponse.setErrorCode(CONFIG_PREBUILD);
            authenResponse.setJsonResponse(null);
            return authenResponse;
        }

        final String finalEncrypt = encrypt;
        destroyThread();
        request = true;
        destroyThread();
        String expiredTime = hashMd5(String.valueOf(expriedTime.getTime() / MILLISECOND));
        String verifyResponse = verifyPlayerSdk(appKey, playerId, expiredTime, finalEncrypt, currentVersion);
        int errorCode = -1;
        JSONUtilParser parser = new JSONUtilParser();
        JSONObject jsonResponse = parser.getJSONObject(verifyResponse);
        if (jsonResponse != null)
            errorCode = parser.getInteger("code", jsonResponse);
        if (errorCode == 1) {
            android.util.Log.e("Player sdk: ", "Create PlayerManagerment successfully.");
            request = false;
            AuthenResponse authenResponse = new AuthenResponse();
            authenResponse.setErrorCode(AUTHEN_SUCCESSFULLY);
            authenResponse.setJsonResponse(jsonResponse);
            return authenResponse;
        } else if (errorCode == -1) {
            android.util.Log.e("Player sdk: ", "Cannot get data");
            request = false;
            AuthenResponse authenResponse = new AuthenResponse();
            authenResponse.setErrorCode(CONFIG_PREBUILD);
            authenResponse.setJsonResponse(jsonResponse);
            return authenResponse;
        } else {
            android.util.Log.e("Player sdk: ", "You have not permission!");
            request = false;

            AuthenResponse authenResponse = new AuthenResponse();
            authenResponse.setErrorCode(PLAYER_DENIED_SERVER);
            authenResponse.setJsonResponse(jsonResponse);
            return authenResponse;
        }
    }

    /**
     * Call this method to verify your player
     *
     * @param context   Context of your player is place
     * @param appKey    AppKey for your app
     * @param secretKey SecretKey for your app
     * @param playerId  PlayerId for your app
     * @param listener  Callback handle when authentication is finished
     */
    public static void authPlayer(Context context, final String appKey, String secretKey,
                                  final String playerId, final String currentVersion, final OnAuthPlayerListener listener) {
        try {
            System.loadLibrary(LIBRAY_NAME);
        } catch (Exception e) {
            if (listener == null) {
                android.util.Log.e("Player sdk: ", "Cannot load extension. Please try again.");
            } else {
                listener.onAuthFailed(CONFIG_PREBUILD, "Cannot load extension. Please try again.");
            }
            return;
        } catch (Error e2) {
            if (listener == null) {
                android.util.Log.e("Player sdk: ", "Cannot load extension. Please try again.");
            } else {
                listener.onAuthFailed(CONFIG_PREBUILD, "Cannot load extension. Please try again.");
            }
            return;
        }

        if (!checkPermission(context, Manifest.permission.INTERNET)) {
            if (listener == null) {
                android.util.Log.e("Player sdk: ", "OnPlayerCallback cannot null");
            } else {
                listener.onAuthFailed(MANIFEST_PERMISSION, "You not permission internet. Please check again!");
            }
            return;
        }

        if (request) {
            if (listener != null)
                listener.onAuthorizing();
            return;
        }
        if (listener != null) {
            listener.onAuthorizing();
        }

        if (TextUtils.isEmpty(appKey)
                || TextUtils.isEmpty(secretKey)) {
            if (listener == null) {
                android.util.Log.e("Player sdk: ", "AppKey, Secret Key must not empty! OnPlayerCallback cannot null");
            } else {
                listener.onAuthFailed(PARAMESTER_REQUIRED, "AppKey, Secret Key must not empty!");
            }
            return;
        }

        String encrypt;

        Date currentDay = new Date();
        final Date expriedTime = addDays(currentDay);

        try {
            Map<String, Object> claims = new HashMap<>();
            claims.put("appkey", appKey);
            claims.put("player", playerId);
            claims.put("platform", PLATFORM);
            claims.put("package_name", context.getPackageName());

            String key = appKey + "-" + secretKey;

            encrypt = Jwts.builder()
                    .setClaims(claims)
                    .setHeaderParam("typ", "JWT")
                    .setIssuedAt(currentDay)
                    .setExpiration(expriedTime)
                    .signWith(SignatureAlgorithm.HS256, key.getBytes())
                    .compact();
        } catch (Exception e) {
            if (listener == null) {
                android.util.Log.e("Player sdk: ", "OnPlayerCallback cannot null");
            } else {
                listener.onAuthFailed(CONFIG_PREBUILD, "You not permission internet. Please check again!");
            }
            return;
        }

        final String finalEncrypt = encrypt;
        destroyThread();
        request = true;
        threadRequest = new Thread(new Runnable() {
            @Override
            public void run() {
                destroyThread();
                String expiredTime = hashMd5(String.valueOf(expriedTime.getTime() / MILLISECOND));
                String verifyResponse = verifyPlayerSdk(appKey, playerId, expiredTime, finalEncrypt, currentVersion);
//                Log.d("verifyResponse",verifyResponse);
                int errorCode = -1;

                JSONUtilParser parser = new JSONUtilParser();
                JSONObject jsonResponse = parser.getJSONObject(verifyResponse);
                if (jsonResponse != null)
                    errorCode = parser.getInteger("code", jsonResponse);
                request = false;
                if (errorCode == 1) {
                    if (listener == null) {
                        android.util.Log.e("Player sdk: ", "Create PlayerManagerment successfully.");
                    } else {
                        listener.onAuthSuccessfully(PERMISSION_ACCEPTED,jsonResponse);
                    }
                } else if (errorCode == -1) {
                    if (listener == null) {
                        android.util.Log.e("Player sdk: ", "Create PlayerManagerment successfully.");
                    } else {
                        listener.onAuthSuccessfully(NON_PERMISSION,jsonResponse);
                    }
                } else {
                    if (listener == null) {
                        android.util.Log.e("Player sdk: ", "You have not permission!");
                    } else {
                        listener.onAuthFailed(PLAYER_DENIED_SERVER, "You have not permission!");
                    }
                }
            }
        }, "ThreadAuthPlayerRequest");
        threadRequest.start();
    }

    /**
     * Created by PC0353 on 7/21/2016.
     */
    public static class JSONUtilParser {

        int getInteger(String name, JSONObject obj) {
            try {
                return obj.getInt(name);
            } catch (Exception ex) {
                return -1;
            }
        }

        public String getString(String name, JSONObject obj) {
            try {
                return obj.getString(name);
            } catch (Exception ex) {
                return "";
            }
        }

        public JSONObject getJSONObject(String name, JSONObject obj) {
            try {
                return obj.getJSONObject(name);
            } catch (Exception ex) {
                return null;
            }
        }

        public JSONArray getJSONArray(String name, JSONObject obj) {
            try {
                return obj.getJSONArray(name);
            } catch (Exception ex) {
                return null;
            }
        }

        public JSONObject getJSONObject(String data) {
            try {
                return new JSONObject(data);
            } catch (Exception ex) {
                return null;
            }
        }

        public Object getObject(String name, JSONObject obj) {
            try {
                return obj.get(name);
            } catch (Exception ex) {
                return null;
            }
        }
    }

}
