package com.vcc.playerwrappersdk.controllers;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.graphics.drawable.Drawable;
import android.os.AsyncTask;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.v4.content.ContextCompat;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.ProgressBar;
import android.widget.Toast;

import com.google.android.exoplayer.TrackRenderer;
import com.google.android.libraries.mediaframework.exoplayerextensions.Video;
import com.google.android.libraries.mediaframework.layeredvideo.PlaybackControlLayer;
import com.vcc.playerwrappersdk.HttpStatus;
import com.vcc.playerwrappersdk.Log;
import com.vcc.playerwrappersdk.entities.PlayResponse;
import com.vcc.playerwrappersdk.entities.TimeLive;
import com.vcc.playerwrappersdk.entities.VerifyResponse;
import com.vcc.playerwrappersdk.utils.EBusData;
import com.vcc.playerwrappersdk.utils.JSONUtilParser;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.lang.ref.WeakReference;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;

import static com.vcc.playerwrappersdk.utils.InvokedMethodUtils.EVENT;
import static com.vcc.playerwrappersdk.utils.InvokedMethodUtils.VALUE;

/**
 * Created by TAKASHI20 on 10/18/2016.
 * Class owned by PlayerAdsSDK at package vcc.com.playersdk
 */
public class PlayerController implements PlaybackControlLayer.FullscreenCallback, PlaybackControlLayer.PlayCallback, Parcelable {
    public static final String TAG = "Player sdk: ";
    //region Initial and variable
    private static final String PORTRAIT = "2";
    private static final String LANDSCAPE = "1";
    private static final Object lock = new Object();
    private static final int NON_PERMISSION = -1;
    private static final int PERMISSION_ACCEPTED = 1;
    private static final int PERMISSION_DENIED = 0;
    /**
     * The player is neither prepared or being prepared.
     */
    public static final int STATE_IDLE = 1;
    /**
     * The player is being prepared.
     */
    public static final int STATE_PREPARING = 2;
    /**
     * The player is prepared but not able to immediately play from the current position. The cause
     * is {@link TrackRenderer} specific, but this state typically occurs when more data needs
     * to be buffered for playback to start.
     */
    public static final int STATE_BUFFERING = 3;
    /**
     * The player is prepared and able to immediately play from the current position. The player will
     * be playing if {getPlayWhenReady} returns true, and paused otherwise.
     */
    public static final int STATE_READY = 4;
    /**
     * The player has finished playing the media.
     */
    public static final int STATE_ENDED = 5;

    public static final int TIME_DEFAULT = 5 * 1000;//5 second
    private static final String SUCCESS_CODE = "200";

    private ImaPlayer imaPlayer;
    /**
     * The {@link FrameLayout} that will contain the video player.
     */
    private FrameLayout videoPlayerContainer;

    private WeakReference<OnPlayerCallback> weakReferencePlayerCallback;
    /**
     * Call this method after config player container is called.
     * Method: setVideoPlayerContainer(...) , setOnPlayerCallback(...), setLogoPlayer(...),addActionButton(...)
     *
     * @param videoContentUrl   @link{@link Video} link to video ads
     * @param adTagUrl   Link video content to play
     * @param videoTitle Title of video
     * @param autoPlay   Set when play available
     */
    private Video videoCopyUrl;
    private String videoTitle;
    private boolean autoPlay;
    private int permission = NON_PERMISSION;
    private ProgressBar progressBar;
    private boolean isBuilded = false;
    private MyParseData myParseData;
    private boolean shouldHide;
    private boolean isFixedContent;
    private long startTime = 0;
    private boolean isRequest = false;
    private int timeInterval;
    private TimerTask timerTask;
    private Timer timer;
    private Handler handler = new Handler();
    private boolean isRunning = false;

    private Handler handlerMainUI = new Handler(Looper.getMainLooper());

    protected PlayerController(Parcel in) {
        videoTitle = in.readString();
        autoPlay = in.readByte() != 0;
        permission = in.readInt();
        isBuilded = in.readByte() != 0;
        shouldHide = in.readByte() != 0;
        isFixedContent = in.readByte() != 0;
        startTime = in.readLong();
        isRequest = in.readByte() != 0;
        timeInterval = in.readInt();
        isRunning = in.readByte() != 0;
    }

    public static final Creator<PlayerController> CREATOR = new Creator<PlayerController>() {
        @Override
        public PlayerController createFromParcel(Parcel in) {
            return new PlayerController(in);
        }

        @Override
        public PlayerController[] newArray(int size) {
            return new PlayerController[size];
        }
    };

    private Date addDays(Date date, int days) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DATE, days); //minus number would decrement the days
        return cal.getTime();
    }

    public boolean setContext(Context context) {
        try {
            Config.shared().setContext(context);
            return true;
        } catch (Exception ex) {
            Log.d(ex.getMessage());
        }
        return false;
    }

    public PlayerController(Context context, String appkey, String secretKey, String playerId, OnPlayerCallback playerCallback) {
        Config.initialized(context);
        setOnPlayerCallback(playerCallback);
        Config.shared().setAppkey(appkey);
        Config.shared().setSecretKey(secretKey);
        Config.shared().setPlayerId(playerId);

        if (!checkPermission(Manifest.permission.INTERNET)) {
            if (playerCallback == null) {
                android.util.Log.e("Player sdk: ", "OnPlayerCallback cannot null");
            } else {
                playerCallback.onError("You not permission internet. Please check again!");
            }
        } else {
            if (TextUtils.isEmpty(appkey)
                    || TextUtils.isEmpty(secretKey)) {
                permission = NON_PERMISSION;
                if (playerCallback == null) {
                    android.util.Log.e("Player sdk: ", "AppKey, Secret Key must not empty! OnPlayerCallback cannot null");
                } else {
                    playerCallback.onError("AppKey, Secret Key must not empty!");
                }
                return;
            }

            String encrypt = "";

            Date currentDay = new Date();
            final Date expriedTime = addDays(currentDay, 1);


            try {
                Map<String, Object> claims = new HashMap<>();
                claims.put("appkey", getAppkey());
                claims.put("player", getPlayerId());
                claims.put("platform", "android");
                claims.put("package_name", getActivity().getPackageName());

                String key = appkey + "-" + secretKey;

                encrypt = Jwts.builder()
                        .setClaims(claims)
                        .setHeaderParam("typ", "JWT")
                        .setIssuedAt(currentDay)
                        .setExpiration(expriedTime)
                        .signWith(SignatureAlgorithm.HS256, key.getBytes())
                        .compact();
            } catch (Exception e) {
                pushPermissionDenied();
                return;
            }

            String baseUrl = "https://adminplayer.sohatv.vn/secure/verify/" + appkey + "/" + getPlayerId() + "/" + encrypt;
            Call<VerifyResponse> call = APITransaction.builderJSONWithCustomUrl().verifySDK(baseUrl);
            Callback<VerifyResponse> callback = new Callback<VerifyResponse>() {
                static final long MILLISECOND = 1000;

                @Override
                public void onResponse(Call<VerifyResponse> call, Response<VerifyResponse> response) {
                    if (response == null || response.body() == null) {
                        pushPermissionDenied();
                        return;
                    }

                    VerifyResponse verifyResponse = response.body();
                    if (verifyResponse.getStatus().equals(SUCCESS_CODE)) {
                        String hashMd5 = hashMd5(String.valueOf(expriedTime.getTime() / MILLISECOND));

                        String md5 = verifyResponse.getData();
                        if (!TextUtils.isEmpty(hashMd5) && !TextUtils.isEmpty(md5)) {
                            if (hashMd5.equals(md5)) {
                                permission = PERMISSION_ACCEPTED;
                                if (isBuilded) {
                                    buildContent();
                                }
                                return;
                            }
                        }
                        pushPermissionDenied();
                    }
                }

                @Override
                public void onFailure(Call<VerifyResponse> call, Throwable t) {
                    pushPermissionDenied();
                }
            };
            call.enqueue(callback);

        }
    }

    private void pushPermissionDenied() {
        permission = PERMISSION_DENIED;
        if (getOnPlayerCallback() == null) {
            android.util.Log.e("Player sdk: ", "You have not permission!");
        } else {
            getOnPlayerCallback().onError("You have not permission!");
        }
    }

    private String hashMd5(String expiredTime) {
        String content = expiredTime + "-player-verified";
        final String MD5 = "MD5";
        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest
                    .getInstance(MD5);
            digest.update(content.getBytes());
            byte messageDigest[] = digest.digest();

            // Create Hex String
            StringBuilder hexString = new StringBuilder();
            for (byte aMessageDigest : messageDigest) {
                String h = Integer.toHexString(0xFF & aMessageDigest);
                while (h.length() < 2)
                    h = "0" + h;
                hexString.append(h);
            }
            return hexString.toString();

        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

//    public static synchronized PlayerController initialized(Context context,
//                                                            String appKey, String secretKey,
//                                                            String playerId, OnPlayerCallback playerCallback) {
//        Config.shared().getContext() = context;
//        synchronized (lock) {
//            if (Config.initialized())
//                return new PlayerController(appKey, secretKey, playerId, playerCallback);
//            else
//                return null;
//        }
//    }

    public void setLogoPlayer(Drawable logo) {
        try {
            imaPlayer.setLogoImage(logo);
        } catch (Exception ex) {
            android.util.Log.e(TAG, "Failed: " + ex.getMessage());
        }
    }

    public void setLogoPlayer(int resLogo) {
        try {
            Drawable logo = Config.shared().getContext().getResources().getDrawable(resLogo);
            imaPlayer.setLogoImage(logo);
        } catch (Exception ex) {
            android.util.Log.e(TAG, "Failed: " + ex.getMessage());
        }
    }

    public boolean addActionButton(Drawable iconbtn, String titleBtn, View.OnClickListener onClickListener) {
        try {
            imaPlayer.addActionButton(iconbtn, titleBtn, onClickListener);
            return true;
        } catch (Exception ex) {
            android.util.Log.e(TAG, "Failed: " + ex.getMessage());
            return false;
        }
    }

    public boolean addActionButtonRes(int resIcon, int resStr, View.OnClickListener onClickListener) {
        try {
            imaPlayer.addActionButton(
                    Config.shared().getContext().getResources().getDrawable(resIcon),
                    Config.shared().getContext().getResources().getString(resStr),
                    onClickListener);
            return true;
        } catch (Exception ex) {
            android.util.Log.e(TAG, "Failed: " + ex.getMessage());
            return false;
        }
    }

    public void releasePlayer() {
        if (imaPlayer != null) {
            imaPlayer.release();
        }
    }

    private ImaPlayer getImaPlayer() {
        return imaPlayer;
    }

    private void setImaPlayer(ImaPlayer imaPlayer) {
        this.imaPlayer = imaPlayer;
    }
    //endregion

    //region Play/Pause action control
    public boolean playVideo() {
        if (permission == NON_PERMISSION || permission == PERMISSION_DENIED) {
            try {
                imaPlayer.pause();
            } catch (Exception e) {
                Log.e(TAG, "Failed " + e.getMessage());
            }
            android.util.Log.e(TAG, "You have not permission!");
            return false;
        }
        try {
            // Now that the player is set up, let's start playing.
            imaPlayer.play();
            return true;
        } catch (Exception ex) {
            Log.e(TAG, "Error: " + ex.getMessage());
            return false;
        }
    }

    public boolean pauseVideo() {
        if (permission == NON_PERMISSION || permission == PERMISSION_DENIED) {
            android.util.Log.e(TAG, "You have not permission!");
            return false;
        }
        try {
            // Now that the player is set up, let's start playing.
            imaPlayer.pause();
            return true;
        } catch (Exception ex) {
            Log.e(TAG, "Error: " + ex.getMessage());
            return false;
        }
    }

    public boolean resumeVideo() {
        if (permission == NON_PERMISSION || permission == PERMISSION_DENIED) {
            try {
                imaPlayer.pause();
            } catch (Exception e) {
                Log.e(TAG, "Failed " + e.getMessage());
            }
            android.util.Log.e(TAG, "You have not permission!");
            return false;
        }
        try {
            // Now that the player is set up, let's start playing.
            imaPlayer.play();
            return true;
        } catch (Exception ex) {
            Log.e(TAG, "Error: " + ex.getMessage());
            return false;
        }
    }

    public boolean stopVideo() {
        try {
            // Now that the player is set up, let's start playing.
            imaPlayer.pause();
            imaPlayer.release();
            return true;
        } catch (Exception ex) {
            Log.e(TAG, "Error: " + ex.getMessage());
            return false;
        }
    }

    public int getCurrentPosition() {
        if (getImaPlayer() != null)
            return getImaPlayer().getCurrentPosition();
        return -1;
    }

    public int getLengthVideo() {
        if (getImaPlayer() != null)
            return getImaPlayer().getLengthVideo();
        return -1;
    }

    public void seekTo(int position) {
        if (getImaPlayer() != null)
            getImaPlayer().seekTo(position);
    }

    public boolean isPlaying() {
        return getImaPlayer() != null && getImaPlayer().isPlaying();
    }

    public void getTimeCurrentLive(@NonNull final OnCurrentTimeLiveCallback callback) {
        if (permission == NON_PERMISSION || permission == PERMISSION_DENIED) {
            if (imaPlayer != null)
                imaPlayer.pause();
            callback.onCurrentTime(false, 0, 0, 0);
            android.util.Log.e(TAG, "You have not permission!");
            return;
        }

        if (videoCopyUrl == null || TextUtils.isEmpty(videoCopyUrl.getUrl())) {
            callback.onCurrentTime(false, 0, 0, 0);
            android.util.Log.e(TAG, "Content url is invalid. Check again");
            return;
        }

        if (videoCopyUrl.getVideoType() != Video.VideoType.HLS) {
            callback.onCurrentTime(false, 0, 0, 0);
            android.util.Log.e(TAG, "Video must live.");
            return;
        }
        String urlEncode = null;
        try {
            urlEncode = URLEncoder.encode(videoCopyUrl.getUrl(), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            android.util.Log.e(TAG, "Encode url failed: " + e.getMessage());
        }
//        android.util.Log.d(TAG, "URLEncode: " + urlEncode);

        if (TextUtils.isEmpty(urlEncode)) {
            callback.onCurrentTime(false, 0, 0, 0);
            android.util.Log.e(TAG, "Url invalid to get time live.");
            return;
        }

        Call<TimeLive> call = APITransaction.builderGetTime().getTimeLive(urlEncode);
        Callback<TimeLive> cbTime = new Callback<TimeLive>() {
            @Override
            public void onResponse(Call<TimeLive> call, Response<TimeLive> response) {
                if (response == null || response.body() == null || response.body().getData() == null) {
                    callback.onCurrentTime(false, 0, 0, 0);
                    android.util.Log.e(TAG, "Cannot get current time error response empty");
                    return;
                }
                long timeStart = 0, timeCurrent = 0, timeEnd = 0;
                try {
                    timeStart = Long.parseLong(response.body().getData().getTimeStart());
                } catch (NumberFormatException e) {
                    e.printStackTrace();
                }
                try {
                    timeCurrent = Long.parseLong(response.body().getData().getTimeCurrent());
                } catch (NumberFormatException e) {
                    e.printStackTrace();
                }
                try {
                    timeEnd = Long.parseLong(response.body().getData().getTimeEnd());
                } catch (NumberFormatException e) {
                    e.printStackTrace();
                }

                callback.onCurrentTime(true, timeStart, timeCurrent, timeEnd);
            }

            @Override
            public void onFailure(Call<TimeLive> call, Throwable t) {
                callback.onCurrentTime(false, 0, 0, 0);
                android.util.Log.e(TAG, "Cannot get current time error require api");
            }
        };
        call.enqueue(cbTime);
    }

    public boolean isMuted() {
        return imaPlayer != null && imaPlayer.isMute();
    }

    public boolean setMute(boolean mute) {
        return imaPlayer != null && imaPlayer.setMute(mute);
    }
    //endregion

    //region Override method and callback

    /**
     * When the video player goes into fullscreen, hide the video list so that the video player can
     * occupy the entire screen.
     */
    @Override
    public void onGoToFullscreen() {
        OnPlayerCallback onPlayerCallback = getOnPlayerCallback();
        if (onPlayerCallback == null) {
            android.util.Log.e(TAG, "OnPlayerCallback cannot null");
            return;
        }
        onPlayerCallback.onFullScreen();
    }

    /**
     * When the player returns from fullscreen, show the video list again.
     */
    @Override
    public void onReturnFromFullscreen() {
        OnPlayerCallback onPlayerCallback = getOnPlayerCallback();
        if (onPlayerCallback == null) {
            android.util.Log.e("Player sdk: ", "OnPlayerCallback cannot null");
            return;
        }
        onPlayerCallback.onNormalScreen();
    }

    @Override
    public void onPlay() {
        if (permission == NON_PERMISSION || permission == PERMISSION_DENIED) {
            imaPlayer.pause();
            android.util.Log.e(TAG, "You have not permission!");
            return;
        }
        OnPlayerCallback onPlayerCallback = getOnPlayerCallback();
        if (onPlayerCallback == null) {
            android.util.Log.e("Player sdk: ", "OnPlayerCallback cannot null");
            return;
        }
        onPlayerCallback.onPlay();
    }

    private void buildContent() {
        if (myParseData != null) {
            if (myParseData.getStatus() == AsyncTask.Status.RUNNING) {
                myParseData.cancel(true);
                myParseData = null;
            }
        }
        if (videoCopyUrl == null) {
            if (getOnPlayerCallback() != null)
                getOnPlayerCallback().onError("Video content url is not null!");
            return;
        }
        myParseData = new MyParseData(videoCopyUrl.getUrl());
        myParseData.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, (Void) null);
    }

    private void createDirectUrl() {
        if (progressBar != null)
            progressBar.setVisibility(View.GONE);

        if (imaPlayer != null) {
            imaPlayer.release();
        }

        // If there was previously a video player in the container, remove it.
        videoPlayerContainer.removeAllViews();

        imaPlayer = new ImaPlayer(Config.shared().getContext(),
                videoPlayerContainer,
                videoCopyUrl,
                videoTitle,
                isFixedContent);
        OnPlayerCallback onPlayerCallback = getOnPlayerCallback();
        if (onPlayerCallback != null)
            imaPlayer.setOnPlayerCallback(onPlayerCallback);
        imaPlayer.hideController(shouldHide);
        imaPlayer.setAutoplay(autoPlay);
        imaPlayer.setFullscreenCallback(this);
        imaPlayer.setPlayCallBack(this);
    }

    public Video getVideoContentUrl() {
        return videoCopyUrl;
    }

    public PlayerController setVideoContentUrl(Video videoContent) {
        this.videoCopyUrl = new Video(videoContent.getUrl(), videoContent.getVideoType(), videoContent.getContentId());
        return this;
    }

    public String getVideoTitle() {
        return videoTitle;
    }

    public PlayerController setVideoTitle(String videoTitle) {
        this.videoTitle = videoTitle;
        return this;
    }

    public boolean isAutoPlay() {
        return autoPlay;
    }

    public PlayerController setAutoPlay(boolean autoPlay) {
        this.autoPlay = autoPlay;
        return this;
    }

    public void toggleFullscreen() {
        if (getImaPlayer() != null) getImaPlayer().toggleFullscreen();
    }

    public void subcribleViewerNumber(int timeInterval) {
        if (timeInterval < TIME_DEFAULT)
            timeInterval = TIME_DEFAULT;
        setTimeInterval(timeInterval);
        if (videoCopyUrl.getVideoType() == Video.VideoType.HLS) {
            startCounterTimer();
        } else if (videoCopyUrl.getVideoType() == Video.VideoType.OTHER) {
            if (videoCopyUrl.getUrl().endsWith(".m3u8")) {
                startCounterTimer();
            }
        } else {
            if (getOnPlayerCallback() != null)
                getOnPlayerCallback().onViewerNumber(videoCopyUrl.getUrl(), false, "0", "Your video is not counter number viewer");
        }
    }

    public void unSubcribleViewerNumber() {
        timeInterval = 0;
        startTime = System.currentTimeMillis();
        isRequest = true;
        if (timer != null) {
            timer.purge();
            timer.cancel();
        }
        timer = null;
        if (timerTask != null)
            timerTask.cancel();
        timerTask = null;
        isRequest = false;
        startTime = 0;
        isRunning = false;
    }

    private void startCounterTimer() {
        if (isRunning)
            return;
        if (timer == null)
            timer = new Timer();
        if (timerTask == null)
            timerTask = new TimerTask() {
                @Override
                public void run() {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            getViewerNumber(timeInterval);
                        }
                    });
                }
            };
        timer.schedule(timerTask, 0, timeInterval);
        isRunning = true;
    }

    private void getViewerNumber(int timeInterval) {
        if (isRequest)
            return;
        if (startTime == 0 || ((System.currentTimeMillis() - startTime) > timeInterval)) {
            isRequest = true;
            startTime = System.currentTimeMillis();
            Log.d("Start: " + startTime);
            requestViewNumber("", false);
        }
    }

    public void requestViewNumber(String id, boolean isVideoId) {
        try {
            JSONObject result = new JSONObject();
            result.put(ID, TextUtils.isEmpty(id) ? videoCopyUrl.getUrl() : id);
            result.put(IS_VIDEOID, isVideoId);
            pushMessage(GET_VIEW_NUMBER, result);
        } catch (JSONException e) {
            if (getOnPlayerCallback() != null)
                getOnPlayerCallback().onViewerNumber(TextUtils.isEmpty(id) ? videoCopyUrl.getUrl() : id, false, "0", "Exception when set data to get. Please try again!");
        }
    }
    //endregion

    //region Config player container
    public OnPlayerCallback getOnPlayerCallback() {
        return weakReferencePlayerCallback == null ? null : weakReferencePlayerCallback.get();
    }

    public PlayerController setOnPlayerCallback(OnPlayerCallback playerCallback) {
        weakReferencePlayerCallback = new WeakReference<>(playerCallback);
        if (imaPlayer != null)
            imaPlayer.setOnPlayerCallback(playerCallback);
        return this;
    }

    public FrameLayout getVideoPlayerContainer() {
        return videoPlayerContainer;
    }

    public PlayerController setVideoPlayerContainer(FrameLayout playerContainer) {
        videoPlayerContainer = playerContainer;
        if (progressBar == null) {
            progressBar = new ProgressBar(Config.shared().getContext());
            FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(100, 100);
            params.gravity = Gravity.CENTER;
            videoPlayerContainer.addView(progressBar, params);
            progressBar.setVisibility(View.VISIBLE);
        } else {
            progressBar.setVisibility(View.VISIBLE);
        }
        return this;
    }

    public void showLoading() {
        if (progressBar != null) {
            progressBar.bringToFront();
            if (videoPlayerContainer != null)
                videoPlayerContainer.bringChildToFront(progressBar);
            progressBar.setVisibility(View.VISIBLE);
        }
    }

    public void hideLoading() {
        if (progressBar != null) {
            progressBar.bringToFront();
            videoPlayerContainer.bringChildToFront(progressBar);
            progressBar.setVisibility(View.GONE);
        }
    }

    public PlayerController build() {
        isBuilded = true;
        if (permission == NON_PERMISSION || permission == PERMISSION_DENIED) {
            android.util.Log.e(TAG, "Waiting!");
            return this;
        }
        buildContent();
        return this;
    }

    private boolean checkPermission(String permissionStr) {
        // Here, thisActivity is the current activity
        return ContextCompat.checkSelfPermission(Config.shared().getContext(), permissionStr) == PackageManager.PERMISSION_GRANTED;
    }

    public PlayerController hideController(boolean shouldHide) {
        this.shouldHide = shouldHide;
        return this;
    }

    public PlayerController setFixedContent(boolean isFixedContent) {
        this.isFixedContent = isFixedContent;
        if (imaPlayer != null)
            imaPlayer.setFixedContent(isFixedContent);
        return this;
    }

    public Context getActivity() {
        return Config.shared().getContext();
    }

    public void setAppkey(String appkey) {
        Config.shared().setAppkey(appkey);
    }

    public String getAppkey() {
        return Config.shared().getAppkey();
    }

    public void setSecretKey(String secretKey) {
        Config.shared().setSecretKey(secretKey);
    }

    public String getSecretKey() {
        return Config.shared().getSecretKey();
    }

    public void setTimeInterval(int timeInterval) {
        this.timeInterval = timeInterval;
    }

    public int getTimeInterval() {
        return timeInterval;
    }

    public String getPlayerId() {
        return Config.shared().getPlayerId();
    }

    public void setPlayerId(String playerId) {
        Config.shared().setPlayerId(playerId);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel parcel, int i) {
        parcel.writeString(videoTitle);
        parcel.writeByte((byte) (autoPlay ? 1 : 0));
        parcel.writeInt(permission);
        parcel.writeByte((byte) (isBuilded ? 1 : 0));
        parcel.writeByte((byte) (shouldHide ? 1 : 0));
        parcel.writeByte((byte) (isFixedContent ? 1 : 0));
        parcel.writeLong(startTime);
        parcel.writeByte((byte) (isRequest ? 1 : 0));
        parcel.writeInt(timeInterval);
        parcel.writeByte((byte) (isRunning ? 1 : 0));
    }

    //endregion

    //region Interface
    public interface OnPlayerCallback {

        void onFullScreen();

        void onNormalScreen();

        void onPlay();

        void onError(String message);

        void playState(int state);

        void videSizeChange(int width, int height, int unappliedRotationDegrees, float pixelWidthHeightRatio);

        void onPlayPressed();

        void onPausePressed();

        void onUpdateProgress(int progress, int length);

        void onBuffering(int buffering);

        /**
         * Callback for showhide controller
         *
         * @param isVisible return true if controller visible otherwise
         */
        void onShowHideControl(boolean isVisible);

        /**
         * This method call when time request delay
         *
         * @param id        id/url of video
         * @param isVideoId status of input id as id or url
         * @param number    number viewer return 0 if request failed or no viewer video
         * @param message   message for request result
         * @see {@link PlayerController}
         * Callback data when request number viewer video
         */
        void onViewerNumber(String id, boolean isVideoId, String number, String message);
    }

    public interface OnCurrentTimeLiveCallback {
        /**
         * Return for time when live is began and current time live of live session
         *
         * @param start     time start as TimeStamp TIMESTAMP type
         * @param current   time current as timestamp type if start time equal 0 else return total time
         *                  since live started (s)
         * @param isSuccess return state for get current time Live info if false ==> return current time of system
         * @param end       return time end of live if end >start then video live is ended otherwise
         */
        void onCurrentTime(boolean isSuccess, long start, long current, long end);
    }
    //endregion

    //region Load data url
    private class MyParseData extends AsyncTask<Void, Void, Void> {

        private static final String TAG = "PLAYER.VCC";
        private final String softUrl;
        private int SIGNAL_OK = 1;
        private Activity mActivity = null;

        MyParseData(String softUrl) {
            this.softUrl = softUrl;
            if (Config.shared().getContext() instanceof Activity)
                mActivity = (Activity) Config.shared().getContext();
            else
                mActivity = null;
        }

        @Override
        protected void onPreExecute() {
            super.onPreExecute();
        }

        @Override
        protected Void doInBackground(Void... params) {
            processURL();
            return null;
        }

        private void processURL() {

            String deviceId = Settings.Secure.getString(Config.shared().getContext().getContentResolver(), Settings.Secure.ANDROID_ID);

            if (softUrl.equals("")) {
                return;
            }

            int position = softUrl.lastIndexOf("/");
            String baseUrl = softUrl.substring(0, position + 1);

            if ((position + 1) == softUrl.length() || TextUtils.isEmpty(baseUrl) || softUrl.equals(baseUrl)) {
                loadDirectLink(softUrl);
                return;
            }
            processWithJson(baseUrl, deviceId);
        }

        private void processWithJson(String baseUrl, String deviceId) {

            Call<PlayResponse> call = APITransaction.builderJSONWithCustomUrl().getLinkPlayVideo(softUrl, deviceId);
            call.enqueue(new Callback<PlayResponse>() {
                @Override
                public void onResponse(Call<PlayResponse> call, Response<PlayResponse> response) {

                    if (response == null || response.body() == null || response.body().getMeta() == null) {
                        if (progressBar != null)
                            progressBar.setVisibility(View.GONE);
                        Log.e(TAG, "===============parse data error!!!===========\n");
                        loadDirectLink(softUrl);
                        return;
                    }
                    PlayResponse playResponse = response.body();
                    String signal = playResponse.getMeta().getSignal();

                    if (Integer.parseInt(signal) == SIGNAL_OK) {
                        String orientation = playResponse.getData().getOrientation();
                        String url = playResponse.getData().getSource();
                        handlePlay(url, orientation);
                    } else {
                        if (progressBar != null)
                            progressBar.setVisibility(View.GONE);
                        loadDirectLink(softUrl);
                    }

                }

                @Override
                public void onFailure(Call<PlayResponse> call, Throwable t) {
                    Log.e(TAG, "" + t.getMessage());
                    if (progressBar != null)
                        progressBar.setVisibility(View.GONE);
                    loadDirectLink(softUrl);
                }
            });
        }

        private void handlePlay(String url, String orientation) {
            if (url.equals("")) {
                Toast.makeText(Config.shared().getContext(), "Cannot play this video!", Toast.LENGTH_SHORT).show();
                if (getOnPlayerCallback() != null) {
                    getOnPlayerCallback().onError("Url video is empty!");
                }
                return;
            }
            checkOrientationScreen(orientation);
            videoCopyUrl.setUrl(url);
            createDirectUrl();
        }

        private void checkOrientationScreen(String orientation) {
            if (mActivity == null)
                return;
            switch (orientation) {
                case PORTRAIT:
                    mActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT);
                    break;
                case LANDSCAPE:
                    mActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE);
                    break;
                default:
                    mActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR);
                    break;
            }
        }

    }

    private void loadDirectLink(String softUrl) {
        new LoadUrlTask().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, softUrl);
    }

    private class LoadUrlTask extends AsyncTask<String, Void, String> {

        String userAgent;

        public LoadUrlTask() {
            userAgent = getDefaultUserAgentString();
        }

        @Override
        protected String doInBackground(String... urls) {
            String loadingUrl = urls[0];
            URL url;
            try {
                url = new URL(loadingUrl);
            } catch (MalformedURLException e) {
                return (loadingUrl != null) ? loadingUrl : "";
            }
            Log.d(TAG, "Checking URL redirect:" + loadingUrl);

            int statusCode = -1;
            HttpURLConnection connection = null;
            String nextLocation = url.toString();

            Set<String> redirectLocations = new HashSet<String>();
            redirectLocations.add(nextLocation);

            try {
                do {
                    connection = (HttpURLConnection) url.openConnection();
                    connection.setRequestProperty("User-Agent", userAgent);
                    connection.setInstanceFollowRedirects(false);

                    statusCode = connection.getResponseCode();
                    if (statusCode == HttpStatus.SC_OK) {
                        connection.disconnect();
                        break;
                    } else {
                        nextLocation = connection.getHeaderField("location");
                        connection.disconnect();
                        if (!redirectLocations.add(nextLocation)) {
                            android.util.Log.d(TAG, "URL redirect cycle detected");
                            return "";
                        }

                        url = new URL(nextLocation);
                    }
                }
                while (statusCode == HttpStatus.SC_MOVED_TEMPORARILY || statusCode == HttpStatus.SC_MOVED_PERMANENTLY
                        || statusCode == HttpStatus.SC_TEMPORARY_REDIRECT || statusCode == HttpStatus.SC_SEE_OTHER);
            } catch (IOException e) {
                return (nextLocation != null) ? nextLocation : "";
            } finally {
                if (connection != null)
                    connection.disconnect();
            }

            return nextLocation;
        }

        @Override
        protected void onPostExecute(String url) {
            Log.d(TAG, "Url response: " + url);
            if (TextUtils.isEmpty(url)) {
                createDirectUrl();
                return;
            }
            handlePlay(url);
        }

        private void handlePlay(String url) {
            if (url.equals("")) {
                Toast.makeText(Config.shared().getContext(), "Cannot play this video!", Toast.LENGTH_SHORT).show();
                if (getOnPlayerCallback() != null) {
                    getOnPlayerCallback().onError("Url video is empty!");
                }
                return;
            }

            videoCopyUrl.setUrl(url);
            createDirectUrl();
        }

        public String getDefaultUserAgentString() {
            String userAgent = System.getProperty("http.agent");
            return userAgent;
        }

    }
    //endregion

    //region Event from plugin

    private static final String GET_VIEW_NUMBER = "GET_VIEW_NUMBER";
    private static final String VIEW_NUMBER = "VIEW_NUMBER";
    private static final String ID = "id";
    private static final String IS_VIDEOID = "isVideoId";
    private static final String TRUE = "true";
    private static final String NUMBER = "number";
    private static final String MESSAGE = "message";


    public static final String TIME_PROGRESS = "time_progress_skin";
    public static final String BUFFER_PROGRESS = "buffer_progress_skin";
    public static final String CURRENT_TIME = "current_time_skin";
    public static final String DURATION_VIDEO = "duration_video_skin";
    public static final String SHOW_CONTROL = "show_control_skin";
    public static final String LIVE_VIDEO = "live_video_skin";
    public static final String LOADING_VIDEO = "loading_video_skin";
    public static final String RELEASE_SKIN = "release_sdk_skin";

    private void pushMessage(String event, Object value) {
        try {
            JSONObject messageInit = new JSONObject();
            messageInit.put(EVENT, event);
            messageInit.put(VALUE, value);
            EBusData.getInstance().postEvent(messageInit);
        } catch (JSONException e) {
        }
    }

    @Subscribe(threadMode = ThreadMode.BACKGROUND)
    public void onEvent(JSONObject object) {


        final JSONUtilParser parser = new JSONUtilParser();
        final String event = parser.getString(EVENT, object);
        final Object data = parser.getObject(VALUE, object);

        if (data == null)
            return;
        if (handlerMainUI == null)
            handlerMainUI = new Handler(Looper.getMainLooper());
        //Post this thread to mainUI thread
        handlerMainUI.post(new Runnable() {
            @Override
            public void run() {
                switch (event) {
                    case VIEW_NUMBER:
                        if (data instanceof JSONObject) {
                            String id = parser.getString(ID, (JSONObject) data);
                            String isVideoId = parser.getString(IS_VIDEOID, (JSONObject) data);
                            String number = parser.getString(NUMBER, (JSONObject) data);
                            String message = parser.getString(MESSAGE, (JSONObject) data);
                            if (getOnPlayerCallback() != null)
                                getOnPlayerCallback().onViewerNumber(id, isVideoId.equals(TRUE), number, message);
                        }
                        break;


                }
            }
        });
    }


    //endregion
}

