package com.vcc.playerads.manager;

import android.content.Context;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import com.vcc.playerads.events.AdsEvent;
import com.vcc.playerads.events.IPlayer;
import com.vcc.playerads.events.OnAdsEventListener;
import com.vcc.playerads.models.AdsModel;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

public class SHAdsManager implements ISHAdsManager {
    private Context context;
    private String urlRequest;
    private FrameLayout adsGroupContainer;
    private IPlayer iPlayer;
    private VASTLayout vastLayout;
    private OnAdsEventListener onAdsEventListener;

    /**
     * Contructer SHAdsManger can use by interface ISHAdsManager
     *
     * @param context
     * @param iPlayer            get data, state from playerView
     * @param onAdsEventListener set listen event from Ads
     */

    public SHAdsManager(Context context, IPlayer iPlayer, OnAdsEventListener onAdsEventListener) {
        this(context, "", iPlayer, onAdsEventListener);
    }

    /**
     * Contructer SHAdsManger can use by interface ISHAdsManager can add Url Ads
     *
     * @param context
     * @param url                url ads (example vast 2, vast 3)
     * @param iPlayer            get data, state from playerView
     * @param onAdsEventListener listen event from Ads
     */
    private SHAdsManager(Context context, String url, IPlayer iPlayer, OnAdsEventListener onAdsEventListener) {
        try {
            System.loadLibrary("APIRequest");
        } catch (Exception | UnsatisfiedLinkError e) {
            if (onAdsEventListener != null) {
                AdsEvent adsEvent = new AdsEvent() {
                    @Override
                    public AdsEventType getType() {
                        return AdsEventType.loadError;
                    }

                    @Override
                    public AdsModel getAds() {
                        return null;
                    }
                };
                onAdsEventListener.onAdsEventListener(adsEvent);
            }
            return;
        }
        if (iPlayer != null)
            this.adsGroupContainer = iPlayer.getAdsContainer();
        this.iPlayer = iPlayer;
        this.context = context;
        this.urlRequest = url;
        setOnAdsEventListener(onAdsEventListener);
        initAdsView();
    }

    /**
     * set UrlRequest for Ads
     *
     * @param urlRequest urlRequest
     */
    @Override
    public void setUrlRequest(String urlRequest) {
        this.urlRequest = urlRequest;
    }

    private void initAdsView() {
        if (adsGroupContainer == null)
            return;
        if (vastLayout == null) {
            vastLayout = new VASTLayout(context);
            vastLayout.setIPlayerContent(iPlayer);
            vastLayout.setLayoutParams(new FrameLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT
            ));

            adsGroupContainer.addView(vastLayout);
            if (onAdsEventListener != null)
                vastLayout.setAdsEventListener(onAdsEventListener);
        }
    }

    /**
     * request Ads default with resetAds after request Ads and skip each Ad
     */
    @Override
    public void requestAds() {
        requestAds(true);
    }

    /**
     * request Ads
     *
     * @param resetAds reset Ad after request
     *                 Default is true . each request reset all .
     */
    @Override
    public void requestAds(boolean resetAds) {
        if (TextUtils.isEmpty(urlRequest)) {
            if (onAdsEventListener != null) {
                AdsEvent adsEvent = new AdsEvent() {
                    @Override
                    public AdsEventType getType() {
                        return AdsEventType.linkRequestEmpty;
                    }

                    @Override
                    public AdsModel getAds() {
                        return null;
                    }
                };
                onAdsEventListener.onAdsEventListener(adsEvent);
            }
            return;
        }

        if (vastLayout != null)
            vastLayout.requestAds(urlRequest, resetAds);

    }

    /**
     * resume Ads when comeback to  Activity
     */
    @Override
    public void resumeAds() {
        if (vastLayout != null)
            vastLayout.resumeAds();
    }

    /**
     * pause Ads when onStop Activity
     */
    @Override
    public void pauseAds() {
        if (vastLayout != null)
            vastLayout.pauseAds();
    }

    /**
     * @param isSkip skip each Ad or All Ads after click Skip Ads
     */

    @Override
    public void setSkipAllAds(boolean isSkip) {
        vastLayout.setSkipAllAds(isSkip);
    }

    /**
     * skip current Ads
     */
    @Override
    public void skipAds() {
        if (vastLayout != null)
            vastLayout.clickSkipAds();
    }

    /**
     * release  Ads
     */
    @Override
    public void releaseAds() {
        if (vastLayout != null)
            vastLayout.resetAllAds();
    }

    /**
     * get Marker (time when show Ads)
     *
     * @return List time can show Ads and Client can draw marker in seekBar
     * Example : return "00:00:15"
     */
    @Override
    public List<String> getMarkerShowAds() {
        if (vastLayout != null) {
            return vastLayout.getListMarkerAds();
        }
        return new CopyOnWriteArrayList<>();
    }

    /**
     * getListAdsShowed (List Ads showed )
     *
     * @return List Ads showed
     * Example : return "00:00:15"
     */
    @Override
    public List<String> getListAdsShowed() {
        if (vastLayout != null) {
            return vastLayout.getListAdsShowed();
        }
        return new CopyOnWriteArrayList<>();
    }

    /**
     * set Ads can play multi or not
     * For Example : if play multi will show [1 of 5] Ads but not will not show this
     *
     * @param isPlayMultiAds Default :true
     */
    @Override
    public void setPlayMultiAds(boolean isPlayMultiAds) {
        if (vastLayout != null) {
            vastLayout.setPlayMultiAds(isPlayMultiAds);
        }
    }

    /**
     * Get type of Ads
     * Example : preroll, midroll...
     *
     * @return preroll = start
     * midroll = 00:00:15 (example)
     * postrol = end
     */
    @Override
    public String getType() {
        if (vastLayout != null) {
            return vastLayout.getTypeOfAds();
        }
        return "";
    }

    /**
     * @return Duration Video of Ads
     */
    @Override
    public String getDurationAds() {
        if (vastLayout != null) {
            return vastLayout.getDurationAds();
        }
        return "";
    }

    /**
     * @return progress played of Ads
     * 0 - 100 (%)
     */
    @Override
    public int getProgressAds() {
        if (vastLayout != null) {
            return vastLayout.getProgressAds();
        }
        return 0;
    }

    /**
     * @return id of Ad
     */
    @Override
    public String getAdsId() {
        if (vastLayout != null) {
            return vastLayout.getAdsId();
        }
        return "";
    }

    /**
     * @return volume of Ads
     */
    @Override
    public float getVolumeAds() {
        if (vastLayout != null) {
            return vastLayout.getMusicVolumeLevel();
        }
        return 0.0f;
    }

    @Override
    public View getViewAds() {
        if (vastLayout != null) {
            return vastLayout.getViewAdsLayout();
        }
        return null;
    }

    @Override
    public AdsModel getCurrentAds() {
        if (vastLayout != null) {
            return vastLayout.getCurrentAdsModel();
        }
        return null;
    }

    /**
     * get State of Ads
     *
     * @return true of false
     */
    @Override
    public boolean isPlayingAds() {
        return vastLayout != null && vastLayout.isPlayingAds();
    }

    public OnAdsEventListener getOnAdsEventListener() {
        return onAdsEventListener;
    }

    /**
     * get Listener from Ads
     *
     * @param onAdsEventListener
     */
    @Override
    public void setOnAdsEventListener(OnAdsEventListener onAdsEventListener) {
        if (vastLayout != null)
            vastLayout.setAdsEventListener(onAdsEventListener);
        this.onAdsEventListener = onAdsEventListener;
    }

    /**
     * mute Ads
     *
     * @param mute or unmute
     */
    @Override
    public void setMuteAds(boolean mute) {
        if (vastLayout != null)
            vastLayout.muteAds(mute);
    }


}
